/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation
--      51 Franklin Street, Fifth Floor
--      Boston, MA 02110-1301
--      USA
--      http://www.fsf.org/licenses
--
-- $Log: list_files.c,v $
-- Revision 1.3  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.2  2006/10/26 01:51:27  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.4  2006/10/25 12:22:28  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.1.2.3  2006/10/07 12:56:27  jsedwards
-- Added file name globing so you specify which files to list.
--
-- Revision 1.1.2.2  2006/10/07 12:27:59  jsedwards
-- Added the ability to print md5sums in the listing with the --md5 option.
--
-- Revision 1.1.2.1  2006/10/06 12:06:57  jsedwards
-- Program to list all files (actually file_paths) in system.
--
*/

#include <fnmatch.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ObjRef path_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int num_paths = 0;
    int argn;
    int i;
    int j;
    char name[256];
    bool do_md5 = false;
    bool match;
    C_struct_File file_obj;
    C_struct_MD5sum md5_object;


    StorageType type = DEFAULT_TYPE;
    char* path = DEFAULT_FILE;

    argn = 1;

    while (argn < argc && *argv[argn] == '-')
    {
	if (strcmp(argv[argn], "--compressed") == 0 && argc > argn + 1)
	{
	    type = Compressed_File_RO;
	    argn++;
	    path = argv[argn];
	    argn++;
	}
	else if (strcmp(argv[argn], "--md5") == 0)
	{
	    do_md5 = true;
	    argn++;
	}
	else
	{
	    fprintf(stderr, "usage: %s [--compressed compressed-file] [--md5] files to extract\n", argv[0]);
	    exit(1);
	}

    }

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, type, path);

    nwos_set_root_object(&root_object_ref);


    assert(nwos_find_class_definition("FILE PATH", &path_class_ref));

    nwos_read_class_definition(&path_class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    printf("num_refs: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &path_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_path_object_size);

	    /* remember ptr_path_obj points to the kludge buffer */

	    for (j = 0; j < ptr_path_obj->count; j++) name[j] = ptr_path_obj->storage[j];

	    name[j] = '\0';

	    if (argn < argc)    /* file names were passed to the program on the command line */
	    {
		match = false;

		for (j = argn; j < argc; j++)
		{
		    if (fnmatch(argv[j], name, 0) == 0)   /* found a match */
		    {
			match = true;
			break;
		    }
		}
	    }
	    else             /* no filenames specified on command line */
	    {
		match = true;   /* list all files */
	    }

	    if (match)
	    {
		if (do_md5)
		{
		    nwos_read_object_from_disk(&ptr_path_obj->file, &file_obj, sizeof(file_obj));  /* read the file object */
		    nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object));

		    for (j = 0; j < 16; j++) printf("%02x", md5_object.md5sum[j]);
		    printf("  ");
		}

		printf("%s\n", name);

		num_paths++;
	    }
	}
    }

    printf("\nNumber of files: %d\n", num_paths);

    free(ref_list);
    ref_list = NULL;

    nwos_terminate_objectify();

    return 0;
}


