/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: export_file.c,v $
-- Revision 1.6  2006/11/29 19:03:45  jsedwards
-- Added code to handle the case where the requested file is not stored in
-- the system.
--
-- Revision 1.5  2006/11/29 18:52:04  jsedwards
-- Change so that this function finds the file reference and passes it along
-- with the path to the restore_file function.
--
-- Revision 1.4  2006/11/18 15:09:09  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.3  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.2  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.5  2006/10/25 12:22:28  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.1.2.4  2006/10/07 12:35:41  jsedwards
-- Changed to allow an '*' to specify exporting all files.
--
-- Revision 1.1.2.3  2006/10/03 12:53:07  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.1.2.2  2006/10/02 02:27:03  jsedwards
-- Added option to read the file from a compressed archive file.
--
-- Revision 1.1.2.1  2006/09/18 01:45:12  jsedwards
-- Program to export a file stored in objectify.
--
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "objectify.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


int list_all(char*** list_ptr)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ObjRef path_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int num_paths = 0;
    int i;
    int j;


    assert(nwos_find_class_definition("FILE PATH", &path_class_ref));

    nwos_read_class_definition(&path_class_ref, &class_def_obj);

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    printf("num_refs: %d\n", num_refs);

    *list_ptr = malloc(num_refs * sizeof(char*));

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &path_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_path_object_size);

	    /* remember ptr_path_obj points to the kludge buffer */

	    (*list_ptr)[num_paths] = malloc(ptr_path_obj->count + 1);

	    for (j = 0; j < ptr_path_obj->count; j++) (*list_ptr)[num_paths][j] = ptr_path_obj->storage[j];

	    (*list_ptr)[num_paths][j] = '\0';

	    num_paths++;
	}
    }

    printf("\nNumber of files: %d\n", num_paths);

    free(ref_list);
    ref_list = NULL;

    *list_ptr = realloc(*list_ptr, num_paths * sizeof(char*));

    return num_paths;
}



void export_list(int num_names, char* names[])
{
    bool result;
    int i;
    ObjRef path_ref;
    ObjRef file_ref;
    C_struct_File file_obj;

    for (i = 0; i < num_names; i++)
    {
	printf("%s: ", names[i]);
	fflush(stdout);

	if (nwos_find_file_path(names[i], &path_ref))
	{
	    nwos_file_path_to_file(&path_ref, &file_ref);

	    nwos_read_object_from_disk(&file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

	    if (is_void_reference(&file_obj.block_list))
	    {
		printf("not stored in system\n");
	    }
	    else
	    {
		result = nwos_restore_file(&file_ref, names[i]);

		if (result)
		{
		    printf("OK\n");
		}
		else
		{
		    printf("failed\n");
		}
	    }
	}
	else
	{
	    printf("no file by that name found in system\n");
	}
    }
}


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    int i;
    StorageType type = DEFAULT_TYPE;
    char* path = DEFAULT_FILE;
    int num_names;
    char** names;

    if (argc > 1 && *argv[1] == '-')
    {
	if (strcmp(argv[1], "--compressed") != 0 || argc < 3)
	{
	    fprintf(stderr, "usage: %s [--compressed compressed-file] files to extract\n", argv[0]);
	    exit(1);
	}

	type = Compressed_File_RO;
	path = argv[2];
	i = 3;
    }
    else
    {
	i = 1;
    }

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, type, path);

    nwos_set_root_object(&root_object_ref);

    if (strcmp(argv[i], "*") == 0)
    {
	num_names = list_all(&names);
	export_list(num_names, names);
    }
    else
    {
	export_list(argc - i, &argv[i]);
    }

    nwos_terminate_objectify();

    return 0;
}


