/*
    Numdiff - compare putatively similar files, 
    ignoring small numeric differences
    Copyright (C) 2005-2007  Ivano Primi  <ivprimi@libero.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<ctype.h>
#include<float.h>
#include"getopt.h"
#include"numdiff.h"

#ifdef _DMALLOC_
#include <dmalloc.h> /* Useful only for the debugging */
#endif

void print_version (const char* progname)
{
#ifdef _USE_MPA
  /* Nothing ! */
#elif defined(_USE_HPA)
  Real Epsilon = xpr2 (xOne, -16 * XDIM + 1);
  int prec_dig = XDIM * 4816 / 1000;
#elif defined (_USE_LDBL)
  Real Epsilon = LDBL_EPSILON;
  int prec_dig = LDBL_DIG;
#else
  Real Epsilon = DBL_EPSILON;
  int prec_dig = DBL_DIG;
#endif

  const char* acclevel[] = {
    N_("Built with support for Double Precision Arithmetic"),
    N_("Built with support for Long Double Precision Arithmetic"),
    N_("Built with support for Extended Precision Arithmetic"),
    N_("Built with support for Multiple Precision Arithmetic")
  };

  printf ("%s %s\n", progname, VERSION);
  printf (_("Copyright (C) 2005-2007  %s <ivprimi@libero.it>\n"), 
	  /* TRANSLATORS: This is a proper name.  See the gettext
	     manual, section Names.
	     Pronounciation is like "evaa-no pree-me".  */
	  _("Ivano Primi"));
  puts (_("This is free software; see the source for copying conditions."));
  puts (_("There is NO warranty; not even for MERCHANTABILITY or\nFITNESS FOR A PARTICULAR PURPOSE."));
  printf ("\n%s:\n", _(acclevel[ACCURACY_LEVEL]));
#ifdef _USE_MPA
  printf ("%s\n%s\n", 
	  _("Epsilon = 1.0e-##, where ## is the precision set by the user"),
	  _("Decimal digits of accuracy =  precision set by the user"));
#else
  printf ("Epsilon = ~");
  printno (Epsilon, 6);
  printf ("\n%s =  %i\n", _("Decimal digits of accuracy"), prec_dig); 
#endif

#ifdef _USE_MPA
  putchar ('\n');
#elif defined (_USE_HPA) || defined (_USE_LDBL)
  printf ("%s = ~%Le\n", _("Minimum normalized positive floating-point number"), LDBL_MIN);
  printf ("%s = ~%Le\n\n", _("Maximum representable finite floating-point number"), LDBL_MAX);
#else
  printf ("%s = ~%e\n", _("Minimum normalized positive floating-point number"), DBL_MIN);
  printf ("%s = ~%e\n\n", _("Maximum representable finite floating-point number"), DBL_MAX);
#endif
}

void print_help (const char* progname)
{
  puts (_("Usage:"));
  printf ("%s -h|v   %s\n\n", progname, _("or"));
  printf ("%s %s\n", progname, "[-a maxerr][-r maxerr][-2][-P][-N][-s ifs][-b][-V][-q][-D][-E][-I][-S][-# prec][-d c1c2][-t c1c2][-g n1n2][-p c1c2][-n c1c2][-e c1c2][-i c1c2][-F f1-f2][-L l1-l2][-l path][-o path] FILE1 FILE2");
  /* %%% */
  printf (_("\nCompare putatively similar files line by line and field by field,\nignoring small numeric differences or/and different numeric formats\n\n"));
  printf ("-a maxerr %s\n          %s\n", 
	  _("Specify the maximum absolute difference permitted\n\t  before that two numeric fields are regarded as different"),
	  _("(The default value is zero)"));
  printf ("-r maxerr %s\n          %s\n", 
	  _("Specify the maximum relative difference permitted\n\t  before that two numeric fields are regarded as different"),
	  _("(The default value is zero)"));
  printf ("-2        %s\n",
	  _("Order that two numerical values are regarded as equal only if\n\t  both absolute and relative difference do not exceed\n\t  the corresponding tolerance threshold"));
  printf ("-P        %s\n",
	  _("Ignore all differences due to numeric fields of the second file that\n\t  are less than the corresponding numeric fields in the first file"));
  printf ("-N        %s\n",
	  _("Ignore all differences due to numeric fields of the second file that\n\t  are greater than the corresponding numeric fields in the first file"));
  printf ("-E        %s\n",
	  _("While printing the differences between the two compared files\n\t  show only the numerical ones"));
  printf ("-D        %s\n",
	  _("While printing the differences between the two compared files\n\t  neglect all the numerical ones (dummy mode)"));
  printf ("-I        %s\n",
	  _("Ignore changes in case while doing literal comparisons"));
  printf ("-s ifs    %s\n\t  %s\n",
	  _("Specify the set of characters to use\n\t  to split the input lines into fields"),
	  _("(The default set of characters is white space, tab and newline)"));
  printf ("-b        %s\n",
	  _("Suppress all messages concerning the differences discovered\n\t  in the structures of the two files"));
  printf ("-V        %s\n",
	  _("For every line differing in at least one field print an header\n\t  to show how this line appears in the two compared files"));
  printf ("-q        %s\n",
	  _("Suppress all the standard output"));
  printf ("-S        %s\n",
	  _("Add some statistics to the standard output"));
  printf ("-# prec   %s\n",
	  _("Specify the number of digits in the significands\n\t  used in multiple-precision arithmetic"));
  printf ("-d c1c2   %s\n",
	  _("Specify the characters representing the decimal point\n\t  in the two files to compare"));
  printf ("-t c1c2   %s\n",
	  _("Specify the characters representing the thousands separator\n\t  in the two files to compare"));
  printf ("-g n1n2   %s\n",
	  _("Specify the number of digits forming each group of thousands\n\t  in the two files to compare"));
  printf ("-p c1c2   %s\n",
	  _("Specify the (optional) prefixes for positive values\n\t  used in the two files to compare"));
  printf ("-n c1c2   %s\n",
	  _("Specify the prefixes for negative values\n\t  used in the two files to compare"));
  printf ("-e c1c2   %s\n",
	  _("Specify the exponent letters\n\t  used in the two files to compare"));
  printf ("-i c1c2   %s\n",
	  _("Specify the characters representing the imaginary unit\n\t  in the two files to compare"));
  printf ("-F f1-f2  %s\n\t  %s\n",
	  _("Select the fields that have to be compared"),
	  _("(The default behavior is comparing all the fields)"));
  printf ("-L l1-l2  %s\n\t  %s\n",
	  _("Select the lines whose fields have to be compared"),
	  _("(The default behavior is comparing all the fields in all the lines)"));
  printf ("-l path   %s\n",
	  _("Redirect warning and error messages from stderr to the indicated file"));
  printf ("-o path   %s\n",
	  _("Redirect output from stdout to the indicated file"));
  printf ("-h        %s\n", _("Show help message and predefined settings"));
  printf ("-v        %s\n", _("Show version number, Copyright and NO-Warranty"));
  printf ("\n%s\n%s\n%s\n%s\n",
	  _("The two arguments after the options are the names of the files to compare."),
	  _("The complete paths of the files should be given,\na directory name is not accepted."),
	  _("They cannot refer to the same file but one of them can be \"-\",\nwhich refers to stdin."),
	  _("Exit status: 1 if files differ, 0 if they are equal, -1 (255) in case of error"));
  /* %%% */
  puts (_("\n  Default numeric format (for both files to compare):\n"));
  printf (_("Decimal point = `%c\'\n"), DP);
#ifdef _USE_MPA
  printf (_("Thousands separator = `%c\'\n"), THSEP);
  printf (_("Number of digits in each thousands group = %u\n"), GROUPING);
#endif
  printf (_("Positive sign = `%c\'\n"), POS_SIGN);
  printf (_("Negative sign = `%c\'\n"), NEG_SIGN);
  printf (_("Prefix for decimal exponent = `%c\'\n"), ECH);
  printf (_("Symbol used to denote the imaginary unit = `%c\'\n\n"), IU);
}

char* return_ifs (const char* optarg)
{
  char *s = (char*) malloc ((strlen(optarg) + 1) * sizeof(char));
  
  if(!s)
    return NULL;
  else
    {
      char *t, *u;

      strcpy (s, optarg);
      for (t = s; *t != '\0'; t++)
	{
	  if (*t == '\\')
	    {
	      switch (*(t+1))
		{
		case 'f':
		  *t = '\f';
		  break;
		case 'n':
		  *t = '\n';
		  break;
		case 'r':
		  *t = '\r';
		  break;
		case 't':
		  *t = '\t';
		  break;
		case 'v':
		  *t = '\v';
		  break;
		default:
		  *t = *(t+1);
		}
	      for (u = t+1; *u != '\0'; *u = *(u+1), u++);
	    }
	}
      return s;
    }
}

int nfset (int opt_ch, const char* opt_arg, argslist* arg_list)
{
  if (strlen(opt_arg) <= 2)
    {
      char _1st = *opt_arg, _2nd = *(opt_arg+1);

      switch (opt_ch)
	{
	case 'd':
	  if ( (is_punct(_1st)) && (_2nd == '\0' || is_punct(_2nd)) )
	    {
	      arg_list->optmask |= _D_MASK;
	      arg_list->nf1.dp = _1st;
	      arg_list->nf2.dp = (_2nd) ? _2nd : _1st; 
	      return 0;
	    }
	  break;
	case 't':
	  if ( (is_punct(_1st)) && (_2nd == '\0' || is_punct(_2nd)) )
	    {
	      arg_list->optmask |= _T_MASK;
	      arg_list->nf1.thsep = _1st;
	      arg_list->nf2.thsep = (_2nd) ? _2nd : _1st;
	      return 0;
	    }
	  break;
	case 'e':
	  if ( is_print(_1st) && (_2nd == '\0' || is_print(_2nd)) )
	    {
	      arg_list->optmask |= _E_MASK;
	      arg_list->nf1.ech = _1st;
	      arg_list->nf2.ech = (_2nd) ? _2nd : _1st;
	      return 0;
	    }
	  break;
	case 'n':
	  if ( is_print(_1st) && (_2nd == '\0' || is_print(_2nd)) )
	    {
	      arg_list->optmask |= _N_MASK;
	      arg_list->nf1.neg_sign = _1st;
	      arg_list->nf2.neg_sign = (_2nd) ? _2nd : _1st;
	      return 0;
	    }
	  break;
	case 'i':
	  if ( is_print(_1st) && (_2nd == '\0' || is_print(_2nd)) )
	    {
	      arg_list->optmask |= _I_MASK;
	      arg_list->nf1.iu = _1st;
	      arg_list->nf2.iu = (_2nd) ? _2nd : _1st;
	      return 0;
	    }
	  break;
	case 'p':
	  if ( is_print(_1st) && (_2nd == '\0' || is_print(_2nd)) )
	    {
	      arg_list->optmask |= _P_MASK;
	      arg_list->nf1.pos_sign = _1st;
	      arg_list->nf2.pos_sign = (_2nd) ? _2nd : _1st;
	      return 0;
	    }
	  break;
	case 'g':
	  if ( (is_digit(_1st)) && (_2nd == '\0' || is_digit(_2nd)) )
	    {
	      arg_list->optmask |= _G_MASK;
	      arg_list->nf1.grouping = _1st - '0';
	      arg_list->nf2.grouping = (_2nd) ? _2nd - '0': _1st - '0';
	      return 0;
	    }
	  break;
	}
    }
  return -1;
}

int fselect (const char* str, unsigned char* mask, int mask_size)
{
  long beg, end;
  unsigned long n;
  char *ptr, *endptr;

  beg = end = -1;
  if (!str || !*str)
    return 0; // no field selected
  // If we arrive here we are sure that *str != '\0' !
  if ((beg = strtol (str, &endptr, 10)) == 0
      || beg > mask_size || beg < -mask_size)
    return -1; // illegal input
  else if (beg < 0)
    {
      end = -beg;
      beg = 1;
    }
  else if (*endptr == '\0')
    end = beg;
  else if (*endptr == '-')
    {
      if (*(ptr = endptr + 1) == '\0')
	end = mask_size; 
      else
	{
	  if ((end = strtol (ptr, &endptr, 10)) <= 0
	      || *endptr != '\0' || end > mask_size)
	    return -1; // illegal input
	}
    }
  if (beg > end)
    return -1;
  else
    {
      for (n = beg - 1; n <= end - 1; n++)
	mask[n >> 3] |= 0x80 >> (n & 0x7);
      return 1;
    }
}

int lrselect (const char* str, integer* Begin, integer* End)
{
  char *ptr, *endptr;
  integer beg, end;

  beg = end = -1;
  if (!str || !*str)
    return 0; // no field selected
  // If we arrive here we are sure that *str != '\0' !
  if ((beg = STR2INT (str, &endptr, 10)) == 0
      || beg > INTEGER_MAX || beg < -INTEGER_MAX)
    return -1; // illegal input
  else if (beg < 0)
    {
      end = -beg;
      beg = 1;
    }
  else if (*endptr == '\0')
    end = beg;
  else if (*endptr == '-')
    {
      if (*(ptr = endptr + 1) == '\0')
	end = INTEGER_MAX; 
      else
	{
	  if ((end = STR2INT (ptr, &endptr, 10)) <= 0
	      || *endptr != '\0' || end > INTEGER_MAX)
	    return -1; // illegal input
	}
    }
  if (beg > end)
    return -1;
  else
    {
      *Begin = beg, *End = end;
      return 1;
    }
}


int valid_numfmt (const struct numfmt* pnf)
{
  char store[NUMFMT_CHARS];
  int i, j;

  store[0] = pnf->dp;
  store[1] = pnf->thsep;
  store[2] = pnf->pos_sign;
  store[3] = pnf->neg_sign;
  store[4] = pnf->ech;
  store[5] = pnf->iu;
  for (i=0; i < NUMFMT_CHARS; i++)
    {
      for (j = i+1; j < NUMFMT_CHARS; j++)
	if (store[i] == store[j])
	  return 0;
    }
  return 1;
}

extern int optind;

int setargs (int argc, char* argv[], argslist *list)
{
  const char *optstring = "h2bVqDESIPN#:s:a:r:d:t:g:p:n:e:i:F:L:l:o:v";
  char *tail;
  int optch, i; 
  struct numfmt defaults;

  /*
    We start by loading the default values
    for the user settable options.
  */
#ifndef _USE_MPA
  /*
    When _USE_MPA is defined, the initialization
    of these variables is done within main()
    through init_mpa_support() .
  */
  list->maxrelerr = list->maxabserr = Zero;
  list->Labserr  = list->Crelerr  = list->Lrelerr  = list->Cabserr  = Zero;
#endif
  list->optmask = 0x0;
  list->firstln = 1;
  list->lastln = INTEGER_MAX;
  for (i=0; i < FIELDMASK_SIZE; list->fieldmask[i] = 0xFF, i++); 
  list->flag = 0;
  list->ifs = NULL;
  list->iscale = ISCALE;
  list->nf1.dp = DP;
  list->nf1.thsep = THSEP;
  list->nf1.grouping = GROUPING;
  list->nf1.pos_sign = POS_SIGN;
  list->nf1.neg_sign = NEG_SIGN;
  list->nf1.ech = ECH;
  list->nf1.iu = IU;
  list->file1 = list->file2 = NULL;
  defaults = list->nf2 = list->nf1;
  /*
    defaults.dp == DP
    defaults.thsep == THSEP;
    defaults.grouping == GROUPING;
    defaults.pos_sign == POS_SIGN;
    defaults.neg_sign == NEG_SIGN;
    defaults.ech == ECH;
    defaults.iu == IU;

    Since 'defaults' is not modified in the following
    we are sure that it will always contain
    these default values.
  */

  /* i is used as a flag. It becomes !0 the first time the option -F */
  /* is successfully managed.                                        */
  i = 0;
  while ( (optch = getopt (argc, argv, optstring)) != -1 )
    {
      switch (optch)
	{
	case 'h':
	  list->optmask |= _H_MASK;
	  break;
	case '2':
	  list->optmask |= _2_MASK;
	  break;	  
	case 'b':
	  list->optmask |= _B_MASK;
	  break;
	case 'V':
	  list->optmask |= _SV_MASK;
	  break;
	case 'q':
	  list->optmask |= _Q_MASK;
	  break;
	case 'D':
	  list->optmask |= _SD_MASK;
	  break;
	case 'E':
	  list->optmask |= _SE_MASK;
	  break;
	case 'S':
	  list->optmask |= _SS_MASK;
	  break;
	case 'I':
	  list->optmask |= _SI_MASK;
	  break;
	case 'P':
	  list->optmask |= _SP_MASK;
	  list->flag = 1;
	  break;
	case 'N':
	  list->optmask |= _SN_MASK;
	  list->flag = -1;
	  break;
	case '#':
	  list->iscale = strtol (optarg, &tail, 10);
	  if (*tail != '\0' || list->iscale < 0 || list->iscale > MAX_ISCALE)
	    {
	      fprintf (stderr, _("%s: invalid argument after `-%c\' option\n"),
		       argv[0], optch);
	      return -1;
	    }
	  else
	    list->optmask |= _X_MASK;	  
	  break;
	case 's':
	  list->optmask |= _S_MASK;
	  if ((list->ifs))
	    free((void*)list->ifs);
	  list->ifs = return_ifs (optarg);
#ifdef __DEBUG__
	  fprintf (stderr, "-s argument: %s\n", optarg);
	  fputs ("Internal field separator: ", stderr);
	  for (tail = list->ifs; *tail != '\0'; tail++)
	    fprintf (stderr, "`%c\' ", *tail);
	  fputc ('\n', stderr);
#endif /* __DEBUG__ */
	  break;
	case 'a':
#ifdef _USE_MPA
	  delR (&list->maxabserr); /* To avoid unpleasant memory leaks */
	  str2R (optarg, &tail, ISCALE, &defaults, &list->maxabserr);
#else
	  list->maxabserr = str2R (optarg, &tail);
#endif
	  if (*tail != '\0')
	    {
	      fprintf (stderr, _("%s: invalid argument after `-%c\' option\n"),
		       argv[0], optch);
	      return -1;
	    }
	  else
	    list->optmask |= _A_MASK;
	  break;
	case 'r':
#ifdef _USE_MPA
	  delR (&list->maxrelerr); /* To avoid unpleasant memory leaks */
	  str2R (optarg, &tail, ISCALE, &defaults, &list->maxrelerr);
#else
	  list->maxrelerr = str2R (optarg, &tail);
#endif
	  if (*tail != '\0')
	    {
	      fprintf (stderr, _("%s: invalid argument after `-%c\' option\n"),
		       argv[0], optch);
	      return -1;
	    }
	  else
	    list->optmask |= _R_MASK;
	  break;
	case 'd':
	case 't':
	case 'g':
	case 'p':
	case 'n':
	case 'e':
	case 'i':
	  if (nfset (optch, optarg, list) < 0)
	    {
	      fprintf (stderr, _("%s: invalid argument after `-%c\' option\n"),
		       argv[0], optch);
	      return -1;
	    }
	  break;
	case 'F':
	  if (i == 0)
	    for (; i < FIELDMASK_SIZE; list->fieldmask[i] = 0x0, i++);
	  if (fselect (optarg, list->fieldmask, FIELDMASK_SIZE*8) <= 0)
	    {
	      fprintf (stderr, _("%s: invalid argument after `-%c\' option\n"),
		       argv[0], optch);
	      return -1;
	    }
	  else
	    list->optmask |= _SF_MASK; 
	  break;
	case 'L':
	  if (lrselect (optarg, &list->firstln, &list->lastln) <= 0)
	    {
	      fprintf (stderr, _("%s: invalid argument after `-%c\' option\n"),
		       argv[0], optch);
	      return -1;
	    }
	  else
	    list->optmask |= _SL_MASK; 
	  break;
	case 'l':
	  if (!freopen (optarg, "w", stderr))
	    {
	      fprintf (stderr, _("%s: Cannot open file \"%s\": "),
		       argv[0], optarg);
	      perror(0);
	      return -1;
	    }
	  break;
	case 'o':
	  if (!freopen (optarg, "w", stdout))
	    {
	      fprintf (stderr, _("%s: Cannot open file \"%s\": "),
		       argv[0], optarg);
	      perror(0);
	      return -1;
	    }
	  break;
	case 'v':
	  list->optmask |= _V_MASK;
	  break;
	default:
	  /* 	  
	    fprintf (stderr, 
	    _("%s: unrecognized option `-%c\' \n"), argv[0], optch); 
	  */
	  return -1;
	}
    }
  if (!(list->optmask & (_H_MASK | _V_MASK)) && argc - optind != 2)
    {
      print_help (PACKAGE);
      return -1;
    }
  else if ( !valid_numfmt(&list->nf1) )
    {
      fprintf (stderr, 
	       _("The numeric format specified for the first file is illegal,\n"));
      fprintf (stderr,
	       _("the following symbols should be all different\nwhile two or more of them are actually equal:\n"));
      fprintf (stderr, _("\nDecimal point = `%c\'\n"), list->nf1.dp);
      fprintf (stderr, _("Thousands separator = `%c\'\n"), list->nf1.thsep);
      fprintf (stderr, _("Positive sign = `%c\'\n"), list->nf1.pos_sign);
      fprintf (stderr, _("Negative sign = `%c\'\n"), list->nf1.neg_sign);
      fprintf (stderr, _("Prefix for decimal exponent = `%c\'\n"), 
	       list->nf1.ech);
      fprintf (stderr, 
	       _("Symbol used to denote the imaginary unit = `%c\'\n\n"), 
	       list->nf1.iu);
      return -1;
    }
  else if ( !valid_numfmt(&list->nf2) )
    {
      fprintf (stderr, 
	       _("The numeric format specified for the second file is illegal,\n"));
      fprintf (stderr,
	       _("the following symbols should be all different\nwhile two or more of them are actually equal:\n"));
      fprintf (stderr, _("\nDecimal point = `%c\'\n"), list->nf2.dp);
      fprintf (stderr, _("Thousands separator = `%c\'\n"), list->nf2.thsep);
      fprintf (stderr, _("Positive sign = `%c\'\n"), list->nf2.pos_sign);
      fprintf (stderr, _("Negative sign = `%c\'\n"), list->nf2.neg_sign);
      fprintf (stderr, _("Prefix for decimal exponent = `%c\'\n"), 
	       list->nf2.ech);
      fprintf (stderr, 
	       _("Symbol used to denote the imaginary unit = `%c\'\n\n"), 
	       list->nf2.iu);
      return -1;
    }
  else
    {
      if( !(list->optmask & (_H_MASK | _V_MASK)) )
	{
	  list->file1 = (const char*) argv[optind];
	  list->file2 = (const char*) argv[optind+1];
	}
      return 0;
    }
}  
