/*
    Numdiff - compare putatively similar files, 
    ignoring small numeric differences
    Copyright (C) 2005-2007  Ivano Primi  <ivprimi@libero.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include<stdio.h>
#include"numdiff.h"

#ifdef _USE_MPA

#include"number.c"
#include"errors.c"
#include"new.c"

Real Zero, Inf, Epsilon, MEpsilon;

void init_mpa(void)
{
  bc_init_numbers();
  bc_init_num(&Zero);
  Inf = bc_new_num (1,0);
  Inf->n_value[0] = 10;
}

void mpa_define_epsilon (int iscale, const struct numfmt* pnf)
{
  Epsilon = bc_10_raised_to (-iscale, 0);
  bc_sqrt (&Epsilon, iscale);
  MEpsilon = bc_10_raised_to (-iscale, 0);
  bc_sqrt (&MEpsilon, iscale);
  MEpsilon->n_sign = MINUS;
#ifdef _MPA_DEBUG
  fputs ("+Epsilon = ", stderr);
  debug_printno (Epsilon, 20);
  fputs ("\n-Epsilon = ", stderr);
  debug_printno (MEpsilon, 20);
  fprintf (stderr, "\niscale = %d\n", iscale);
#endif  
}

void initR (Real* px)
{
  bc_init_num (px);
}

void initC (Complex* pz)
{
  bc_init_num (&pz->re);
  bc_init_num (&pz->im);
}

Real copyR (Real x)
{
  return bc_copy_num (x);
}

Complex copyC (Complex z)
{
  Complex w;

  w.re = bc_copy_num (z.re);
  w.im = bc_copy_num (z.im);
  return w;
}

void str2R (const char *q, char **endptr, int iscale,
	    const struct numfmt* pnf, Real* pr)
{
  bc_a2num (pr, q, endptr, iscale, pnf);
}

void str2C (const char *q, char **endptr, int iscale,
	    const struct numfmt* pnf, Complex* pc)
{
  char *ptr, *ptr2;

  bc_init_num (&pc->re);
  bc_init_num (&pc->im);
  bc_a2num (&pc->re, q, &ptr, iscale, pnf);
  if ((endptr))
    *endptr = ptr;
  if (ptr != q)
    {
      if (*ptr == pnf->iu)
	{
	  pc->im = bc_copy_num (pc->re);
	  pc->re = bc_copy_num (_zero_);
	  if ((endptr))
	    *endptr = ptr + 1;
	}
      else
	{
	  while (isspace (*ptr))
	    ptr++;
	  if (*ptr == '+' || *ptr == '-')
	    {
	      bc_a2num (&pc->im, ptr, &ptr2, iscale, pnf);
	      if (*ptr2 != pnf->iu)
		pc->im = bc_copy_num (_zero_);
	      else
		{
		  if ((endptr))
		    *endptr = ptr2 + 1;
		}
	    }
	  /*
	     else
	     : we have successfully read a real number
	     but there is no another number after it.
	     : So, we leave pc->im set to zero.
	  */
	}
    }
  else
    /* 
       : q does not contain any valid number
       ==> pc->re is 0. Then we set pc->im to 0.
       We remark that, if endptr is
       not NULL, then *endptr == q.
    */
    pc->im = bc_copy_num (_zero_);
}

void divide (Real s, Real t, Real* q, int iscale)
{
  bc_divide (s, t, q, iscale);
}

void Cabs (Complex z, Real* pm, int iscale)
{
  bc_num a, b, q;

  if ( z.re->n_sign == PLUS )
    a = bc_copy_num (z.re);
  else
    {
      bc_init_num (&a);
      bc_sub (Zero, z.re, &a, iscale); 
    }
  if ( (bc_is_zero(z.im)) )
    {
      *pm = a;
      return;
    }
  else if ( z.im->n_sign == PLUS )
    b = bc_copy_num (z.im);
  else
    {
      bc_init_num (&b);
      bc_sub (Zero, z.im, &b, iscale); 
    }
  bc_init_num (&q);
  if ( bc_compare (b, a) > 0 )
    {
      bc_divide (a, b, &q, iscale);
      bc_multiply (q, q, &a, iscale);
      bc_add (a, _one_, &q, iscale);
      bc_sqrt (&q, iscale);
      bc_multiply (b, q, pm, iscale);
    }
  else
    {
      /* a >= b ===> a > 0 */
      bc_divide (b, a, &q, iscale);
      bc_multiply (q, q, &b, iscale);
      bc_add (b, _one_, &q, iscale);
      bc_sqrt (&q, iscale);
      bc_multiply (a, q, pm, iscale);
    }
  bc_free_num (&q);
  bc_free_num (&b);
  bc_free_num (&a);
}

void Csub (Complex z1, Complex z2, Complex* pw, int iscale)
{
  bc_sub (z1.re, z2.re, &pw->re, iscale);
  bc_sub (z1.im, z2.im, &pw->im, iscale);
}

int cmp (const Real *p, const Real *q)
{
  return bc_compare (*p, *q);
}

int is0 (const Real *u)
{
  return bc_is_zero (*u);
}

int smart_cmp (const Complex* pz1, const Complex* pz2, int flag)
{
  if (flag == 0)
    return 1;
  else if (flag > 0)
    return (bc_compare (pz1->re, pz2->re) >= 0 &&
	    bc_compare (pz1->im, pz2->im) >= 0);
  else /* flag < 0 */
    return (bc_compare (pz1->re, pz2->re) <= 0 &&
	    bc_compare (pz1->im, pz2->im) <= 0);
}

void printno (Real u, int m)
{
  bc_print_num (u, out_char, m);
}

#ifdef _MPA_DEBUG
void debug_printno (Real u, int m)
{
  bc_print_num (u, out_char_stderr, m);
}
#endif

void delR (Real* px)
{
  bc_free_num (px);
}

void delC (Complex* pz)
{
  bc_free_num (&pz->re);
  bc_free_num (&pz->im);
}

void mpa_undef_epsilon (void)
{
  bc_free_num (&MEpsilon);
  bc_free_num (&Epsilon);  
}

void end_mpa(void)
{
  bc_free_num (&Inf);
  bc_free_num (&Zero);
  bc_end();
}

#elif defined(_USE_HPA)

const Real Zero = { {0x0, 0x0} };    /* xZero */
const Real Inf = { {0x7fff, 0x0} };  /* xPinf */

Real str2R (const char *q, char **endptr)
{
  return strtox (q, endptr);
}

Complex str2C (const char *q, char **endptr)
{
  return strtocx (q, endptr);
}

int cmp (const Real *p, const Real *q)
{
  return xprcmp (p, q);
}

int is0 (const Real *u)
{
  return xis0 (u);
}

int smart_cmp (const Complex* pz1, const Complex* pz2, int flag)
{
  if (flag == 0)
    return 1;
  else if (flag > 0)
    return (xprcmp (&pz1->re, &pz2->re) >= 0 &&
	    xprcmp (&pz1->im, &pz2->im) >= 0);
  else /* flag < 0 */
    return (xprcmp (&pz1->re, &pz2->re) <= 0 &&
	    xprcmp (&pz1->im, &pz2->im) <= 0);
}

Real divide (Real s, Real t)
{
  return xdiv (s, t);
}

void printno (Real u, int m)
{
  xprxpr (u, m);
}

Real Cabs (Complex z)
{
  return cxabs (z);
}

Complex  Csub (Complex z1, Complex z2)
{
  return cxadd (z1, z2, 1);
}

#else /* _USE_LDBL || _USE_DOUBLE */

#include<stdlib.h>
#include<ctype.h>
#include<float.h>
#include<math.h>

#ifdef _USE_LDBL

const Real Zero = 0.0L;
const Real Inf = LDBL_MAX;
extern long double strtold(const char *nptr, char **endptr);
#define STRTOR strtold
#define SQRT   sqrtl
#define FORMAT "%.*Le"

#else /* _USE_DOUBLE */

const Real Zero = 0.0;
const Real Inf = DBL_MAX;
#define STRTOR strtod
#define SQRT   sqrt
#define FORMAT "%.*e"

#endif

#define ABS(x) ((x) >= 0.0 ? (x) : -(x)) 

Real str2R (const char *q, char **endptr)
{
  return STRTOR (q, endptr);
}

Complex str2C (const char *q, char **endptr)
{
  Complex z = {Zero, Zero};
  char *ptr, *ptr2;

  z.re = STRTOR (q, &ptr);
  if ((endptr))
    *endptr = ptr;
  if (ptr != q)
    {
      if (*ptr == IU)
	{
	  z.im = z.re;
	  z.re = Zero;
	  if ((endptr))
	    *endptr = ptr + 1;
	}
      else
	{
	  while (isspace (*ptr))
	    ptr++;
	  if (*ptr == '+' || *ptr == '-')
	    {
	      z.im = STRTOR (ptr, &ptr2);
	      if (*ptr2 != IU)
		z.im = Zero;
	      else
		{
		  if ((endptr))
		    *endptr = ptr2 + 1;
		}
	    }
	  /*
	     else
	     : we have successfully read a real number
	     but there is no another number after it.
	     : So, we leave z.im set to zero and return z.
	  */
	}
    }
  else
    /* 
       : q does not contain any valid number
       ==> z.re is 0. Then we set z.im to
       0 and return z.
       We remark that, if endptr is
       not NULL, then *endptr == q.
    */
    z.im = Zero;
  return z;
}

int cmp (const Real *p, const Real *q)
{
  if (*p == *q)
    return 0;
  else
    return (*p > *q ? 1 : -1);
}

int is0 (const Real *u)
{
  return (*u == Zero);
}

int smart_cmp (const Complex* pz1, const Complex* pz2, int flag)
{
  if (flag == 0)
    return 1;
  else if (flag > 0)
    return (pz1->re >= pz2->re && pz1->im >= pz2->im);
  else /* flag < 0 */
    return (pz1->re <= pz2->re && pz1->im <= pz2->im);
}

Real divide (Real s, Real t)
{
  return s / t;
}

void printno (Real u, int m)
{
  printf (FORMAT, m, u);
}

Real Cabs (Complex z)
{
  Real a, b, q;

  a = ABS (z.re);
  if (z.im == 0.0)
    return a;
  else
    b = ABS (z.im);
  /* b > 0 ! */
  if (b > a)
    {
      q = a / b;
      return b * SQRT (1+q*q);
    }
  else
    {
      /* a >= b ===> a > 0 */
      q = b / a;
      return a * SQRT (1+q*q);
    }
}

Complex  Csub (Complex z1, Complex z2)
{
  Complex w;

  w.re = z1.re - z2.re;
  w.im = z1.im - z2.im;
  return w;
}

#endif
