/*
    Numdiff - compare putatively similar files, 
    ignoring small numeric differences
    Copyright (C) 2005,2006  Ivano Primi  <ivprimi@libero.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include<stdio.h>
#include<stdlib.h> /* for free() */
#ifdef ENABLE_NLS
#include<locale.h>
#endif
#include"numdiff.h"

/* See io.c */
extern int open_files (const char* file1, const char* file2, 
		       FILE** fpp1, FILE** fpp2);

/* See cmpfns.c */
extern int cmp_files (FILE* pf1, FILE* pf2, argslist* argl);

/* See options.c */
extern void print_version (const char* progname);
extern void print_help (const char* progname);
extern int setargs (int argc, char* argv[], argslist *list);

#ifdef _USE_MPA

void init_mpa_support (argslist* list)
{
  init_mpa();
  initR (&list->maxabserr);
  initR (&list->maxrelerr);
  initR (&list->Labserr);
  initR (&list->Crelerr);
  initR (&list->Lrelerr);
  initR (&list->Cabserr);
}

void dismiss_mpa_support (argslist* list)
{
  delR (&list->maxabserr);
  delR (&list->maxrelerr);
  delR (&list->Labserr);
  delR (&list->Crelerr);
  delR (&list->Lrelerr);
  delR (&list->Cabserr);
  end_mpa();
}

#endif /* _USE_MPA */

void print_statistics (const argslist* list)
{
  fputs (_("\nLargest absolute error in the set of relevant numerical differences:\n"),
	 stdout);
  printno (list->Labserr, DEF_LIM);
  fputs (_("\nCorresponding relative error =\n"), stdout);
  printno (list->Crelerr, DEF_LIM);

  fputs (_("\nLargest relative error in the set of relevant numerical differences:\n"),
	 stdout);
  printno (list->Lrelerr, DEF_LIM);
  fputs (_("\nCorresponding absolute error =\n"), stdout);
  printno (list->Cabserr, DEF_LIM);
  putchar ('\n');
}

int main (int argc, char* argv[])
{
  argslist list;

#ifdef ENABLE_NLS
  setlocale (LC_CTYPE, "");
  setlocale (LC_MESSAGES, "");
#endif
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#ifdef _USE_MPA
  init_mpa_support (&list);
#endif

  if ( setargs (argc, argv, &list) != 0 )
    {
#ifdef _USE_MPA
      dismiss_mpa_support (&list);
#endif
      if ((list.ifs))
	free ((void*)list.ifs);
      return -1;
    }
  else if ( (list.optmask & (_H_MASK | _V_MASK)) )
    {
      if ((list.optmask & _H_MASK))
	print_help(PACKAGE);
      if ((list.optmask & _V_MASK))
	print_version(PACKAGE);
#ifdef _USE_MPA
      dismiss_mpa_support (&list);
#endif
      if ((list.ifs))
	free ((void*)list.ifs);
      if (argc > 2)
	return -1;
      else
	return 0;
    }
  else
    {
      int test;
      FILE *fp1, *fp2;
      int qm = list.optmask & _Q_MASK;
#ifdef _USE_MPA
      struct numfmt defaults;

      defaults.dp = DP;
      defaults.thsep = THSEP;
      defaults.grouping = GROUPING;
      defaults.pos_sign = POS_SIGN;
      defaults.neg_sign = NEG_SIGN;
      defaults.ech = ECH;
      defaults.iu = IU;
      mpa_define_epsilon(list.iscale, &defaults);
#endif
      if ( open_files (list.file1, list.file2, &fp1, &fp2) != 0 )
	return -1;
      test = cmp_files (fp1, fp2, &list);
      if ((list.optmask & _SS_MASK) && test <= 1)
	print_statistics (&list);
      if (test == 0 && !qm)
	printf (_("\n+++  Files \"%s\" and \"%s\" are equal\n"),
		list.file1, list.file2);
      if (test == 1 && !qm)
	printf (_("\n+++  File \"%s\" differs from file \"%s\"\n"),
		list.file1, list.file2);
#ifdef _USE_MPA
      mpa_undef_epsilon();
      dismiss_mpa_support (&list);
#endif
      if ((list.ifs))
	free ((void*)list.ifs);
      if (fclose (fp1) == EOF)
	fprintf (stderr, _("\n***  Cannot close file \"%s\"\n"), list.file1);
      if (fclose (fp2) == EOF)
	fprintf (stderr, _("\n***  Cannot close file \"%s\"\n"), list.file2);
      return test;
    }
}
