/* Copyright (C) 1999--2001 Chris Vaill
   This file is part of normalize.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#define _POSIX_C_SOURCE 2

#include "config.h"

#include <stdio.h>
#if STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# if HAVE_STDLIB_H
#  include <stdlib.h>
# endif
# if HAVE_STRING_H
#  include <string.h>
# else
#  ifndef HAVE_STRCHR
#   define strchr index
#   define strrchr rindex
#  endif
#  ifndef HAVE_MEMCPY
#   define memcpy(d,s,n) bcopy((s),(d),(n))
#   define memmove(d,s,n) bcopy((s),(d),(n))
#  endif
# endif
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#if HAVE_FCNTL_H
# include <fcntl.h>
#endif

#ifdef ENABLE_NLS
# define _(msgid) gettext (msgid)
# include <libintl.h>
#else
# define _(msgid) (msgid)
#endif
#define N_(msgid) (msgid)

/*#include "mini_af.h"*/
#include "riff.h"
#include "common.h"

extern char *progname;

/* FIXME: make static or remove */
riff_chunk_t *
get_wav_data(riff_t *riff, struct wavfmt *fmt)
{
  riff_chunk_t *chnk;

  chnk = riff_chunk_read(riff);
  if (chnk == NULL) {
    fprintf(stderr, _("%s: error reading riff chunk\n"), progname);
    goto error2;
  }
  if (!riff_fourcc_equals(chnk->id, "RIFF")) {
    fprintf(stderr, _("%s: not a RIFF WAV file\n"), progname);
    goto error3;
  }
  riff_list_descend(riff, chnk);
  riff_chunk_unref(chnk);

  /* read format header */
  chnk = riff_chunk_read(riff);
  if (chnk == NULL) {
    fprintf(stderr, _("%s: error reading riff chunk\n"), progname);
    goto error2;
  }
  if (!riff_fourcc_equals(chnk->id, "fmt ")) {
    fprintf(stderr, _("%s: no format chunk found\n"), progname);
    goto error3;
  }
  fread(fmt, sizeof(struct wavfmt), 1, riff_chunk_get_stream(chnk));
  riff_chunk_unref(chnk);
#ifdef WORDS_BIGENDIAN
  fmt->format_tag        = bswap_16(fmt->format_tag);
  fmt->channels          = bswap_16(fmt->channels);
  fmt->samples_per_sec   = bswap_32(fmt->samples_per_sec);
  fmt->avg_bytes_per_sec = bswap_32(fmt->avg_bytes_per_sec);
  fmt->block_align       = bswap_16(fmt->block_align);
  fmt->bits_per_sample   = bswap_16(fmt->bits_per_sample);
#endif

  /*
   * Make sure we can handle this type of wav
   */
  if (fmt->format_tag != 1) {
    fprintf(stderr, _("%s: this is a non-PCM WAV file\n"), progname);
    errno = EINVAL;
    goto error3;
  }
  if (fmt->bits_per_sample > 32) {
    fprintf(stderr, _("%s: more than 32 bits per sample not implemented\n"),
	    progname);
    errno = EINVAL;
    goto error3;
  }

  /* read until data chunk */
  chnk = NULL;
  do {

    if (chnk) {
      riff_chunk_unref(chnk);
      chnk = NULL;
    }

    errno = 0;
    chnk = riff_chunk_read(riff);
    if (chnk == NULL) {
      fprintf(stderr, _("%s: no data chunk found\n"), progname);
      if (errno == 0)
	errno = EINVAL;
      goto error2;
    }

  } while (!riff_fourcc_equals(chnk->id, "data"));

  return chnk;

  /* error handling stuff */
 error3:
  riff_chunk_unref(chnk);
 error2:
  return NULL;
}

#if 0

struct _AFfilesetup {
  int format;
  int byte_order;
  int nchannels;
  double rate;
  int sample_format;
  int sample_width;
};

struct _AFfilehandle {
  FILE *file;
  riff_chunk_t *chnk;
  struct wavfmt fmt;
};

AFfilehandle
afOpenFile(const char *filename, const char *mode, AFfilesetup setup)
{
  int fd, flags;
  if (mode[0] == 'w')
    flags = O_WRONLY | O_CREAT | O_BINARY;
  else
    flags = O_RDONLY | O_BINARY;
  fd = open(filename, flags, 0666);
  if (fd == -1)
    return NULL;
  return afOpenFD(fd, mode, setup);
}

/* FIXME: fix write case */
AFfilehandle
afOpenFD(int fd, const char *mode, AFfilesetup setup)
{
  riff_t *riff;
  struct wavfmt *fmt;
  int bytes_per_sample;
  AFfilehandle newfh;

  newfh = (AFfilehandle)malloc(sizeof(struct _AFfilehandle));
  if (newfh == NULL) {
    fprintf(stderr, _("%s: unable to malloc\n"), progname);
    goto error1;
  }

  if (mode[0] == 'w') {

    /*
     * open for writing
     */

    riff = riff_new(fd, RIFF_WRONLY);
    if (riff == NULL) {
      fprintf(stderr, _("%s: error making riff object\n"), progname);
      goto error2;
    }

    /* check that we can do what the setup asks */
    if (setup->format != AF_FILE_WAVE
	|| (setup->sample_format != AF_SAMPFMT_TWOSCOMP
	    && setup->sample_format != AF_SAMPFMT_UNSIGNED)
	|| setup->byte_order != AF_BYTEORDER_LITTLEENDIAN
	|| setup->nchannels < 1) {
      /* bad setup error */
    }
    switch (setup->sample_format) {
    case AF_SAMPFMT_TWOSCOMP:
      if (setup->sample_width <= 8) {
	/* bad setup error */
      }
      break;
    case AF_SAMPFMT_UNSIGNED:
      if (setup->sample_width > 8) {
	/* bad setup error */
      }
      break;
    default:
      /* bad setup error */
      ;
    }

    /* construct WAV fmt header from setup struct */
    fmt = &newfh->fmt;
    fmt->format_tag = 1;
    fmt->channels = setup->nchannels;
    fmt->samples_per_sec = setup->rate;
    fmt->bits_per_sample = setup->sample_width;
    bytes_per_sample = (fmt->bits_per_sample - 1) / 8 + 1;
    fmt->block_align = 0;
    fmt->avg_bytes_per_sec = (bytes_per_sample * fmt->channels
			      * fmt->samples_per_sec);
#ifdef WORDS_BIGENDIAN
    fmt->format_tag        = bswap_16(fmt->format_tag);
    fmt->channels          = bswap_16(fmt->channels);
    fmt->samples_per_sec   = bswap_32(fmt->samples_per_sec);
    fmt->avg_bytes_per_sec = bswap_32(fmt->avg_bytes_per_sec);
    fmt->block_align       = bswap_16(fmt->block_align);
    fmt->bits_per_sample   = bswap_16(fmt->bits_per_sample);
#endif
    if (riff_chunk_write(riff, riff_string_to_fourcc("fmt "),
			 sizeof(struct wavfmt), fmt) == -1) {
      fprintf(stderr, _("%s: error writing WAV header\n"), progname);
      goto error3;
    }

    /* FIXME: need to riff_chunk_get_write_stream() or
       riff_chunk_open() or something */

  } else {

    /*
     * open for reading
     */

    riff = riff_new(fd, RIFF_RDONLY);
    if (riff == NULL) {
      fprintf(stderr, _("%s: error making riff object\n"), progname);
      goto error2;
    }

    /* WAV format info will be passed back */
    fmt = &newfh->fmt;

    newfh->chnk = get_wav_data(riff, fmt);
    if (newfh->chnk == NULL) {
      fprintf(stderr, _("%s: error getting wav data\n"), progname);
      goto error3;
    }
#if DEBUG
    if (verbose >= VERBOSE_DEBUG) {
      fprintf(stderr,
	      "fmt chunk for %s:\n"
	      "  format_tag:        %u\n"
	      "  channels:          %u\n"
	      "  samples_per_sec:   %u\n"
	      "  avg_bytes_per_sec: %u\n"
	      "  block_align:       %u\n"
	      "  bits_per_sample:   %u\n",
	      filename, fmt->format_tag, fmt->channels, fmt->samples_per_sec,
	      fmt->avg_bytes_per_sec, fmt->block_align, fmt->bits_per_sample);
    }
#endif

    newfh->file = fdopen(fd, "rb");
    if (newfh->file == NULL) {
      fprintf(stderr, _("%s: failed fdopen: %s\n"),
	      progname, strerror(errno));
      goto error4;
    }
    fseek(newfh->file, newfh->chnk->offset + 8, SEEK_SET);
  }

  riff_unref(riff);

  return newfh;

  /* error handling stuff */
 error4:
  riff_chunk_unref(newfh->chnk);
 error3:
  riff_unref(riff);
 error2:
  free(newfh);
 error1:
  return NULL;
}

int
afCloseFile(AFfilehandle fh)
{
  if (fh) {
    riff_chunk_unref(fh->chnk);
    fclose(fh->file);
    free(fh);
  }
  return 0;
}

AFfilesetup
afNewFileSetup(void)
{
  AFfilesetup retval;
  retval = (AFfilesetup)malloc(sizeof(struct _AFfilesetup));
  if (retval) {
    /* set defaults: WAV, 44.1k, mono, 16-bit LE */
    retval->format = AF_FILE_WAVE;
    retval->byte_order = AF_BYTEORDER_LITTLEENDIAN;
    retval->nchannels = 1;
    retval->rate = 44100;
    retval->sample_format = AF_SAMPFMT_TWOSCOMP;
    retval->sample_width = 16;
  }
  return retval;
}

void
afFreeFileSetup(AFfilesetup setup)
{
  free(setup);
}

/* FIXME: implement */
int
afReadFrames(AFfilehandle fh, int track, void *buffer, int frameCount)
{
  return 0;
}

/* FIXME: implement */
int
afWriteFrames(AFfilehandle file, int track, void *buffer, int frameCount)
{
  return 0;
}

/* FIXME: implement */
float
afGetFrameSize(AFfilehandle file, int track, int expand3to4)
{
  return 0;
}

/* FIXME: implement */
AFerrfunc
afSetErrorHandler(AFerrfunc errorFunction)
{
  return NULL;
}

void
afInitFileFormat(AFfilesetup setup, int format)
{
  setup->format = format;
}

void
afInitByteOrder(AFfilesetup setup, int track, int byte_order)
{
  setup->byte_order = byte_order;
}

void
afInitChannels(AFfilesetup setup, int track, int nchannels)
{
  setup->nchannels = nchannels;
}

void
afInitRate(AFfilesetup setup, int track, double rate)
{
  setup->rate = rate;
}

void
afInitSampleFormat(AFfilesetup setup, int track, int sample_format,
		   int sample_width)
{
  setup->sample_format = sample_format;
  setup->sample_width = sample_width;
}


#endif
