/* Copyright (C) 1999--2001 Chris Vaill
   This file is part of normalize.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#define _POSIX_C_SOURCE 2

#include "config.h"

#if STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# if HAVE_STDLIB_H
#  include <stdlib.h>
# endif
# if HAVE_STRING_H
#  include <string.h>
# else
#  ifndef HAVE_STRCHR
#   define strchr index
#   define strrchr rindex
#  endif
#  ifndef HAVE_MEMCPY
#   define memcpy(d,s,n) bcopy((s),(d),(n))
#   define memmove(d,s,n) bcopy((s),(d),(n))
#  endif
# endif
#endif
#if HAVE_MATH_H
# include <math.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif

#ifdef ENABLE_NLS
# define _(msgid) gettext (msgid)
# include <libintl.h>
#else
# define _(msgid) (msgid)
#endif
#define N_(msgid) (msgid)

#include "riff.h"
#include "common.h"

extern void progress_callback(char *prefix, float fraction_completed);
extern void *xmalloc(size_t size);

/* FIXME: remove, use audiofile-style interface */
extern riff_chunk_t *get_wav_data(riff_t *riff, struct wavfmt *fmt);

extern char *progname;
extern int verbose;

static __inline__ long
get_sample(unsigned char *pdata, int bytes_per_sample)
{
  long sample;

  switch(bytes_per_sample) {
  case 1:
    sample = *pdata - 128;
    break;
  case 2:
#ifdef WORDS_BIGENDIAN
    sample = *((int8_t *)pdata + 1) << 8;
    sample |= *((int8_t *)pdata) & 0xFF;
#else
    sample = *((int16_t *)pdata);
#endif
    break;
  case 3:
    sample = *((int8_t *)pdata + 2) << 16;
    sample |= (*((int8_t *)pdata + 1) << 8) & 0xFF00;
    sample |= *((int8_t *)pdata) & 0xFF;
    break;
  case 4:
    sample = *((int32_t *)pdata);
#ifdef WORDS_BIGENDIAN
    sample = bswap_32(sample);
#endif
    break;
  default:
    /* shouldn't happen */
    fprintf(stderr,
	    _("%s: I don't know what to do with %d bytes per sample\n"),
	    progname, bytes_per_sample);
    sample = 0;
  }

  return sample;
}


static __inline__ void
put_sample(long sample, unsigned char *pdata, int bytes_per_sample)
{
  switch(bytes_per_sample) {
  case 1:
    *pdata = sample + 128;
    break;
  case 2:
#ifdef WORDS_BIGENDIAN
    sample = bswap_16(sample);
#endif
    *((int16_t *)pdata) = (int16_t)sample;
    break;
  case 3:
    *pdata = (unsigned char)sample;
    *(pdata + 1) = (unsigned char)(sample >> 8);
    *(pdata + 2) = (unsigned char)(sample >> 16);
    break;
  case 4:
#ifdef WORDS_BIGENDIAN
    sample = bswap_32(sample);
#endif
    *((int32_t *)pdata) = (int32_t)sample;
    break;
  default:
    /* shouldn't happen */
    fprintf(stderr,
	    _("%s: I don't know what to do with %d bytes per sample\n"),
	    progname, bytes_per_sample);
  }
}


typedef struct {
  double *buf;
  int buflen;  /* elements allocated to buffer */
  int start;   /* index of first element in buffer */
  int n;       /* num of elements in buffer */
} datasmooth_t;

/*
 * Takes a full smoothing window, and returns the value of the center
 * element, smoothed.  Currently, just does a mean filter, but we could
 * do a median or gaussian filter here instead.
 */
static __inline__ double
get_smoothed_data(datasmooth_t *s)
{
  int i;
  /*int center = (s->n + 1) / 2;*/
  double smoothed;

  smoothed = 0;
  for (i = 0; i < s->n; i++)
    smoothed += s->buf[i];
  smoothed = smoothed / s->n;

  return smoothed;
}


/*
 * Get the maximum power level of the wav file
 * (and the peak sample, if ppeak is not NULL)
 */
double
signal_max_power(int fd, char *filename, struct signal_info *psi)
{
  riff_t *riff;
  riff_chunk_t *chnk;
  struct wavfmt *fmt;
  unsigned int nsamples;

  int bytes_per_sample, stride;
  int last_window;
  unsigned int windowsz;
  unsigned int win_start, old_start, win_end, old_end;

  int i, c, offset;
  long sample, samplemax, samplemin;
  double *sums;
  double pow, maxpow;
  datasmooth_t *powsmooth;

  float progress, last_progress = 0.0;
  char prefix_buf[18];

  FILE *in;
  unsigned char *data_buf = NULL;
  int filled_sz;


  riff = riff_new(fd, RIFF_RDONLY);
  if (riff == NULL) {
    fprintf(stderr, _("%s: error making riff object\n"), progname);
    goto error1;
  }

  /* WAV format info will be passed back */
  fmt = &psi->fmt;

  chnk = get_wav_data(riff, fmt);
  if (chnk == NULL) {
    fprintf(stderr, _("%s: error getting wav data\n"), progname);
    goto error2;
  }
#if DEBUG
  if (verbose >= VERBOSE_DEBUG) {
    fprintf(stderr,
	    "fmt chunk for %s:\n"
	    "  format_tag:        %u\n"
	    "  channels:          %u\n"
	    "  samples_per_sec:   %u\n"
	    "  avg_bytes_per_sec: %u\n"
	    "  block_align:       %u\n"
	    "  bits_per_sample:   %u\n",
	    filename, fmt->format_tag, fmt->channels, fmt->samples_per_sec,
	    fmt->avg_bytes_per_sec, fmt->block_align, fmt->bits_per_sample);
  }
#endif
  windowsz = (unsigned int)(fmt->samples_per_sec / 100);

  bytes_per_sample = (fmt->bits_per_sample - 1) / 8 + 1;
  samplemax = (1 << (bytes_per_sample * 8 - 1)) - 1;
  samplemin = -samplemax - 1;
  nsamples = chnk->size / bytes_per_sample / fmt->channels;
  /* initialize peaks to effectively -inf and +inf */
  psi->max_sample = samplemin;
  psi->min_sample = samplemax;
#if DEBUG
  if (verbose >= VERBOSE_DEBUG) {
    fprintf(stderr,
	    "bytes_per_sample: %d nsamples: %d\n"
	    "samplemax: %ld samplemin: %ld\n",
	    bytes_per_sample, nsamples, samplemax, samplemin);
  }
#endif

  sums = (double *)xmalloc(fmt->channels * sizeof(double));
  for (c = 0; c < fmt->channels; c++)
    sums[c] = 0;

  data_buf = (unsigned char *)xmalloc(windowsz
				      * fmt->channels * bytes_per_sample);

  /* set up smoothing window buffer */
  powsmooth = (datasmooth_t *)xmalloc(fmt->channels * sizeof(datasmooth_t));
  for (c = 0; c < fmt->channels; c++) {
    powsmooth[c].buflen = 100; /* use a 100-element (1 second) window */
    powsmooth[c].buf = (double *)xmalloc(powsmooth[c].buflen * sizeof(double));
    powsmooth[c].start = powsmooth[c].n = 0;
  }

  /* initialize progress meter */
  if (verbose >= VERBOSE_PROGRESS) {
    if (strrchr(filename, '/') != NULL) {
      filename = strrchr(filename, '/');
      filename++;
    }
    strncpy(prefix_buf, filename, 17);
    prefix_buf[17] = 0;
    progress_callback(prefix_buf, 0.0);
    last_progress = 0.0;
  }

  in = fdopen(fd, "rb");
  if (in == NULL) {
    fprintf(stderr, _("%s: failed fdopen: %s\n"),
	    progname, strerror(errno));
    goto error7;
  }
  fseek(in, chnk->offset + 8, SEEK_SET);


  /*
   * win_start, win_end, old_start, windowsz, interval, and i are in
   * units of samples.  c is in units of channels.
   *
   * The actual window extends from win_start to win_end - 1, inclusive.
   */
  old_start = win_start = 0;
  win_end = 0;
  last_window = FALSE;
  maxpow = 0.0;

  stride = fmt->channels * bytes_per_sample;

  do {

    /* set up the window end */
    old_end = win_end;
    win_end = win_start + windowsz;
    if (win_end >= nsamples) {
      win_end = nsamples;
      last_window = TRUE;
    }

    /* read a windowsz sized chunk */
    filled_sz = fread(data_buf, bytes_per_sample,
		      windowsz * fmt->channels, in);
    for (c = 0; c < fmt->channels; c++) {
      sums[c] = 0;
      offset = c * bytes_per_sample;
      for (i = 0; i < (win_end - win_start); i++) {
	sample = get_sample(data_buf + offset, bytes_per_sample);
	offset += stride;
	sums[c] += sample * (double)sample;
	/* track peak */
	if (sample > psi->max_sample)
	  psi->max_sample = sample;
	if (sample < psi->min_sample)
	  psi->min_sample = sample;
      }
    }

    /* compute power for each channel */
    for (c = 0; c < fmt->channels; c++) {
      int end;
      pow = sums[c] / (double)(win_end - win_start);

      end = (powsmooth[c].start + powsmooth[c].n) % powsmooth[c].buflen;
      powsmooth[c].buf[end] = pow;
      if (powsmooth[c].n == powsmooth[c].buflen) {
	powsmooth[c].start = (powsmooth[c].start + 1) % powsmooth[c].buflen;
	pow = get_smoothed_data(&powsmooth[c]);
	if (pow > maxpow)
	  maxpow = pow;
      } else {
	powsmooth[c].n++;
      }
    }

    /* update progress meter */
    if (verbose >= VERBOSE_PROGRESS) {
      if (nsamples - windowsz == 0)
	progress = 0;
      else
	progress = (win_end - windowsz) / (float)(nsamples - windowsz);
      /*progress = win_end / ((nsamples - windowsz) / (float)windowsz);*/
      if (progress >= last_progress + 0.01) {
	/*printf(_("progress: %f       \n"), progress);*/
	progress_callback(prefix_buf, progress);
	last_progress += 0.01;
      }
    }

    /* slide the window ahead */
    old_start = win_start;
    win_start += windowsz;

  } while (!last_window);

  if (maxpow < EPSILON) {
    /*
     * Either this whole file has zero power, or was too short to ever
     * fill the smoothing buffer.  In the latter case, we need to just
     * get maxpow from whatever data we did collect.
     */
    for (c = 0; c < fmt->channels; c++) {
      pow = get_smoothed_data(&powsmooth[c]);
      if (pow > maxpow)
	maxpow = pow;
    }
  }

  for (c = 0; c < fmt->channels; c++)
    free(powsmooth[c].buf);
  free(powsmooth);
  free(data_buf);
  free(sums);
  riff_chunk_unref(chnk);
  riff_unref(riff);

  /* scale the pow value to be in the range 0.0 -- 1.0 */
  maxpow = maxpow / (samplemin * (double)samplemin);

  /* fill in the signal_info struct */
  psi->level = sqrt(maxpow);
  if (-psi->min_sample > psi->max_sample)
    psi->peak = psi->min_sample / (double)samplemin;
  else
    psi->peak = psi->max_sample / (double)samplemax;

  return maxpow;

  /* error handling stuff */
 error7:
  for (c = 0; c < fmt->channels; c++)
    free(powsmooth[c].buf);
  /*error6:*/
  free(powsmooth);
  /*error5:*/
  free(data_buf);
  /*error4:*/
  free(sums);
  /*error3:*/
  riff_chunk_unref(chnk);
 error2:
  riff_unref(riff);
 error1:
  return -1.0;
}


/*
 * Get the maximum power level of the data read from a stream
 * (and the peak sample, if ppeak is not NULL)
 */
double
signal_max_power_stream(FILE *in, char *filename, struct signal_info *psi)

{
  struct wavfmt *fmt;
  int bytes_per_sample;
  int last_window;
  unsigned int windowsz;
  unsigned int win_start, old_start, win_end, old_end;

  int i, c;
  long sample, samplemax, samplemin;
  double *sums;
  double pow, maxpow;
  datasmooth_t *powsmooth;

  char prefix_buf[18];

  unsigned char *data_buf = NULL;
  int filled_sz;

  if (filename == NULL || strcmp(filename, "-") == 0)
    filename = "STDIN";

  /* WAV format info must be passed to us in psi->fmt */
  fmt = &psi->fmt;

  windowsz = (unsigned int)(fmt->samples_per_sec / 100);

  bytes_per_sample = (fmt->bits_per_sample - 1) / 8 + 1;
  samplemax = (1 << (bytes_per_sample * 8 - 1)) - 1;
  samplemin = -samplemax - 1;
  /* initialize peaks to effectively -inf and +inf */
  psi->max_sample = samplemin;
  psi->min_sample = samplemax;

  sums = (double *)xmalloc(fmt->channels * sizeof(double));
  for (c = 0; c < fmt->channels; c++)
    sums[c] = 0;

  data_buf = (unsigned char *)xmalloc(windowsz
				      * fmt->channels * bytes_per_sample);

  /* set up smoothing window buffer */
  powsmooth = (datasmooth_t *)xmalloc(fmt->channels * sizeof(datasmooth_t));
  for (c = 0; c < fmt->channels; c++) {
    powsmooth[c].buflen = 100; /* use a 100-element (1 second) window */
    powsmooth[c].buf = (double *)xmalloc(powsmooth[c].buflen * sizeof(double));
    powsmooth[c].start = powsmooth[c].n = 0;
  }

  /* initialize progress meter */
  if (verbose >= VERBOSE_PROGRESS) {
    if (strrchr(filename, '/') != NULL) {
      filename = strrchr(filename, '/');
      filename++;
    }
    strncpy(prefix_buf, filename, 17);
    prefix_buf[17] = 0;
    progress_callback(prefix_buf, 0.0);
  }


  /*
   * win_start, win_end, old_start, windowsz, interval, and i are in
   * units of samples.  c is in units of channels.
   *
   * The actual window extends from win_start to win_end - 1, inclusive.
   */
  old_start = win_start = 0;
  win_end = 0;
  last_window = FALSE;
  maxpow = 0.0;

  do {

    /* set up the window end */
    old_end = win_end;
    win_end = win_start + windowsz;

    /* read a windowsz sized chunk */
    filled_sz = fread(data_buf, bytes_per_sample,
		      windowsz * fmt->channels, in);

    /* if we couldn't read a complete chunk, then this is the last chunk */
    if (filled_sz < windowsz * fmt->channels) {
      win_end = win_start + (filled_sz / fmt->channels);
      last_window = TRUE;
    }

    for (c = 0; c < fmt->channels; c++) {
      sums[c] = 0;
      for (i = 0; i < (win_end - win_start); i++) {
	sample = get_sample(data_buf + (i * fmt->channels * bytes_per_sample)
			    + (c * bytes_per_sample), bytes_per_sample);
	sums[c] += sample * (double)sample;
	/* track peak */
	if (sample > psi->max_sample)
	  psi->max_sample = sample;
	if (sample < psi->min_sample)
	  psi->min_sample = sample;
      }
    }

    /* compute power for each channel */
    for (c = 0; c < fmt->channels; c++) {
      int end;
      pow = sums[c] / (double)(win_end - win_start);

      end = (powsmooth[c].start + powsmooth[c].n) % powsmooth[c].buflen;
      powsmooth[c].buf[end] = pow;
      if (powsmooth[c].n == powsmooth[c].buflen) {
	powsmooth[c].start = (powsmooth[c].start + 1) % powsmooth[c].buflen;
	pow = get_smoothed_data(&powsmooth[c]);
	if (pow > maxpow)
	  maxpow = pow;
      } else {
	powsmooth[c].n++;
      }
    }

    /* slide the window ahead */
    old_start = win_start;
    win_start += windowsz;

  } while (!last_window);

  if (maxpow < EPSILON) {
    /*
     * Either this whole file has zero power, or was too short to ever
     * fill the smoothing buffer.  In the latter case, we need to just
     * get maxpow from whatever data we did collect.
     */
    for (c = 0; c < fmt->channels; c++) {
      pow = get_smoothed_data(&powsmooth[c]);
      if (pow > maxpow)
	maxpow = pow;
    }
  }

  for (c = 0; c < fmt->channels; c++)
    free(powsmooth[c].buf);
  free(powsmooth);
  free(data_buf);
  free(sums);

  /* scale the pow value to be in the range 0.0 -- 1.0 */
  maxpow = maxpow / (samplemin * (double)samplemin);

  /* fill in the signal_info struct */
  psi->level = sqrt(maxpow);
  if (-psi->min_sample > psi->max_sample)
    psi->peak = psi->min_sample / (double)samplemin;
  else
    psi->peak = psi->max_sample / (double)samplemax;

  return maxpow;
}
