/* Copyright (C) 1999, 2000 Chris Vaill
   This file is part of normalize.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#define _POSIX_C_SOURCE 2

#include "config.h"

#include <stdio.h>
#include <errno.h>
#include <time.h>

#if STDC_HEADERS
# include <stdlib.h>
# include <unistd.h>
# include <string.h>
# include <math.h>
#else
# ifndef HAVE_STRCHR
#  define strchr index
#  define strrchr rindex
# endif
char *strchr(); char *strrchr();
# ifndef HAVE_MEMCPY
#  define memcpy(d,s,n) bcopy((s),(d),(n))
#  define memmove(d,s,n) bcopy((s),(d),(n))
# endif
#endif

#if HAVE_FCNTL_H
# include <fcntl.h>
#endif
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_SYS_MMAN_H
# include <sys/mman.h>
#endif

#if HAVE_BYTESWAP_H
# include <byteswap.h>
#else
# define bswap_16(x) \
    ((((x) >> 8) & 0xff) | (((x) & 0xff) << 8))
# define bswap_32(x) \
    ((((x) & 0xff000000) >> 24) | (((x) & 0x00ff0000) >>  8) |       \
     (((x) & 0x0000ff00) <<  8) | (((x) & 0x000000ff) << 24))
#endif /* HAVE_BYTESWAP_H */

#if HAVE_GETHOSTNAME
extern int gethostname();
#else
# define gethostname(a, b) strcpy((a), "?")
#endif

#if HAVE_LOCALE_H
# include <locale.h>
#endif

#if ENABLE_NLS
# include <libintl.h>
# define _(s) gettext (s)
#else
# define _(s) (s)
#endif

#include "getopt.h"

#include "riff.h"


#ifndef MIN
# define MIN(a,b) ((a)<(b)?(a):(b))
#endif
#ifndef MAX
# define MAX(a,b) ((a)>(b)?(a):(b))
#endif

#ifndef FALSE
# define FALSE  (0)
#endif
#ifndef TRUE
# define TRUE   (!FALSE)
#endif

#ifndef EPSILON
# define EPSILON 0.00000000001
#endif

/* warn about clipping if we clip more than this fraction of the samples */
#define CLIPPING_WARN_THRESH 0.001

struct wavfmt {
  uint16_t format_tag;              /* Format category */
  uint16_t channels;                /* Number of channels */
  uint32_t samples_per_sec;         /* Sampling rate */
  uint32_t avg_bytes_per_sec;       /* For buffer estimation */
  uint16_t block_align;             /* Data block size */

  uint16_t bits_per_sample;         /* Sample size */
};

struct progress_struct {
  time_t file_start;   /* what time we started processing the file */
  time_t batch_start;  /* what time we started processing the batch */
  off_t *file_sizes;   /* sizes of each file, in kb */
  off_t batch_size;    /* sum of all file sizes, in kb */
  off_t finished_size; /* sum of sizes of all completed files, in kb */
  int on_file;         /* the index of the file we're working on */
} progress_info;

double analyze_levels(double *levels, int nfiles, double threshold);
double signal_max_power(int fd, char *filename);
int apply_gain(int read_fd, int write_fd, char *filename, double gain);
void *xmalloc(size_t size);

extern char version[];
char *progname;

#define VERBOSE_QUIET    0
#define VERBOSE_PROGRESS 1
#define VERBOSE_INFO     2
#define VERBOSE_DEBUG    3
int verbose = VERBOSE_PROGRESS;
int do_apply_gain = TRUE;

void
usage()
{
  fprintf(stderr, _("\
Usage: %s [OPTION]... [FILE]...\n\
Normalize volume of multiple WAV files\n\
\n\
  -a, --amplitude=AMP  normalize the RMS volume to the target amplitude\n\
                         AMP; must be between 0.0 and 1.0 [default 0.25]\n\
  -n, --no-adjust      compute and output the volume adjustment, but\n\
                         don't apply it to any of the files\n\
  -g, --gain=ADJ       don't compute the volume adjustment, just apply\n\
                         adjustment ADJ to the files\n\
  -b, --batch          batch mode: average the levels of all files given\n\
                         on the command line, and use one adjustment, based\n\
                         on the average level, for all files\n\
  -m, --mix            mix mode: average the levels of all files given\n\
                         on the command line, and normalize volume of\n\
                         each file to the average level\n\
  -t, --threshold=THR  in batch mode, throw out any level values more\n\
                         than THR decibels different from the average.\n\
                         By default, use twice the standard deviation of\n\
                         all the power levels.\n\
  -v, --verbose        increase verbosity\n\
  -q, --quiet          quiet (decrease verbosity to zero)\n\
  -V, --version        display version information and exit\n\
  -h, --help           display this help and exit\n\
\n\
Report bugs to <cvaill@cs.columbia.edu>.\n"), progname);
}

int
main(int argc, char *argv[])
{
  int fd, fd2, c, i, nfiles;
  double *levels;
  double level, gain = 1.0;
  char **fnames;
  struct stat st;

  /* arguments */
  double target = 0.25;
  double threshold = -1.0; /* in decibels */
  int do_compute_levels = TRUE;
  int normalize_separately = TRUE;
  int mix_mode = FALSE;

  struct option longopts[] = {
    {"help", 0, NULL, 'h'},
    {"version", 0, NULL, 'V'},
    {"no-adjust", 0, NULL, 'n'},
    {"quiet", 0, NULL, 'q'},
    {"verbose", 0, NULL, 'v'},
    {"batch", 0, NULL, 'b'},
    {"amplitude", 1, NULL, 'a'},
    {"threshold", 1, NULL, 't'},
    {"gain", 1, NULL, 'g'},
    {"mix", 1, NULL, 'm'},
    {NULL, 0, NULL, 0}
  };

  /* get program name */
  if ((progname = strrchr(argv[0], '/')) == NULL)
    progname = argv[0];
  else
    progname++;
  if (strlen(progname) > 16)
    progname[16] = '\0';

#if ENABLE_NLS
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);
#endif

  /* get args */
  while ((c = getopt_long(argc, argv, "hVnvqbmg:a:t:",longopts,NULL)) != EOF) {
    switch(c) {
    case 'a':
      target = strtod(optarg, NULL);
      if (target < 0 || target > 1.0) {
	usage();
	exit(1);
      }
      break;
    case 't':
      /* a negative threshold means don't use threshold (use 2*stddev) */
      threshold = strtod(optarg, NULL);
      break;
    case 'g':
      gain = strtod(optarg, NULL);
      do_compute_levels = FALSE;
      normalize_separately = FALSE;
      if (gain < 0) {
	usage();
	exit(1);
      }
      break;
    case 'n':
      do_apply_gain = FALSE;
      break;
    case 'b':
      normalize_separately = FALSE;
      break;
    case 'm':
      mix_mode = TRUE;
      break;
    case 'v':
      verbose++;
      break;
    case 'q':
      verbose = VERBOSE_QUIET;
      break;
    case 'V':
      fprintf(stderr, "normalize %s\n", version);
      exit(0);
    case 'h':
      usage();
      exit(0);
    default:
      usage();
      exit(1);
    }
  }
  if (mix_mode && !normalize_separately) {
    fprintf(stderr,
	    _("%s: error: the -m and -b options are mutually exclusive\n"),
	    progname);
    exit(1);
  }
  if (optind >= argc) {
    usage();
    exit(1);
  }


  /*
   * get sizes of all files, for progress calculation
   */
  nfiles = 0;
  progress_info.batch_size = 0;
  fnames = (char **)xmalloc((argc - optind) * sizeof(char *));
  progress_info.file_sizes = (off_t *)xmalloc((argc - optind) * sizeof(off_t));
  for (i = optind; i < argc; i++) {
    if (stat(argv[i], &st) == -1) {
      fprintf(stderr, _("%s: file %s: %s\n"),
	      progname, argv[i], strerror(errno));
    } else {
      /* we want the size of the data chunk in kilobytes, so subtract
         the size of the wav header and divide by size of kb */
      progress_info.file_sizes[nfiles] = (st.st_size - 36) / 1024;
      /* add the size of the file, in kb */
      progress_info.batch_size += progress_info.file_sizes[nfiles];
      fnames[nfiles] = argv[i];
      nfiles++;
    }
  }
  if (nfiles == 0) {
    fprintf(stderr, _("%s: no files!\n"), progname);
    return 1;
  }


  /*
   * allocate space to store levels
   */
  levels = (double *)xmalloc(nfiles * sizeof(double));


  /*
   * Compute the power levels of the files, then figure out the
   * necessary gain to get the power value to the target.
   */
  if (do_compute_levels) {

    if (verbose >= VERBOSE_PROGRESS)
      fprintf(stderr, _("Computing levels...\n"));

    progress_info.batch_start = time(NULL);
    progress_info.finished_size = 0;

    for (i = 0; i < nfiles; i++) {

      fd = open(fnames[i], O_RDONLY);
      if (fd == -1) {
	fprintf(stderr, _("%s: error opening %s: %s\n"), progname, fnames[i],
		strerror(errno));
	continue;
      }

      progress_info.file_start = time(NULL);
      progress_info.on_file = i;
      errno = 0;
      level = signal_max_power(fd, fnames[i]);
      if (level < -EPSILON) {
	fprintf(stderr, _("%s: error reading %s: %s\n"), progname, fnames[i],
		strerror(errno));
	goto error1_1;
      }
      if (level < EPSILON) {
	if (verbose >= VERBOSE_PROGRESS) {
	  fprintf(stderr,
		  "\r                                     "
		  "                                     \r");
	  fprintf(stderr,
		  _("File %s has zero power, ignoring...\n"), fnames[i]);
	}
	goto error1_1;
      }

      progress_info.finished_size += progress_info.file_sizes[i];

      levels[i] = level;
      if (!do_apply_gain) {

	/* clear the progress meter first */
	if (verbose >= VERBOSE_PROGRESS)
	  fprintf(stderr,
		  "\r                                     "
		  "                                     \r");

	printf("%f\t%s\n", sqrt(levels[i]), fnames[i]);

      } else if (verbose >= VERBOSE_INFO) {
	fprintf(stderr,
		"\r                                     "
		"                                     \r");
	fprintf(stderr, _("Level for %s: %0.4f\n"),
		fnames[i], sqrt(levels[i]));
      }

    error1_1:
      close(fd);
    }

    /* we're done with the level calculation progress meter, so go to
       next line */
    if (verbose == VERBOSE_PROGRESS && do_apply_gain)
      fputc('\n', stderr);

    if (!normalize_separately || mix_mode) {
      level = analyze_levels(levels, nfiles, threshold);

      if (!do_apply_gain)
	printf(_("%f\taverage level\n"), sqrt(level));
      else if (verbose >= VERBOSE_INFO)
	printf(_("Average level: %f\n"), sqrt(level));

      /* level refers to signal power, but target is an amplitude, so we
	 take the square root of the power to get the RMS amplitude */
      gain = target / sqrt(level);
    } /* end of if (!normalize_separately) */

    /*
     * For mix mode, we set the target to the average level
     */
    if (mix_mode)
      target = sqrt(level);

  } /* end of if (do_compute_levels) */


  /*
   * Apply the gain
   */
  if (do_apply_gain) {
    struct stat stbuf;
    char *tmpfile, *p;
    int tmpnum;

    if (verbose >= VERBOSE_PROGRESS && !normalize_separately)
      printf(_("Applying adjustment of %0.3f...\n"), gain);

    progress_info.batch_start = time(NULL);
    progress_info.finished_size = 0;

    for (i = 0; i < nfiles; i++) {

      fd = open(fnames[i], O_RDONLY);
      if (fd == -1) {
	fprintf(stderr, _("%s: error opening %s: %s\n"), progname, fnames[i],
		strerror(errno));
	continue;
      }
#if 1
      fstat(fd, &stbuf);
      /* Create temporary file name, and open it for writing.  We
       * don't use tmpnam() because we want it to be in the same
       * directory (and therefore, in the same filesystem, for a fast
       * rename). */
      tmpfile = (char *)xmalloc(strlen(fnames[i]) + 32);
      strcpy(tmpfile, fnames[i]);
      if ((p = strrchr(tmpfile, '/')) == NULL) {
	tmpfile[0] = '\0';
	p = tmpfile;
      } else {
	p++;
	*p = '\0';
      }
      strcat(p, "_norm"); p += 5;
      gethostname(p, 20);
      p = tmpfile + strlen(tmpfile);
      tmpnum = getpid();
      do {
	sprintf(p, "%d", tmpnum);
	fd2 = open(tmpfile, O_WRONLY | O_CREAT | O_EXCL | O_TRUNC,
		   stbuf.st_mode);
	tmpnum++;
      } while (fd2 == -1 && errno == EEXIST);
      if (fd2 == -1) {
	fprintf(stderr, _("%s: error opening temp file: %s\n"), progname,
		strerror(errno));
	close(fd);
	free(tmpfile);
	continue;
      }
#else
      fd2 = open(fnames[i], O_WRONLY);
      if (fd2 == -1) {
	fprintf(stderr, _("%s: error opening %s: %s\n"), progname, fnames[i],
		strerror(errno));
	close(fd);
	continue;
      }
#endif

      if (normalize_separately)
	gain = target / sqrt(levels[i]);

      if (verbose >= VERBOSE_PROGRESS && normalize_separately)
	fprintf(stderr, _("Applying adjustment of %0.3f to %s...\n"),
		gain, fnames[i]);

      progress_info.file_start = time(NULL);
      progress_info.on_file = i;
      if (apply_gain(fd, fd2, fnames[i], gain) == -1) {
	fprintf(stderr, _("%s: error applying adjustment to %s: %s\n"),
		progname, fnames[i], strerror(errno));
      }

      progress_info.finished_size += progress_info.file_sizes[i];

      close(fd);
      close(fd2);

#if 1
      /*
       * Move the temporary file back to the original file,
       * copying if they are on different filesystems.
       */
      if (rename(tmpfile, fnames[i]) == -1) {
	if (errno == EXDEV) {
	  /* temp file is on a different filesystem, so we have to copy it */
	  FILE *in, *out;
	  in = fopen(tmpfile, "rb");
	  out = fopen(fnames[i], "wb");
	  if (in == NULL || out == NULL) {
	    fprintf(stderr, _("%s: error moving %s to %s: %s\n"), progname,
		    tmpfile, fnames[i], strerror(errno));
	    exit(1);
	  }
	  while ((c = getc(in)) != EOF)
	    putc(c, out);
	  if (fclose(out) == EOF || fclose(in) == EOF) {
	    fprintf(stderr, _("%s: error moving %s to %s: %s\n"), progname,
		    tmpfile, fnames[i], strerror(errno));
	    exit(1);
	  }
	  if (unlink(tmpfile) == -1)
	    fprintf(stderr, _("%s: error removing %s: %s\n"), progname,
		    tmpfile, strerror(errno));
	} else {
	  fprintf(stderr, _("%s: error moving %s to %s: %s\n"), progname,
		  tmpfile, fnames[i], strerror(errno));
	}
      }
      free(tmpfile);
#endif

      if (verbose >= VERBOSE_PROGRESS && normalize_separately)
	fprintf(stderr, "\n");
    }

    /* we're done with the second progress meter, so go to next line */
    if (verbose >= VERBOSE_PROGRESS && !normalize_separately)
      fputc('\n', stderr);

  } else if (!normalize_separately) {

    /* if we're not applying the gain, just print it out, and we're done */
    printf(_("%f\tvolume adjustment\n"), gain);

  }

  free(levels);
  free(progress_info.file_sizes);
  free(fnames);

  return 0;
}

/*
 * For batch mode, we take the levels for all the input files, throw
 * out any that appear to be statistical aberrations, and average the
 * rest together to get one level and one gain for the whole batch.
 */
double
analyze_levels(double *levels, int nlevels, double threshold)
{
  int i, files_to_avg;
  double sum, level_difference, std_dev, variance;
  double level, mean_level;
  char *badlevels;

  /* badlevels is a boolean array marking the level values to be thrown out */
  badlevels = (char *)xmalloc(nlevels * sizeof(char));
  memset(badlevels, 0, nlevels * sizeof(char));

  /* get mean level */
  sum = 0;
  for (i = 0; i < nlevels; i++)
    sum += levels[i];
  mean_level = sum / nlevels;

  /* if no threshold is specified, use 2 * standard dev */
  if (threshold < 0.0) {

    /*
     * We want the standard dev of the levels, but we need it in decibels.
     * Therefore, if u is the mean, the variance is
     *                  (1/N)summation((10*log10(x/u))^2)
     *       instead of (1/N)summation((x-u)^2),
     * which it would be if we needed straight variance "by the numbers".
     */

    /* get variance */
    sum = 0;
    for (i = 0; i < nlevels; i++) {
      double tmp = 10 * log10(levels[i] / mean_level);
      sum += tmp * tmp;
    }
    variance = sum / nlevels;

    /* get standard deviation */
    if (variance < EPSILON)
      std_dev = 0.0;
    else
      std_dev = sqrt(variance);
    if (verbose >= VERBOSE_INFO)
      printf(_("Standard deviation is %0.2f dB\n"), std_dev);

    threshold = 2 * std_dev;
  }

  /*
   * Throw out level values that seem to be aberrations
   * (so that one "quiet song" doesn't throw off the average)
   * We define an aberration as a level that is > 2*stddev dB from the mean.
   */
  if (threshold > EPSILON && nlevels > 1) {
    for (i = 0; i < nlevels; i++) {

      /* Find how different from average the i'th file's level is.
       * The "level" here is actually the signal's maximum power,
       * from which we can compute the difference in decibels. */
      if (levels[i] < mean_level)
	level_difference = 10 * log10(mean_level / levels[i]);
      else
	level_difference = 10 * log10(levels[i] / mean_level);

      /* mark as bad any level that is > threshold different than the mean */
      if (level_difference > threshold) {
	if (verbose >= VERBOSE_INFO)
	  printf(_("Throwing out level of %0.3f (different by %0.2fdB)\n"),
		 sqrt(levels[i]), level_difference);
	badlevels[i] = TRUE;
      }
    }
  }

  /* throw out the levels marked as bad */
  files_to_avg = 0;
  sum = 0;
  for (i = 0; i < nlevels; i++)
    if (!badlevels[i]) {
      sum += levels[i];
      files_to_avg++;
    }

  if (files_to_avg == 0) {
    fprintf(stderr, _("%s: all files ignored, try using -t 100\n"), progname);
    exit(1);
  }

  free(badlevels);

  level = sum / files_to_avg;

  return level;
}

#define LINE_LENGTH 79
void
progress_callback(char *prefix, float fraction_completed)
{
  char buf[LINE_LENGTH + 32]; /* need +1, but +32 in case of huge ETA's */
  time_t now, time_spent;
  unsigned int file_eta_hr, file_eta_min, file_eta_sec;
  off_t kb_done;
  float batch_fraction;
  unsigned int batch_eta_hr, batch_eta_min, batch_eta_sec;

  now = time(NULL);
  if (fraction_completed > 1.0)
    fraction_completed = 1.0;

  /* figure out the ETA for this file */
  file_eta_hr = file_eta_sec = file_eta_min = 0;
  time_spent = now - progress_info.file_start;
  file_eta_sec = floor((float)time_spent / fraction_completed
		       - (float)time_spent + 0.5);
  while (file_eta_sec >= 60) {
    file_eta_min++;
    file_eta_sec -= 60;
  }
  while (file_eta_min >= 60) {
    file_eta_hr++;
    file_eta_min -= 60;
  }
  if (file_eta_hr > 99)
    file_eta_hr = 99;

  /* figure out the ETA for the whole batch */
  kb_done = progress_info.finished_size
    + fraction_completed * progress_info.file_sizes[progress_info.on_file];
  batch_fraction = (float)kb_done / (float)progress_info.batch_size;
  batch_eta_hr = batch_eta_min = batch_eta_sec = 0;
  time_spent = now - progress_info.batch_start;
  batch_eta_sec = floor((float)time_spent / batch_fraction
			- (float)time_spent + 0.5);
  while (batch_eta_sec >= 60) {
    batch_eta_min++;
    batch_eta_sec -= 60;
  }
  while (batch_eta_min >= 60) {
    batch_eta_hr++;
    batch_eta_min -= 60;
  }
  if (batch_eta_hr > 99)
    batch_eta_hr = 99;

  sprintf(buf, _(" %s %3.0f%% done, ETA %02d:%02d:%02d (batch %3.0f%% done, ETA %02d:%02d:%02d)"),
	  prefix, fraction_completed * 100,
	  file_eta_hr, file_eta_min, file_eta_sec,
	  batch_fraction * 100,
	  batch_eta_hr, batch_eta_min, batch_eta_sec);

  fprintf(stderr, "%s\r", buf);
}


static __inline__ long
get_sample(unsigned char *pdata, int bytes_per_sample)
{
  long sample;

  switch(bytes_per_sample) {
  case 1:
    sample = *pdata - 128;
    break;
  case 2:
#ifdef WORDS_BIGENDIAN
    sample = *((int8_t *)pdata + 1) << 8;
    sample |= *((int8_t *)pdata) & 0xFF;
#else
    sample = *((int16_t *)pdata);
#endif
    break;
  case 3:
    sample = *((int8_t *)pdata + 2) << 16;
    sample |= (*((int8_t *)pdata + 1) << 8) & 0xFF00;
    sample |= *((int8_t *)pdata) & 0xFF;
    break;
  case 4:
    sample = *((int32_t *)pdata);
#ifdef WORDS_BIGENDIAN
    sample = bswap_32(sample);
#endif
    break;
  default:
    /* shouldn't happen */
    fprintf(stderr,
	    _("%s: I don't know what to do with %d bytes per sample\n"),
	    progname, bytes_per_sample);
    sample = 0;
  }

  return sample;
}


static __inline__ void
put_sample(long sample, unsigned char *pdata, int bytes_per_sample)
{
  switch(bytes_per_sample) {
  case 1:
    *pdata = sample + 128;
    break;
  case 2:
#ifdef WORDS_BIGENDIAN
    sample = bswap_16(sample);
#endif
    *((int16_t *)pdata) = (int16_t)sample;
    break;
  case 3:
    *pdata = (unsigned char)sample;
    *(pdata + 1) = (unsigned char)(sample >> 8);
    *(pdata + 2) = (unsigned char)(sample >> 16);
    break;
  case 4:
#ifdef WORDS_BIGENDIAN
    sample = bswap_32(sample);
#endif
    *((int32_t *)pdata) = (int32_t)sample;
    break;
  default:
    /* shouldn't happen */
    fprintf(stderr,
	    _("%s: I don't know what to do with %d bytes per sample\n"),
	    progname, bytes_per_sample);
  }
}


static riff_chunk_t *
get_wav_data(riff_t *riff, struct wavfmt *fmt)
{
  riff_chunk_t *chnk;

  chnk = riff_chunk_read(riff);
  if (chnk == NULL) {
    fprintf(stderr, _("%s: error reading riff chunk\n"), progname);
    goto error2;
  }
  if (!riff_fourcc_equals(chnk->id, "RIFF")) {
    fprintf(stderr, _("%s: not a RIFF WAV file\n"), progname);
    goto error3;
  }
  riff_list_descend(riff, chnk);
  riff_chunk_unref(chnk);

  /* read format header */
  chnk = riff_chunk_read(riff);
  if (chnk == NULL) {
    fprintf(stderr, _("%s: error reading riff chunk\n"), progname);
    goto error2;
  }
  if (!riff_fourcc_equals(chnk->id, "fmt ")) {
    fprintf(stderr, _("%s: no format chunk found\n"), progname);
    goto error3;
  }
  fread(fmt, sizeof(struct wavfmt), 1, riff_chunk_get_stream(chnk));
  riff_chunk_unref(chnk);
#ifdef WORDS_BIGENDIAN
  fmt->format_tag        = bswap_16(fmt->format_tag);
  fmt->channels          = bswap_16(fmt->channels);
  fmt->samples_per_sec   = bswap_32(fmt->samples_per_sec);
  fmt->avg_bytes_per_sec = bswap_32(fmt->avg_bytes_per_sec);
  fmt->block_align       = bswap_16(fmt->block_align);
  fmt->bits_per_sample   = bswap_16(fmt->bits_per_sample);
#endif

  /*
   * Make sure we can handle this type of wav
   */
  if (fmt->format_tag != 1) {
    fprintf(stderr, _("%s: this is a non-PCM WAV file\n"), progname);
    errno = EINVAL;
    goto error3;
  }
  if (fmt->bits_per_sample > 32) {
    fprintf(stderr, _("%s: more than 32 bits per sample not implemented\n"),
	    progname);
    errno = EINVAL;
    goto error3;
  }

  /* read until data chunk */
  chnk = NULL;
  do {

    if (chnk) {
      riff_chunk_unref(chnk);
      chnk = NULL;
    }

    errno = 0;
    chnk = riff_chunk_read(riff);
    if (chnk == NULL) {
      fprintf(stderr, _("%s: no data chunk found\n"), progname);
      if (errno == 0)
	errno = EINVAL;
      goto error2;
    }

  } while (!riff_fourcc_equals(chnk->id, "data"));

  return chnk;

  /* error handling stuff */
 error3:
  riff_chunk_unref(chnk);
 error2:
  return NULL;
}

typedef struct {
  double *buf;
  int buflen;  /* elements allocated to buffer */
  int start;   /* index of first element in buffer */
  int n;       /* num of elements in buffer */
} datasmooth_t;

/*
 * Takes a full smoothing window, and returns the value of the center
 * element, smoothed.  Currently, just does a mean filter, but we could
 * do a median or gaussian filter here instead.
 */
static __inline__ double
get_smoothed_data(datasmooth_t *s)
{
  int i;
  /*int center = (s->n + 1) / 2;*/
  double smoothed;

  smoothed = 0;
  for (i = 0; i < s->n; i++)
    smoothed += s->buf[i];
  smoothed = smoothed / s->n;

  return smoothed;
}

/*
 * Get the maximum power level of the wav file
 */
double
signal_max_power(int fd, char *filename)
{
  riff_t *riff;
  riff_chunk_t *chnk;
  struct wavfmt fmt;
  unsigned int nsamples;

  int bytes_per_sample;
  int last_window;
  unsigned int windowsz;
  unsigned int win_start, old_start, win_end, old_end;

  int i, c;
  long sample, samplemax;
  double *sums;
  double pow, maxpow;
  datasmooth_t *powsmooth;

  float progress, last_progress = 0.0;
  char prefix_buf[18];

  FILE *in;
  unsigned char *data_buf = NULL;
  int filled_sz;

  riff = riff_new(fd, RIFF_RDONLY);
  if (riff == NULL) {
    fprintf(stderr, _("%s: error making riff object\n"), progname);
    goto error1;
  }

  chnk = get_wav_data(riff, &fmt);
  if (chnk == NULL) {
    fprintf(stderr, _("%s: error getting wav data\n"), progname);
    goto error2;
  }
  windowsz = (unsigned int)(fmt.samples_per_sec / 100);

  bytes_per_sample = (fmt.bits_per_sample - 1) / 8 + 1;
  samplemax = 1 << (bytes_per_sample * 8 - 1);
  nsamples = chnk->size / bytes_per_sample / fmt.channels;

  sums = (double *)xmalloc(fmt.channels * sizeof(double));
  for (c = 0; c < fmt.channels; c++)
    sums[c] = 0;

  data_buf = (unsigned char *)xmalloc(windowsz
				      * fmt.channels * bytes_per_sample);

  /* set up smoothing window buffer */
  powsmooth = (datasmooth_t *)xmalloc(fmt.channels * sizeof(datasmooth_t));
  for (c = 0; c < fmt.channels; c++) {
    powsmooth[c].buflen = 100; /* use a 100-element (1 second) window */
    powsmooth[c].buf = (double *)xmalloc(powsmooth[c].buflen * sizeof(double));
    powsmooth[c].start = powsmooth[c].n = 0;
  }

  /* initialize progress meter */
  if (verbose >= VERBOSE_PROGRESS) {
    if (strrchr(filename, '/') != NULL) {
      filename = strrchr(filename, '/');
      filename++;
    }
    strncpy(prefix_buf, filename, 17);
    prefix_buf[17] = 0;
    progress_callback(prefix_buf, 0.0);
    last_progress = 0.0;
  }

  in = fdopen(fd, "r");
  if (in == NULL) {
    fprintf(stderr, _("%s: unable to fdopen: %s\n"),
	    progname, strerror(errno));
    goto error7;
  }
  fseek(in, chnk->offset + 8, SEEK_SET);


  /*
   * win_start, win_end, old_start, windowsz, interval, and i are in
   * units of samples.  c is in units of channels.
   *
   * The actual window extends from win_start to win_end - 1, inclusive.
   */
  old_start = win_start = 0;
  win_end = 0;
  last_window = FALSE;
  maxpow = 0.0;

  do {

    /* set up the window end */
    old_end = win_end;
    win_end = win_start + windowsz;
    if (win_end >= nsamples) {
      win_end = nsamples;
      last_window = TRUE;
    }

    /* read a windowsz sized chunk */
    filled_sz = fread(data_buf, bytes_per_sample,
		      windowsz * fmt.channels, in);
    for (c = 0; c < fmt.channels; c++) {
      sums[c] = 0;
      /*for (i = 0; i * bytes_per_sample * fmt.channels < filled_sz; i++) {*/
      for (i = 0; i < (win_end - win_start); i++) {
	sample = get_sample(data_buf + (i * fmt.channels * bytes_per_sample)
			    + (c * bytes_per_sample), bytes_per_sample);
	sums[c] += sample * sample;
      }
    }

    /* compute power for each channel */
    for (c = 0; c < fmt.channels; c++) {
      int end;
      pow = sums[c] / (double)(win_end - win_start);
#if 0
      if (c == 0)
	printf("before: %d %f\n", win_start, pow);
#endif

      end = (powsmooth[c].start + powsmooth[c].n) % powsmooth[c].buflen;
      powsmooth[c].buf[end] = pow;
      if (powsmooth[c].n == powsmooth[c].buflen) {
	powsmooth[c].start = (powsmooth[c].start + 1) % powsmooth[c].buflen;
	pow = get_smoothed_data(&powsmooth[c]);
#if 0
	if (c == 0)
	  printf("after: %d %f\n", win_start, pow);
#endif
	if (pow > maxpow)
	  maxpow = pow;
      } else {
	powsmooth[c].n++;
      }
    }

    /* update progress meter */
    if (verbose >= VERBOSE_PROGRESS) {
      if (nsamples - windowsz == 0)
	progress = 0;
      else
	progress = (win_end - windowsz) / (float)(nsamples - windowsz);
      /*progress = win_end / ((nsamples - windowsz) / (float)windowsz);*/
      if (progress >= last_progress + 0.01) {
	/*printf(_("progress: %f       \n"), progress);*/
	progress_callback(prefix_buf, progress);
	last_progress += 0.01;
      }
    }

    /* slide the window ahead */
    old_start = win_start;
    win_start += windowsz;

  } while (!last_window);

  if (maxpow < EPSILON) {
    /*
     * Either this whole file has zero power, or was too short to ever
     * fill the smoothing buffer.  In the latter case, we need to just
     * get maxpow from whatever data we did collect.
     */
    for (c = 0; c < fmt.channels; c++) {
      pow = get_smoothed_data(&powsmooth[c]);
      if (pow > maxpow)
	maxpow = pow;
    }
  }

  for (c = 0; c < fmt.channels; c++)
    free(powsmooth[c].buf);
  free(powsmooth);
  free(data_buf);
  free(sums);
  riff_chunk_unref(chnk);
  riff_unref(riff);

  /* scale the pow value to be in the range 0.0 -- 1.0 */
  maxpow = maxpow / (samplemax * samplemax);

  return maxpow;

  /* error handling stuff */
 error7:
  for (c = 0; c < fmt.channels; c++)
    free(powsmooth[c].buf);
  /*error6:*/
  free(powsmooth);
  /*error5:*/
  free(data_buf);
  /*error4:*/
  free(sums);
  /*error3:*/
  riff_chunk_unref(chnk);
 error2:
  riff_unref(riff);
 error1:
  return -1.0;
}

int
apply_gain(int read_fd, int write_fd, char *filename, double gain)
{
  riff_t *riff;
  riff_chunk_t *chnk;
  struct wavfmt fmt;
  unsigned int nsamples, samples_done, nclippings;
  int bytes_per_sample, i;
  long sample, samplemax, samplemin;
  float clip_loss;
  FILE *rd_stream, *wr_stream;

  float last_progress = 0, progress;
  char prefix_buf[18];

  unsigned char *data_buf = NULL;
  int samples_in_buf, samples_recvd;

  riff = riff_new(read_fd, RIFF_RDONLY);
  if (riff == NULL) {
    fprintf(stderr, _("%s: error making riff object\n"), progname);
    goto error1;
  }

  chnk = get_wav_data(riff, &fmt);
  if (chnk == NULL) {
    fprintf(stderr, _("%s: error getting wav data\n"), progname);
    goto error2;
  }

  bytes_per_sample = (fmt.bits_per_sample - 1) / 8 + 1;
  samplemax = (1 << (bytes_per_sample * 8 - 1)) - 1;
  samplemin = -samplemax - 1;

  /* ignore different channels, apply gain to all samples */
  nsamples = chnk->size / bytes_per_sample;

  /* set up sample buffer to hold 1/100 of a second worth of samples */
  /* (make sure it can hold at least the wav header, though) */
  samples_in_buf = (fmt.samples_per_sec / 100) * fmt.channels;
  if (chnk->offset + 8 > samples_in_buf * bytes_per_sample)
    data_buf = (unsigned char *)xmalloc(chnk->offset + 8);
  else
    data_buf = (unsigned char *)xmalloc(samples_in_buf * bytes_per_sample);

  /* open streams for reading and writing */
  rd_stream = fdopen(read_fd, "rb");
  wr_stream = fdopen(write_fd, "wb");
  if (rd_stream == NULL || wr_stream == NULL) {
    fprintf(stderr, _("%s: failed fdopen: %s\n"), progname, strerror(errno));
    goto error4;
  }
  /* copy the wav header */
  rewind(rd_stream);
  rewind(wr_stream);
  if (fread(data_buf, chnk->offset + 8, 1, rd_stream) < 1) {
    fprintf(stderr, _("%s: read failed: %s\n"), progname, strerror(errno));
    goto error4;
  }
  if (fwrite(data_buf, chnk->offset + 8, 1, wr_stream) < 1) {
    fprintf(stderr, _("%s: write failed: %s\n"), progname, strerror(errno));
    goto error4;
  }

  /* initialize progress meter */
  if (verbose >= VERBOSE_PROGRESS) {
    if (strrchr(filename, '/') != NULL) {
      filename = strrchr(filename, '/');
      filename++;
    }
    /*sprintf(prefix_buf, "%-16s ", filename);*/
    strncpy(prefix_buf, filename, 17);
    prefix_buf[17] = 0;
    progress_callback(prefix_buf, 0.0);
    last_progress = 0.0;
  }

  /* read, apply gain, and write, one chunk at time */
  nclippings = samples_done = 0;
  while ((samples_recvd = fread(data_buf, bytes_per_sample,
				samples_in_buf, rd_stream)) > 0) {

    for (i = 0; i < samples_recvd; i++) {
      sample = get_sample(data_buf + (i * bytes_per_sample), bytes_per_sample);

      /* apply the gain to the sample */
      sample *= gain;

      /* perform clipping */
      if (sample > samplemax) {
	sample = samplemax;
	nclippings++;
      } else if (sample < samplemin) {
	sample = samplemin;
	nclippings++;
      }

      put_sample(sample, data_buf + (i * bytes_per_sample), bytes_per_sample);
    }

    if (fwrite(data_buf, bytes_per_sample,
	       samples_recvd, wr_stream) == 0) {
      fprintf(stderr, _("%s: failed fwrite: %s\n"), progname, strerror(errno));
    }

    samples_done += samples_recvd;

    /* update progress meter */
    if (verbose >= VERBOSE_PROGRESS) {
      progress = samples_done / (float)nsamples;
      if (progress >= last_progress + 0.01) {
	progress_callback(prefix_buf, progress);
	last_progress += 0.01;
      }
    }
  }


#ifdef OLD
  nclippings = 0;
  for (i = 0; i < nsamples; i++) {

    sample = get_sample(chnk->fp, bytes_per_sample);

    /* apply the gain to the sample */
    sample *= gain;

    /* perform clipping */
    if (sample > samplemax) {
      sample = samplemax;
      nclippings++;
    } else if (sample < samplemin) {
      sample = samplemin;
      nclippings++;
    }

    put_sample(sample, wr_stream, bytes_per_sample);

    /* update progress meter */
    if (i % 100 == 0 && verbose >= VERBOSE_PROGRESS) {
      progress = (i + 1) / (float)nsamples;
      if (progress >= last_progress + 0.01) {
	progress_callback(prefix_buf, progress);
	last_progress += 0.01;
      }
    }
  }
#endif

  /* make sure progress meter is finished */
  if (verbose >= VERBOSE_PROGRESS)
    progress_callback(prefix_buf, 1.0);

  if (fflush(rd_stream) == -1) {
    fprintf(stderr, _("%s: failed fflush: %s\n"), progname, strerror(errno));
  }
  if (fflush(wr_stream) == -1) {
    fprintf(stderr, _("%s: failed fflush: %s\n"), progname, strerror(errno));
  }

  clip_loss = (float)nclippings / (float)nsamples;

  if (verbose >= VERBOSE_INFO) {
    if (nclippings) {
      fprintf(stderr, "\n");
      fprintf(stderr, _("%s: %d clippings performed, %.4f%% loss\n"),
	      progname, nclippings, clip_loss * 100);
    }
  } else if (verbose >= VERBOSE_PROGRESS) {
    if (clip_loss > CLIPPING_WARN_THRESH)
      fprintf(stderr, _("%s: Warning: lost %.2f%% of data due to clipping\n"),
	      progname, clip_loss * 100);
  }

  free(data_buf);
  riff_chunk_unref(chnk);
  riff_unref(riff);
  return 0;


  /* error handling stuff */
 error4:
  free(data_buf);
  /*error3:*/
  riff_chunk_unref(chnk);
 error2:
  riff_unref(riff);
 error1:
  return -1;
}

void *
xmalloc(size_t size)
{
  void *ptr = malloc(size);
  if (ptr == NULL) {
    fprintf(stderr, _("%s: unable to malloc\n"), progname);
    exit(1);
  }
  return ptr;
}
