/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#ifndef XML_H_
#define XML_H_


/**
 * Xml data, from file or string.
 */
struct _AlXml
{

	char * file;   /** file name       */
	char * string; /** xml string      */

	char * parse;  /** string to parse */

};

typedef struct _AlXml AlXml;


/**
 * Create a new empty xml.
 *
 * @return a new xml
 */
AlXml * al_xml_new(void);


/**
 * Create a new xml from a file.
 *
 * @param s file name
 *
 * @return a new xml
 */
AlXml * al_xml_new_from_file(const char * s);


/**
 * Create a new xml from string.
 *
 * @param s string
 *
 * @return a new xml
 */
AlXml * al_xml_new_from_string(const char * s);


/**
 * Delete xml.
 *
 * @param xml xml
 */
void al_xml_delete(AlXml * xml);


/**
 * Verify if a xml is valid.
 *
 * @param xml xml to verify
 *
 * @return On success 0
 */
int al_xml_verify(AlXml * xml);


/**
 * Verify if a xml is valid.
 *
 * @param xml xml to verify
 *
 * @return On success 0
 */
int al_xml_verify_from_string(const char * string);


/**
 * An xml attribute
 */
struct _AlXmlAttribute
{
	struct _AlXmlAttribute * prev;
	struct _AlXmlAttribute * next;

	char                       * name;  /** name of attribute */
	char                       * value; /** value             */

};

typedef struct _AlXmlAttribute AlXmlAttribute;


/**
 */
#define al_xml_attribute_set_name(attr, v)          (attr)->name = al_strdup(v)


/**
 */
#define al_xml_attribute_get_name(attr)             (attr)->name


/**
 */
#define al_xml_attribute_set_value(attr, v)         (attr)->value = al_strdup(v)


/**
 */
#define al_xml_attribute_get_value(attr)            (attr)->value


/**
 * An xml node.
 */
struct _AlXmlNode
{
	struct _AlXmlNode * prev;
	struct _AlXmlNode * next;

	char              * name;       /** name of node           */
	AlXmlAttribute    * attributes; /** attributes of the node */

	char              * cdata;      /** Content of CDATA if present */
};

typedef struct _AlXmlNode AlXmlNode;


/**
 * Set the name of a node.
 *
 * @param node node
 * @param v name
 */
#define al_xml_node_set_name(node, v)             (node)->name = al_strdup(v)


/**
 * Return the name of a node.
 *
 * @param node node
 *
 * @return the node name or NULL if node is not valid
 */
#define al_xml_node_get_name(node)                ((node) ? (node)->name : NULL)


/**
 * Set the CDATA of a node.
 *
 * @param node node
 * @param v CDATA
 */
#define al_xml_node_set_cdata(node, v)             (node)->cdata = al_strdup(v)


/**
 * Return the CDATA of a node.
 *
 * @param node node
 *
 * @return the node CDATA or NULL if node is not valid
 */
#define al_xml_node_get_cdata(node)                ((node) ? (node)->cdata : NULL)


/**
 * A list of xml nodes
 */
struct _AlXmlList
{
	AlXmlNode      * parent; /** parent node     */

	AlXmlNode      * nodes; /** nodes in list    */

	AlXmlNode      * node;  /** Node of the list */

};

typedef struct _AlXmlList AlXmlList;


/**
 * Create a new xml attribute.
 *
 * @return a new xml attribute
 */
AlXmlAttribute * al_xml_attribute_new(void);


/**
 * Delete a xml attribute.
 *
 * @param attr attribute
 */
void al_xml_attribute_delete(AlXmlAttribute * attr);


/**
 * Create a new xml node.
 *
 * @return a new xml node
 */
AlXmlNode * al_xml_node_new(void);


/**
 * Delete a xml node with attribute list.
 *
 * @param node node
 */
void al_xml_node_delete(AlXmlNode * node);


/**
 * Add attribute at the node attribute list.
 *
 * @param node node
 * @param attr attribute
 */
void al_xml_node_attribute_add(AlXmlNode * node, AlXmlAttribute * attr);


/**
 * Create a new xml list.
 *
 * @return a new xml list
 */
AlXmlList * al_xml_list_new(void);


/**
 * Delete a xml list and all the nodes.
 *
 * @param list list
 */
void al_xml_list_delete(AlXmlList * list);


/**
 * Add node at the node list.
 *
 * @param list list
 * @param node node
 */
void al_xml_list_node_add(AlXmlList * list, AlXmlNode * node);


/**
 * Read a list of node.
 * Ex:
 * <al>
 *    <agents>
 *       <agent .../>
 *    </agents>
 * </al>
 *
 * Get list "agents" return a list with all node in agents (agent).
 *
 * @param xml xml
 * @param name name of node to find
 *
 * @return the list or NULL if not found
 */
AlXmlList * al_xml_get_list(AlXml * xml, const char * name);


/**
 * Search for a node with specific attribute in a list.
 * Ex:
 * <al>
 *    <agents>
 *       <agent type="1".../>
 *    </agents>
 * </al>
 *
 * If search with list_name=agents, node_name= agent, attr_name=type and attr_value = 1 return
 * the node <agent type="1".../>.
 *
 * @param xml xml
 * @param list_name name of list node
 * @param node_name name of node
 * @param attr_name name of attribute
 * @param attr_value search value of attribute
 *
 * @return the node or NULL if not found
 */
AlXmlNode * al_xml_get_node_in_list(AlXml * xml, const char * list_name, const char * node_name, const char * attr_name, const char * attr_value);


/**
 * Return a node from name.
 *
 * @param xml xml
 * @param name name of node to search
 *
 * @return the node or NULL if not found
 */
AlXmlNode * al_xml_get_node(AlXml * xml, const char * name);


/**
 * Create a new attribute and add at given node.
 *
 * @param node node
 * @param name name of new attribute
 * @param value value of attribute
 */
void al_xml_node_attribute_new(AlXmlNode * node, const char * name, const char * value);


/**
 * Return the value of an attribute of a node.
 *
 * @param node node
 * @param attr_name name of attribute
 *
 * @return the value of attribute or NULL if not found
 */
char * al_xml_get_attribute_value(const AlXmlNode * node, const char * attr_name);


/**
 * Dump the list to XML.
 *
 * @param list list to dump
 * @param dtd_file file with dtd. NULL if not used
 * @param doctype doc type for dtd
 *
 * @return the XML or NULL if list is not valid
 */
AlString * al_xml_list_to_string(AlXmlList * list, const char * dtd_file, const char * doctype);


#endif /* XML_H_ */
