/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include "al.h"

#include <spawn.h>
#include <sys/types.h>
#include <sys/stat.h>



/**
 */
int al_spawn(const char * bin, const char * args, const char * user, const char * std_in, AlString * std_out, int * status)
{
	int                          ret = 0;
	int                          argc;
	char                      ** argv = NULL;
	char                         buf;
	int                          sts;
	int                          i;
	int                          len;
	int                          err;
	pid_t                        pid;
	posix_spawn_file_actions_t   actions;
	posix_spawnattr_t            attrs;
	int                          ifd[2];
	int                          ofd[2];



	al_return_val_if_fail(bin, 99);


	if ((err = posix_spawn_file_actions_init (&actions)) != 0)
	{
		fprintf(stderr, "Error %d executing posix_spawn_file_actions_init\n", err);
		ret = 1;
		goto bomb_out;
	}


	/* Create pipe for redirect stdin e stdout */
	if (pipe (ifd) == -1)
	{
		fprintf(stderr, "Error %d executing pipe\n", errno);
		ret = 2;
		goto bomb_out;
	}

	if (pipe (ofd) == -1)
	{
		fprintf(stderr, "Error %d executing pipe\n", errno);
		ret = 3;
		goto bomb_out;
	}

	posix_spawn_file_actions_adddup2 (&actions, ofd[0], STDIN_FILENO);
	posix_spawn_file_actions_adddup2 (&actions,ifd[1], STDOUT_FILENO);
	posix_spawn_file_actions_addclose (&actions, ofd[0]);
	posix_spawn_file_actions_addclose (&actions, ifd[1]);
	posix_spawn_file_actions_addclose (&actions, ofd[1]);
	posix_spawn_file_actions_addclose (&actions, ifd[0]);


	if (std_in)
	{
		write(ofd[1], std_in, strlen(std_in));
		write(ofd[1], "\n", 1);
	}

	
	if (args && user == NULL)
	{
		argv = al_split(args, ' ');
	}
	else if (args && user)
	{
		char * new_args;

		new_args = al_new_n(char, strlen(bin) + strlen(args) + strlen(user) + 50);
		sprintf(new_args, "/bin/su -c %s %s - %s", bin, args, user);
		argv = al_split(new_args, ' ');

		al_delete(new_args);
	}
	else if (user)
	{
		char * new_args;

		new_args = al_new_n(char, strlen(bin) + strlen(user) + 50);
		sprintf(new_args, "/bin/su -c %s - %s", bin, user);
		argv = al_split(new_args, ' ');

		al_delete(new_args);
	}


	if (user)
	{
		if ((err = posix_spawn (&pid, "/bin/su", &actions, NULL, argv, NULL)) != 0)
		{
			fprintf(stderr, "Error %d during posix_spawn with user %s\n", err, user);
			al_delete(argv);
			ret = 4;
			goto bomb_out;
		}
		al_delete(argv);
	}
	else
	{
		if ((err = posix_spawn (&pid, bin, &actions, NULL, argv, NULL)) != 0)
		{
			fprintf(stderr, "Error %d during posix_spawn\n", err);
			ret = 5;
			goto bomb_out;
		}
	}

	close(ifd[1]);
	close(ofd[0]);
	close(ofd[1]);

	if (std_out == NULL)
		close(ifd[0]);
	
	/* .. and wait to end */
	waitpid(pid, &sts, 0);

	if (status) (*status) = sts;

	if (std_out)
	{
		while (read(ifd[0], &buf, 1) > 0)
			al_string_append_char(std_out, buf);

		close(ifd[0]);
	}

	
bomb_out:
	al_array_delete(argv);

	return ret;
}
