/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include "agentd.h"



/**
 */
AgentdRequest * agentd_request_new(void)
{
	AgentdRequest * request = al_new(AgentdRequest);

	request->prev = request->next = NULL;

	request->id = NULL;
	request->agent_name = NULL;
	request->agent_data = NULL;

	return request;
}


/**
 */
void agentd_request_delete(AgentdRequest * request)
{

	al_return_if_fail (request);

	al_delete(request->id);
	al_delete(request->agent_name);
	al_delete(request->agent_data);
	al_delete(request);

}


/**
 */
AgentdRequestPacket * agentd_request_packet_new(void)
{
	AgentdRequestPacket * request_packet = al_new(AgentdRequestPacket);

	request_packet->requests = NULL;

	return request_packet;
}


/**
 */
void agentd_request_packet_delete(AgentdRequestPacket * request_packet)
{

	al_return_if_fail (request_packet);

	if (request_packet->requests)
	{
		AgentdRequest * request = request_packet->requests;
		AgentdRequest * next;

		while (request)
		{
			next = request->next;
			agentd_request_delete(request);
			request = next;
		}
	}
	
	al_delete(request_packet);

}


/**
 */
void agentd_request_packet_request_add(AgentdRequestPacket * request_packet, AgentdRequest * request)
{

	al_return_if_fail (request_packet && request);

	if (request_packet->requests == NULL)
	{

		request_packet->requests = request;
		return;
	}


	request->next = request_packet->requests->next;
	request->prev = request_packet->requests;

	request_packet->requests->next = request;


}


/**
 */
AgentdRequestPacket * agentd_request_packet_from_xml_string(const AlString * xml)
{
	AlXml               * x = al_xml_new_from_string(al_string_get(xml));
	AgentdRequestPacket * request_packet;


	request_packet = agentd_request_packet_from_xml(x);
	al_xml_delete(x);

	return request_packet;
}


/**
 */
AgentdRequestPacket * agentd_request_packet_from_xml(AlXml * xml)
{
	AgentdRequestPacket * request_packet;
	AgentdRequest       * request;
	AlXmlList           * list;
	AlXmlNode           * node;


	al_return_val_if_fail(xml, NULL);

	al_log(("agentd_request_packet_from_xml:\n%s\n", xml->string));

	request_packet = agentd_request_packet_new();
	list = al_xml_get_list(xml, "requests");
	if (list)
	{
		for (node = list->nodes; node; node = node->next)
		{

			request = agentd_request_from_xml(node);

			if (request) agentd_request_packet_request_add(request_packet, request);

		}
	}
	else
	{
		agentd_request_packet_delete(request_packet);
		request_packet = NULL;
	}

	return request_packet;
}


/**
 */
AgentdRequest * agentd_request_from_xml(AlXmlNode * node)
{
	AgentdRequest * ret;


	al_return_val_if_fail (node, NULL);

	ret = agentd_request_new();

	agentd_request_set_id(ret, al_xml_get_attribute_value(node, "id"));
	agentd_request_set_agent_name(ret, al_xml_get_attribute_value(node, "agent"));
	agentd_request_set_agent_data(ret, al_xml_get_attribute_value(node, "data"));

	return ret;
}


/**
 */
AlXmlList * agentd_request_packet_to_xml(const AgentdRequestPacket * packet)
{
	AlXmlList      * list = al_xml_list_new();
	AgentdRequest  * request;


	al_return_val_if_fail (packet, NULL);


	list->parent = al_xml_node_new();
	al_xml_node_set_name(list->parent, "agentd");

	list->node = al_xml_node_new();
	al_xml_node_set_name(list->node, "requests");
	for (request = packet->requests;request;request = request->next)
	{

		al_xml_list_node_add(list, agentd_request_to_xml(request));

	}

	return list;
}


/**
 */
AlXmlNode * agentd_request_to_xml(const AgentdRequest * request)
{
	AlXmlNode * node;
	char        stype[10];


	al_return_val_if_fail (request, NULL);

	node = al_xml_node_new();
	al_xml_node_set_name(node, "request");

	al_xml_node_attribute_new(node, "id", agentd_request_get_id(request));
	al_xml_node_attribute_new(node, "data", agentd_request_get_agent_data(request));
	al_xml_node_attribute_new(node, "agent", agentd_request_get_agent_name(request));


	return node;
}


/**
 */
AlString * agentd_request_packet_to_xml_string(const AgentdRequestPacket * request_packet)
{
	AlXmlList * list;
	AlString  * s = NULL;


	al_return_val_if_fail(request_packet, NULL);

	list = agentd_request_packet_to_xml(request_packet);
	if (list) s = al_xml_list_to_string(list, "../dtd/agentd-request.dtd", "agentd-request");


	return s;
}
