/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include <agents.h>
#include <agent.h>
#include <agentd.h>


/**
 * Version
 */
#define AGENTS_VERSION_MAJOR               0
#define AGENTS_VERSION_MINOR               0
#define AGENTS_VERSION_MICRO               2
#define AGENTS_VERSION_NAME           agents

#define AGENTS_VERSION                AL_TO_VERSION(AGENTS_VERSION_NAME,AGENTS_VERSION_MAJOR,AGENTS_VERSION_MINOR,AGENTS_VERSION_MICRO)


/**
 * Default config file.
 */
#define AGENTS_CONFIG_FILE           "../etc/agents.conf"


/**
 */
void syntax(void);


/**
 */
AlXml * agents_config(void);


/** Config object */
static AlXml * config_xml = NULL;


/**
 */
int main(int argc, char ** argv)
{
	AlConnectionManager     * connection_manager = NULL;
	AlConnection            * connection = NULL;
	char                    * connection_type = NULL;
	char                    * s_port;
	int                       port;
	AlXmlNode               * node;
	AlString                * s;
	AlString                * input = NULL;
	int                       i;
	const char              * server = NULL;
	const char              * agent = NULL;
	int                       empty = 0;
	AgentsRequestPacket     * request_packet = NULL;
	AgentsResponsePacket    * response_packet = NULL;



	/** Parse the command line */
	for (i=1;i<argc;i++)
	{

		if (!strcmp(argv[i],"-h") || !strcmp(argv[i],"--help"))
		{
			syntax();
		}
		else if (!strcmp(argv[i],"-v") || !strcmp(argv[i],"--version"))
		{
			al_exit(0, "%s\n", AGENTS_VERSION);
		}
		else if (!strcmp(argv[i],"-c") || !strcmp(argv[i],"--connection"))
		{
			if ((++i) < argc)
			{
				connection_type = argv[i];
				al_connection_supported_list(connection_type, AGENTS_VERSION, al_connection_supported);
			}
			else
			{
				syntax();
			}
		}
		else if (!strcmp(argv[i],"-s") || !strcmp(argv[i],"--server"))
		{
			if ((++i) < argc)
			{
				server = argv[i];
			}
			else
			{
				syntax();
			}
		}
		else if (!strcmp(argv[i],"-a") || !strcmp(argv[i],"--agent"))
		{
			if ((++i) < argc)
			{
				agent = argv[i];
			}
			else
			{
				syntax();
			}
		}
		else if (!strcmp(argv[i],"-e") || !strcmp(argv[i],"--empty"))
		{
			empty = 1;
		}
		else if (!strcmp(argv[i],"-q") || !strcmp(argv[i],"--query"))
		{
			empty = 1;
			if (agent == NULL) agent = "agent-info";
		}
		else
		{
			syntax();
		}

	}


	if ((server != NULL && agent == NULL) || (server == NULL && agent != NULL))
		syntax();

	if ((node = al_xml_get_node(agents_config(), "connection")) == NULL)
		al_exit(1, "%s\nInvalid config file.\n", AGENTS_VERSION);

	if (!(connection_manager = al_connection_manager_from_string(connection_type, al_connection_supported, node)))
		al_exit(1, "%s\nInvalid connection type.\n", AGENTS_VERSION);


	openlog(AGENTS_VERSION, LOG_PID, LOG_DAEMON);

	if (!empty)
	{
		input = al_read_file(stdin);
		if (al_xml_verify_from_string(al_string_get(input)))
			al_exit(1, "Invalid xml packet from stdin.\n");
	}


	if (server)
	{
		AgentRequestPacket  * agent_packet = agent_request_packet_new();
		AgentRequest        * agent_request = agent_request_new();

		AgentdRequestPacket * agentd_packet = agentd_request_packet_new();
		AgentdRequest       * agentd_request = agentd_request_new();

		AgentsRequestPacket * agents_packet = agents_request_packet_new();
		AgentsRequest       * agents_request = agents_request_new();


		/** Create agent request */
		agent_request_set_id(agent_request, "00000000");
		if (input) agent_request_set_agent_data(agent_request, al_string_get(input));
		agent_request_packet_request_add(agent_packet, agent_request);
		
		/** Create agentd request */
		agentd_request_set_id(agentd_request, "00000000");
		agentd_request_set_agent_name(agentd_request, agent);
		s = agent_request_packet_to_xml_string(agent_packet);
		agentd_request_set_agent_data(agentd_request, al_string_get(s));
		al_string_delete(s);
		agentd_request_packet_request_add(agentd_packet, agentd_request);

		/** Create agents request */
		agents_request_set_id(agent_request, "00000000");
		agents_request_set_server_name(agents_request, server);
		s = agentd_request_packet_to_xml_string(agentd_packet);
		agents_request_set_server_data(agents_request, al_string_get(s));
		al_string_delete(s);
		agents_request_packet_request_add(agents_packet, agents_request);

		request_packet = agents_packet;
	}
	else
	{
		request_packet = agents_request_packet_from_xml_string(input);
	}

	response_packet = agents_response_packet_from_request_packet(request_packet, connection_manager);
	if (response_packet)
	{
		s = NULL;

		if (server)
		{
			AgentResponsePacket  * agent_packet;
			AgentResponse        * agent_response;

			AgentdResponsePacket * agentd_packet;
			AgentdResponse       * agentd_response;

			AgentsResponsePacket * agents_packet;
			AgentsResponse       * agents_response;


			agents_packet = response_packet;
			agents_response = agents_packet->responses;
			if (agents_response_get_server_data(agents_response))
			{
				s = al_string_new_from_string(agents_response_get_server_data(agents_response));
				agentd_packet = agentd_response_packet_from_xml_string(s);
				al_string_delete(s);
				s = NULL;

				agentd_response = agentd_packet->responses;

				if (agentd_response_get_agent_data(agentd_response))
				{
					s = al_string_new_from_string(agentd_response_get_agent_data(agentd_response));
					agent_packet = agent_response_packet_from_xml_string(s);
					al_string_delete(s);
					s = NULL;

					agent_response = agent_packet->responses;

					s = al_string_new_from_string(agent_response_get_agent_data(agent_response));
				}
			}
			
		}
		else
		{
			s = agents_response_packet_to_xml_string(response_packet);
		}


		if (s)
		{
			fprintf(stdout,"%s", al_string_get(s));
			al_string_delete(s);
		}

	}

	if (connection_manager) al_connection_manager_delete(connection_manager);

	al_exit(0, NULL);

	return 0;
}


/**
 */
void syntax(void)
{

	printf("Usage: agents [OPTION ...]\n");
	printf(" Options:\n");
	printf("\n");
	printf("  -c, --connection       Override the connection type given in config file.\n");
	printf("                         Use -c list to show the supported connection type.\n");
	printf("  -s, --server server    Send the xml data read from stdin to server.\n");
	printf("                         The xml data is only the data for agent. All xml data needed\n");
	printf("                         for agentd is create automaticalli.\n");
	printf("  -a, --agent            To use with -s for specify the agent name\n");
	printf("  -e, --empty            Not send agent data. Use in combination with -s\n");
	printf("  -q, --query            Query the server given with -s if it have agentd\n"); 
	printf("  -h, --help             Give this help list.\n");
	printf("  -v, --version          Show version and exit.\n");


	al_exit(0, NULL);
}


/**
 */
AlXml * agents_config(void)
{
	if (config_xml == NULL)
		config_xml = al_xml_new_from_file(AGENTS_CONFIG_FILE);

	if (al_xml_verify(config_xml))
		al_exit(1, "Could not parse config file %s\n", AGENTS_CONFIG_FILE);
	
	return config_xml;
}
