/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include <agentd.h>
#include <signal.h>


/**
 * Version
 */
#define AGENTD_VERSION_MAJOR           0
#define AGENTD_VERSION_MINOR           0
#define AGENTD_VERSION_MICRO           2
#define AGENTD_VERSION_NAME       agentd

#define AGENTD_VERSION            AL_TO_VERSION(AGENTD_VERSION_NAME,AGENTD_VERSION_MAJOR,AGENTD_VERSION_MINOR,AGENTD_VERSION_MICRO)

/**
 * Default config file.
 */
#define AGENTD_CONFIG_FILE        "../etc/agentd.conf"


/**
 */
void syntax(void);


/**
 */
static void agentd_signal_handler(int signum);


/**
 */
AlXml * agentd_config(void);


/** Global connection manager */
static AlConnectionManager     * connection_manager = NULL;

/** Child pid */
static pid_t                     pid = 0;

/** Config object */
static AlXml * config_xml = NULL;



/**
 */
int main(int argc, char ** argv)
{
#ifdef USE_NOCLDWAIT
	struct sigaction          act;
	struct sigaction          oact;
#endif
	AlConnection            * connection = NULL;
	AgentdRequestPacket     * request_packet = NULL;
	AgentdResponsePacket    * response_packet = NULL;
	char                    * connection_type = NULL;
	char                    * s_port;
	int                       port;
	AlXmlNode               * node;
	AlString                * s;
	int                       i;
	int                       run_daemon = 0;



	/** Parse the command line */
	for (i=1;i<argc;i++)
	{

		if (!strcmp(argv[i],"-h") || !strcmp(argv[i],"--help"))
		{
			syntax();
		}
		else if (!strcmp(argv[i],"-v") || !strcmp(argv[i],"--version"))
		{
			al_exit(0, "%s\n", AGENTD_VERSION);
		}
		else if (!strcmp(argv[i],"-d") || !strcmp(argv[i],"--daemon"))
		{
			run_daemon = 1;
		}
		else if (!strcmp(argv[i],"-c") || !strcmp(argv[i],"--connection"))
		{
			if ((++i) < argc)
			{
				connection_type = argv[i];
				al_connection_supported_list(connection_type, AGENTD_VERSION, al_connection_supported);
			}
			else
			{
				syntax();
			}
		}
		else
		{
			syntax();
		}

	}


	if ((node = al_xml_get_node(agentd_config(), "connection")) == NULL)
		al_exit(1, "%s\nInvalid config file.\n",AGENTD_VERSION);

	if (!(connection_manager = al_connection_manager_from_string(connection_type, al_connection_supported, node)))
		al_exit(1, "%s\nInvalid connection type.\n", AGENTD_VERSION);
		

	if (run_daemon)
	{
		daemon(1, 0);
	}

	/** Set up signal handlers */
	signal(SIGHUP, agentd_signal_handler);
	signal(SIGINT, agentd_signal_handler);
	signal(SIGQUIT, agentd_signal_handler);
	signal(SIGABRT, agentd_signal_handler);
	signal(SIGKILL, agentd_signal_handler);
	signal(SIGSEGV, agentd_signal_handler);
	signal(SIGTERM, agentd_signal_handler);
	signal(SIGSTOP, agentd_signal_handler);
	signal(SIGTSTP, agentd_signal_handler);

	if (al_connection_manager_need_fork(connection_manager))
	{
		// No transform child in zombie.
#ifdef USE_NOCLDWAIT
		act.sa_handler = SIG_IGN;
		act.sa_flags = SA_NOCLDWAIT;
		act.sa_sigaction = NULL;
		sigaction(SIGCHLD, &act, &oact);
#else
		signal(SIGCHLD, agentd_signal_handler);
#endif
	}

	openlog(AGENTD_VERSION, LOG_PID, LOG_DAEMON);

	while (1)
	{
		connection = al_connection_manager_wait(connection_manager);
		if (connection == NULL) break;

		if (al_connection_manager_need_fork(connection_manager)) pid = fork();
		if (pid == 0 || !al_connection_manager_need_fork(connection_manager))
		{

			if ((pid == 0 && al_connection_manager_need_fork(connection_manager)) || !al_connection_manager_need_fork(connection_manager))
			{
				// Restore signal handler for wait child to finish.
#ifdef USE_NOCLDWAIT
				sigaction(SIGCHLD, &oact, NULL);
#else
				signal(SIGCHLD, SIG_DFL);
#endif
			}

			while (1)
			{
				s = al_connection_read(connection);
				if (s == NULL) break;

				request_packet = agentd_request_packet_from_xml_string(s);
				al_string_delete(s);

				response_packet = agentd_response_packet_from_request_packet(request_packet, agentd_config());
				if (response_packet)
				{
					s = agentd_response_packet_to_xml_string(response_packet);
					al_connection_send(connection, s);
					al_string_delete(s);
				}
				else
				{
					// No valid request/response data. Close the connection.
					break;
				}
			}

			al_connection_delete(connection);
			if (al_connection_manager_need_fork(connection_manager)) break;
		}

	}


	if (pid == 0 && connection_manager) al_connection_manager_delete(connection_manager);

	syslog(LOG_NOTICE, "Closing.\n");

	al_exit(0, NULL);

	return 0;
}


/**
 */
static void agentd_signal_handler(int signum)
{
#ifndef USE_NOCLDWAIT
	int ret;
#endif
	
	
	switch (signum)
	{

#ifndef USE_NOCLDWAIT
	case SIGCHLD:
		wait(&ret);
		break;
#endif

	case SIGINT:
	case SIGQUIT:
	case SIGABRT:
	case SIGKILL:
	case SIGHUP:
	case SIGTERM:
	case SIGSTOP:
	case SIGTSTP:
		if (pid == 0 && connection_manager) al_connection_manager_delete(connection_manager);
		al_exit (0, NULL);
		break;

	case SIGSEGV:
		al_exit (0, NULL);
		break;

	}

}


/**
 */
void syntax(void)
{
	printf("%s\n", AGENTD_VERSION);

	printf("Usage: agentd [OPTION ...]\n");
	printf("\n");
	printf(" Options:\n");
	printf("\n");
	printf("  -d, --daemon           Run as a daemon.\n");
	printf("  -c, --connection       Override the connection type given in config file.\n");
	printf("                         Use -c list to show the supported connection type.\n");
	printf("  -h, --help             Give this help list.\n");
	printf("  -v, --version          Show version and exit.\n");

	al_exit(0, NULL);
}


/**
 */
AlXml * agentd_config(void)
{
	if (config_xml == NULL)
		config_xml = al_xml_new_from_file(AGENTD_CONFIG_FILE);

	if (al_xml_verify(config_xml))
		al_exit (1, "Could not parse config file %s\n", AGENTD_CONFIG_FILE);
	
	return config_xml;
}

