/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include "al.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <fcntl.h>
#include <signal.h>
#include <openssl/ssl.h>
#include <openssl/err.h>


#ifndef ALLOW_OLD_VERSIONS
#if (OPENSSL_VERSION_NUMBER < 0x00905100L)
#error "Must use OpenSSL 0.9.6 or later"
#endif
#endif


/**
 * Delete the connection manager.
 *
 * @param connection_manager connection manager to delete
 */
void al_ssl_connection_manager_delete(AlConnectionManager * connection_manager);


/**
 * Wait for a new connection.
 *
 * @param connection_manager connection manager
 *
 * @return the new connection or NULL if not found.
 */
AlConnection * al_ssl_connection_manager_wait(AlConnectionManager * connection_manager);


/**
 * Do nothing.
 *
 * @param connection connection to close
 */
void al_ssl_connection_close(AlConnection * connection);


/**
 * Read a request.
 *
 * @param connection_manager connection manager
 *
 * @return the new packet or NULL if not found.
 */
AlString * al_ssl_connection_read(AlConnection * connection);


/**
 * Send a response.
 *
 * @param connection_manager connection manager
 * @param packet packet to send
 */
void al_ssl_connection_send(AlConnection * connection, const AlString * response);


/**
 * Connect to a server.
 *
 * @param connection_manager the connection manager
 * @param server server to connect
 *
 * @return the new connection or NULL if fail
 */
AlConnection * al_ssl_connection_manager_connect(AlConnectionManager * connection_manager, const char * server);


/**
 */
struct _AlSslManager
{
	int                 init;

	int                 sock;
	int                 data;
	int                 on;
	struct sockaddr_in  name;
	unsigned long       ioctlarg;
	int                 port;

	int                 s_server_session_id_context;
	int                 s_server_auth_session_id_context;

	SSL_CTX           * ctx;

	DH                * dh;

	char              * key_file;
	char              * cert_file;
	char              * dh_file;
	char              * ca_list_file;
	char              * password;

};

typedef struct _AlSslManager AlSslManager;


/**
 */
struct _AlSsl
{
	
	int                  sock;
	struct sockaddr_in   sn;
	socklen_t            snsize;
	struct hostent     * hostname;
	SSL                * ssl;
	BIO                * sbio;

};

typedef struct _AlSsl AlSsl;


/**
 */
static int al_ssl_password_cb(char *buf,int num, int rwflag, void * userdata);


/**
 */
static void al_ssl_sigpipe_handle(int x);


/**
 */
SSL_CTX * al_ssl_initialize_ctx(AlSslManager * manager);


/**
 */
void al_ssl_destroy_ctx(SSL_CTX * ctx);


/**
 */
int al_ssl_check_cert(SSL * ssl, const char * host);


/**
 */
DH * al_ssl_load_dh_params(SSL_CTX *ctx, const char * file);


/**
 */
AlConnection * al_ssl_connection_new(AlConnectionManager * connection_manager);


/**
 */
void al_ssl_connection_delete(AlConnection * connection);


/**
 */
AlConnectionManager * al_ssl_connection_manager_new(int port, const char * key_file, const char * cert_file, const char * dh_file, const char * ca_list_file, const char * password, int no_peer_verify)
{
	AlSslManager        * ssl_manager = al_new(AlSslManager);
	AlConnectionManager * connection = al_connection_manager_new(al_ssl_connection_manager_wait,
								     al_ssl_connection_manager_delete,
								     al_ssl_connection_manager_connect,
								     1,
								     ssl_manager);
 
	ssl_manager->s_server_session_id_context = 1;
	ssl_manager->s_server_auth_session_id_context = 2;
	
	ssl_manager->port = port;
	ssl_manager->init = 0;
	ssl_manager->key_file = al_strdup(key_file);
	ssl_manager->cert_file = al_strdup(cert_file);
	ssl_manager->dh_file = al_strdup(dh_file);
	ssl_manager->ca_list_file = al_strdup(ca_list_file);
	ssl_manager->password = al_strdup(password);

	/* Build our SSL context*/
	ssl_manager->ctx = al_ssl_initialize_ctx(ssl_manager);

	ssl_manager->dh = (ssl_manager->dh_file) ? al_ssl_load_dh_params(ssl_manager->ctx, ssl_manager->dh_file) : NULL;

	SSL_CTX_set_session_id_context(ssl_manager->ctx,(void*)&(ssl_manager->s_server_session_id_context), sizeof(ssl_manager->s_server_session_id_context)); 
    
	if (!no_peer_verify) SSL_CTX_set_verify(ssl_manager->ctx, SSL_VERIFY_PEER | SSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);


	return connection;
}


/**
 */
void al_ssl_connection_manager_delete(AlConnectionManager * connection_manager)
{
	AlSslManager        * ssl_manager;


	al_return_if_fail (connection_manager);

	if ((ssl_manager = al_connection_manager_get_data(connection_manager, AlSslManager)))
	{
		if (ssl_manager->ctx) al_ssl_destroy_ctx(ssl_manager->ctx);

		al_delete(ssl_manager->key_file);
		al_delete(ssl_manager->cert_file);
		al_delete(ssl_manager->dh_file);
		al_delete(ssl_manager->ca_list_file);
		al_delete(ssl_manager->password);
		al_delete(ssl_manager);
	}

}


/**
 */
AlConnection * al_ssl_connection_new(AlConnectionManager * connection_manager)
{
	AlSsl        * ssl = al_new(AlSsl);
	AlConnection * connection = al_connection_new(connection_manager,
						      al_ssl_connection_read,
						      al_ssl_connection_send,
						      al_ssl_connection_close,
						      al_ssl_connection_delete,
						      ssl);
	
	ssl->snsize = sizeof(ssl->sn);
	
	return connection;
}


/**
 */
void al_ssl_connection_delete(AlConnection * connection)
{
	AlSsl * ssl;


	al_return_if_fail(connection);

	ssl = al_connection_get_data(connection, AlSsl);

	if (ssl->sbio) BIO_free(ssl->sbio);
//@fixme	if (ssl->ssl) SSL_free(ssl);

	al_delete(ssl);
}


/**
 */
AlConnection * al_ssl_connection_manager_wait(AlConnectionManager * connection_manager)
{
	AlConnection  * ret = NULL;
	AlSsl         * ret_ssl = NULL;
	AlSslManager  * ssl_manager;
	int             err;
	long            l_err;
	char            err_msg[200]; 


	al_return_val_if_fail(connection_manager, NULL);

	ssl_manager = al_connection_manager_get_data(connection_manager, AlSslManager);

	if (!ssl_manager->init)
	{

		ssl_manager->sock = socket(PF_INET, SOCK_STREAM, 0);
		if (ssl_manager->sock == -1)
		{
			syslog(LOG_ERR, "Could not create the socket.\n");
			return NULL;
		}

		/* Give the socket a name. */
		ssl_manager->name.sin_family = AF_INET;
		ssl_manager->name.sin_port = htons (ssl_manager->port);
		ssl_manager->name.sin_addr.s_addr = htonl (INADDR_ANY);
		setsockopt(ssl_manager->sock, SOL_SOCKET, SO_REUSEADDR, &(ssl_manager->on), sizeof(ssl_manager->on));
		
		if (bind (ssl_manager->sock, (struct sockaddr *) &(ssl_manager->name), sizeof (ssl_manager->name)) < 0)
		{
			syslog (LOG_ERR, "Could not bind the server. Error %d\n", errno);
			return NULL;
		}

		if (listen (ssl_manager->sock, 5) < 0)
		{
			syslog (LOG_ERR, "Could not SSL start the listener.\n");
			return NULL;
		}

		syslog(LOG_NOTICE, "Acceptiong SSL connections on port %d\n", ssl_manager->port);

		ssl_manager->init = 1;
	}


	do
	{

		ret = al_ssl_connection_new(connection_manager);
		ret_ssl = al_connection_get_data(ret, AlSsl);

		ret_ssl->sock = accept(ssl_manager->sock, (struct sockaddr *)&(ret_ssl->sn), &(ret_ssl->snsize));

		if(ret_ssl->sock == -1)
		{
			al_connection_delete(ret);

			syslog(LOG_NOTICE, "Close connections.\n");

			return NULL;
		}

		ret_ssl->ssl = SSL_new(ssl_manager->ctx);
		ret_ssl->sbio = BIO_new_socket(ret_ssl->sock, BIO_NOCLOSE);
        	SSL_set_bio(ret_ssl->ssl, ret_ssl->sbio, ret_ssl->sbio);

		if((err = SSL_accept(ret_ssl->ssl)) <= 0)
		{
			l_err = ERR_get_error();
			ERR_error_string(l_err, err_msg);
			syslog(LOG_ERR, "SSL accept error %d - %d - %s\n",SSL_get_error(ret_ssl->ssl,err), l_err, err_msg);

			al_connection_delete(ret);
	
			ret = NULL;
        	}

	} while (ret == NULL);

	ret_ssl->hostname = gethostbyaddr(&(ret_ssl->sn.sin_addr), sizeof(ret_ssl->sn.sin_addr), AF_INET);
	if (ret_ssl->hostname) syslog(LOG_NOTICE, "Accepted new SLL connection from %s.\n", ret_ssl->hostname->h_name);


	return ret;
}


/**
 */
void al_ssl_connection_close(AlConnection * connection)
{
	AlSsl * ssl;


	al_return_if_fail(connection);

	ssl = al_connection_get_data(connection, AlSsl);

}


/**
 */
AlString * al_ssl_connection_read(AlConnection * connection)
{
	AlString    * ret = NULL;
	AlString    * s = NULL;
	AlSsl       * ssl_sock;
	const int     len = 3;
	int           i;
	char          buf[len];
	int           rec;


	al_return_val_if_fail(connection, NULL);

	ssl_sock = al_connection_get_data(connection, AlSsl);

	s = al_string_new();
	
	while ((rec = SSL_read(ssl_sock->ssl, buf, sizeof(char) * len)) > 0)
	{

		if (rec == 3 && buf[0] == 'E' && buf[1] == 'O' && buf[2] == 'P')
		{
			// End of request packet
			ret = s;
			break;
		}
		else if (rec == 3 && buf[0] == 'E' && buf[1] == 'O' && buf[2] == 'F')
		{
			// End of trasmisson
			ret = NULL;
			if (ssl_sock->hostname) syslog(LOG_NOTICE, "Close connection from %s.\n", ssl_sock->hostname->h_name);
			shutdown(ssl_sock->sock, SHUT_RDWR);
			break;
		}


		for (i=0;i < rec;i++) al_string_append_char(s, buf[i]);

	}


	if (ret == NULL && s)
	{
		// Incomplete data. discard and close connection.
		al_string_delete(s);
		if (ssl_sock->hostname) syslog(LOG_NOTICE, "Close connection from %s.\n", ssl_sock->hostname->h_name);
		shutdown(ssl_sock->sock, SHUT_RDWR);
	}


	return ret;
}


/**
 */
void al_ssl_connection_send(AlConnection * connection, const AlString * response)
{
	AlSsl * ssl_sock;


	al_return_if_fail(connection && response);

	ssl_sock = al_connection_get_data(connection, AlSsl);

	if (SSL_write(ssl_sock->ssl, al_string_get(response), al_string_len(response)) == -1)
	{
		syslog(LOG_ERR, "Could not send data %s\n", al_string_get(response));
	}

	if (SSL_write(ssl_sock->ssl, "EOP", strlen("EOP")) == -1)
	{
		syslog(LOG_ERR, "Could not send data %s\n", "EOP");
	}

}


/**
 */
AlConnection * al_ssl_connection_manager_connect(AlConnectionManager * connection_manager, const char * server)
{
	AlSslManager   * ssl_manager;
	AlConnection   * ret;
	AlSsl          * ret_ssl;
	struct hostent * pent;
	int             err;
	long            l_err;
	char            err_msg[200];



	al_return_val_if_fail(connection_manager && server, NULL);
	
	ssl_manager = al_connection_manager_get_data(connection_manager, AlSslManager);

	if (!ssl_manager->init)
	{
		ssl_manager->init = 1;
	}

	ret = al_ssl_connection_new(connection_manager);
	ret_ssl = al_connection_get_data(ret, AlSsl);

	ret_ssl->sn.sin_family = AF_INET;

	/* Figure out the internet address. */
	ret_ssl->sn.sin_addr.s_addr = inet_addr(server);
	
	if (ret_ssl->sn.sin_addr.s_addr == INADDR_NONE)
	{
		pent = gethostbyname(server);
		if (pent)
		{
			ret_ssl->sn.sin_addr.s_addr = *((unsigned long *) (pent->h_addr));
		}
		else
		{
			al_log (("Could not resolve the host %s\n", server));
			al_connection_delete(ret);
			return NULL;
		}
	}

	ret_ssl->sock = socket(PF_INET, SOCK_STREAM, 0);
	if (ret_ssl->sock == -1)
	{
		al_log(("Could not create the socket.\n"));
		al_connection_delete(ret);
		return NULL;
	}

	/* Figure out the port name. */
	ret_ssl->sn.sin_port =  htons(ssl_manager->port);

	if (connect(ret_ssl->sock, (struct sockaddr*)&ret_ssl->sn, ret_ssl->snsize) == -1)
	{
		al_log (("Could not connect to %s at port %d\n", server, ssl_manager->port));
		al_connection_delete(ret);
		ret = NULL;
	}
	else
	{
		ret_ssl->ssl = SSL_new(ssl_manager->ctx);
		ret_ssl->sbio = BIO_new_socket(ret_ssl->sock, BIO_NOCLOSE);
        	SSL_set_bio(ret_ssl->ssl, ret_ssl->sbio, ret_ssl->sbio);

		if ((err = SSL_connect(ret_ssl->ssl)) <= 0)
		{
			l_err = ERR_get_error();
			ERR_error_string(l_err, err_msg);
			fprintf(stderr,"SSL connect error %d - %d - %s\n",SSL_get_error(ret_ssl->ssl,err), l_err, err_msg);
			al_connection_delete(ret);
			ret = NULL;
		}
		else
		{
			if (!al_ssl_check_cert(ret_ssl->ssl, server))
			{
				// Invalid certificate
				al_connection_delete(ret);
				ret = NULL;
			}
		}
	}


	return ret;
}


/**
 */
SSL_CTX * al_ssl_initialize_ctx(AlSslManager * manager)
{
	SSL_METHOD * meth;
	SSL_CTX    * ctx;


	/* Global system initialization*/
	SSL_library_init();
	SSL_load_error_strings();

	/* Set up a SIGPIPE handler */
	signal(SIGPIPE,al_ssl_sigpipe_handle);

	/* Create our context*/
	meth = SSLv23_method();
	ctx = SSL_CTX_new(meth);

	/* Load our keys and certificates*/
	if(!(SSL_CTX_use_certificate_chain_file(ctx, manager->cert_file)))
		fprintf(stderr, "Can't read certificate file %s.\n", manager->cert_file);

	SSL_CTX_set_default_passwd_cb(ctx, al_ssl_password_cb);
	SSL_CTX_set_default_passwd_cb_userdata(ctx, manager->password);

	if(!(SSL_CTX_use_PrivateKey_file(ctx, manager->key_file,SSL_FILETYPE_PEM)))
		fprintf(stderr, "Can't read key file %s.\n", manager->key_file);

	if (manager->ca_list_file)
	{
		/* Load the CAs we trust*/
		if(!(SSL_CTX_load_verify_locations(ctx, manager->ca_list_file,0)))
			fprintf(stderr, "Can't read CA list %s.\n", manager->ca_list_file);
	}

#if (OPENSSL_VERSION_NUMBER < 0x00905100L)
	SSL_CTX_set_verify_depth(ctx,1);
#endif

	return ctx;
}
 

/**
 */
void al_ssl_destroy_ctx(SSL_CTX *ctx)
{
	SSL_CTX_free(ctx);
}


/**
 * The password code is not thread safe
 */
static int al_ssl_password_cb(char *buf, int num, int rwflag, void * userdata)
{
	char * password = userdata;

	if(num < strlen(password) + 1)
		return 0;

	strcpy(buf, password);

	return strlen(password);
}


/**
 */
static void al_ssl_sigpipe_handle(int x)
{
}


/**
 */
int al_ssl_check_cert(SSL * ssl, const char * host)
{
	X509 * peer;
	char   peer_CN[256];



	if(SSL_get_verify_result(ssl) != X509_V_OK)
	{
		fprintf(stderr, "Certificate doesn't verify (host %s)\n", host);
		return 0;
	}


	/**
	 * Check the cert chain. The chain length is automatically checked by OpenSSL when
	 * we set the verify depth in the ctx.
         */

	/* Check the common name*/
	peer = SSL_get_peer_certificate(ssl);
	X509_NAME_get_text_by_NID(X509_get_subject_name(peer), NID_commonName, peer_CN, 256);
	if(strcasecmp(peer_CN,host))
	{
		fprintf(stderr, "Common name doesn't match host name (host %s)\n", host);
		return 0;
	}

	return 1;
}


/**
 */
DH * al_ssl_load_dh_params(SSL_CTX *ctx, const char * file)
{
	DH  * ret = NULL;
	BIO * bio;


	al_return_val_if_fail(file && ctx, NULL);

	if ((bio=BIO_new_file(file,"r")) == NULL)
	{
		fprintf(stderr, "Couldn't open DH file %s\n", file);
	}
	else
	{
		ret = PEM_read_bio_DHparams(bio, NULL, NULL, NULL);
		BIO_free(bio);

		if(SSL_CTX_set_tmp_dh(ctx,ret) < 0)
		{
			ret = NULL;
			fprintf(stderr, "Couldn't set DH parameters\n");
		}
	}


	return ret;
}

