# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'test/unit'
require 'time'
require 'msgcab/nov'

include MsgCab

class TestNov < Test::Unit::TestCase
  Fields0 = [
    1,                          # number
    'none',                     # subject
    'nobody',                   # from
    Time.at(0).rfc2822,         # date
    '<aaa@bbb>',                # message_id
    '<ccc@ddd> <eee@fff> <ggg@hhh>', # references
    1024,                       # chars
    76                          # lines
  ]

  Fields1 = [
    1,                          # number
    nil,                        # subject
    nil,                        # from
    Time.at(0).rfc2822,         # date
    '<aaa@bbb>',                # message_id
    '<ccc@ddd> <eee@fff> <ggg@hhh>', # references
    1024,                       # chars
    76                          # lines
  ]

  def test_s_parse
    nov = Nov.parse(Fields0.join("\t"))
    assert_equal(1, nov.number)
    assert_equal('none', nov.subject)
    assert_equal('nobody', nov.from)
    assert_equal(Time.at(0), nov.date)
    assert_equal('<aaa@bbb>', nov.message_id)
    assert_equal(['<ccc@ddd>', '<eee@fff>', '<ggg@hhh>'], nov.references)
    assert_equal(1024, nov.chars)
    assert_equal(76, nov.lines)
    assert_equal([], nov.extra)

    nov = Nov.parse(Fields1.join("\t"))
    assert_equal(1, nov.number)
    assert_nil(nov.subject)
    assert_nil(nov.from)
    assert_equal(Time.at(0), nov.date)
    assert_equal('<aaa@bbb>', nov.message_id)
    assert_equal(['<ccc@ddd>', '<eee@fff>', '<ggg@hhh>'], nov.references)
    assert_equal(1024, nov.chars)
    assert_equal(76, nov.lines)
    assert_equal([], nov.extra)

    nov = Nov.parse(Fields0.join("\t") + "\t")
    assert_equal([], nov.extra)
    fields = Fields0.dup
    fields << 'a:1' << 'b:2'
    nov = Nov.parse(fields.join("\t"))
    assert_equal([['a', '1'], ['b', '2']], nov.extra)
  end

  def test_to_s
    str = Fields0.join("\t")
    nov = Nov.parse(str)
    assert_equal(str, nov.to_s.sub(/\t+\z/, ''))

    str = Fields1.join("\t")
    nov = Nov.parse(str)
    assert_equal(str, nov.to_s.sub(/\t+\z/, ''))
  end

  def test_decode
    str = Fields0.join("\t")
    nov = Nov.parse(str)
    decoded = nov.decode('UTF-8')
    assert_equal(Time.at(0).strftime('%Y-%m-%d'), decoded.date)
    assert_equal('nobody', decoded.from)
    assert_equal('none', decoded.subject)

    str = Fields1.join("\t")
    nov = Nov.parse(str)
    decoded = nov.decode('UTF-8')
    assert_equal(Time.at(0).strftime('%Y-%m-%d'), decoded.date)
    assert_equal('(nobody)', decoded.from)
    assert_equal('(none)', decoded.subject)
  end

  def test_s_canonicalize_subject
    assert_equal('bbb', MsgCab.canonicalize_subject('Re[10]: Forward: [aaa] bbb (re: ccc)'))
    assert_equal('bbb', MsgCab.canonicalize_subject('Re[10]: Forward: [aaa] bbb (was ccc)'))
  end

  def test_s_canonicalize_from
    assert_equal('aaa@bbb', MsgCab.canonicalize_from('<aaa@bbb>'))
    assert_equal('aaa@bbb', MsgCab.canonicalize_from('aaa@bbb'))
    assert_equal('aaa', MsgCab.canonicalize_from('aaa <aaa@bbb>'))
    assert_equal('aaa', MsgCab.canonicalize_from('"aaa" <aaa@bbb>'))
    assert_equal('aaa', MsgCab.canonicalize_from('aaa@bbb (aaa)'))
  end
end
