# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'msgcab/disjoint'
require 'msgcab/nov'
require 'msgcab/eword'
require 'msgcab/cli/common'
require 'rmail'
require 'digest/sha1'
require 'msgcab/progressbar'
require 'msgcab/logger'

module MsgCab
  module CLI
    class Register < Common
      include Logging

      def register(source)
        numbers = Array.new
        pbar = ProgressBar.new('register', source.size) unless opts[:quiet]
        source.each do |chunk|
          digest = Digest::SHA1.hexdigest(chunk.data)
          number = CLI.database.numbers_by_digest(digest).detect {|number|
            begin
              CLI.mailtree.fetch(number) == chunk.data
            rescue IndexError
              false
            end
          }
          if number
            log(2, "#{chunk.desc} >>> #{number} (already exists)")
          else
            number = CLI.database.max.succ
            update_database(number, chunk.data)
            CLI.mailtree.store(number, chunk.data)
            log(2, "#{chunk.desc} >>> #{number}")
            numbers << number
          end
          pbar.inc unless opts[:quiet]
        end
        run_callback(:after_import, numbers)
        pbar.finish unless opts[:quiet]
        numbers
      end

      private
      class StreamHandler < RMail::StreamHandler # :nodoc:
        def initialize
          @header = RMail::Header.new
          @chars = 0
          @lines = 0
        end
        attr_reader :header, :chars, :lines

        def header_field(field, name, value)
          @header.add(name, value)
        end

        def body_chunk(chunk)
          @chars += chunk.size
          @lines += chunk.count("\r\n")
        end
      end

      def make_msg_id(input)
        "<#{Digest::SHA1.hexdigest(input)}@unknown>"
      end

      def update_database(number, input)
        handler = StreamHandler.new
        RMail::StreamParser.parse(input, handler)
        header = handler.header

        body = header['message-id']
        if body
          msg_id, = body.scan(/<.+?>/)
        end
        unless msg_id
          msg_id = make_msg_id(input)
        end

        ref_ids = []
        header.select('in-reply-to', 'references').each do |name, body|
          body.scan(/<.+?>/) do |ref_id|
            ref_ids |= [ref_id]
          end
        end
        ref_ids.delete(msg_id)

        digest = Digest::SHA1.hexdigest(input)
        subject = header.fetch('subject', '(none)')
        body = header.fetch('date', Time.at(0).rfc2822)
        date = Time.rfc2822(body) rescue Time.parse(body)
        nov = MsgCab::Nov.new(number,
                              subject,
                              header['from'] || '(nobody)',
                              date,
                              msg_id,
                              ref_ids,
                              handler.chars,
                              handler.lines)
        CLI.database.update_message(number,
                                    msg_id,
                                    digest,
                                    date,
                                    nov,
                                    ref_ids,
                                    header)
      end
    end
  end
end

module MsgCab
  class Database
    def update_parent(msg_id)
      if @adapter.select_one(<<'End', msg_id)
SELECT * FROM parent WHERE msg_id = ?
End
        log(3, "#{msg_id} is already registered as parent")
        return
      end

      not_arrived_msg_ids =
        @adapter.select_all(<<'End', msg_id).collect {|row| row[0]}
SELECT msg_id FROM not_arrived WHERE ref_id = ?
End
      @adapter.execute(<<'End', msg_id)
DELETE FROM not_arrived WHERE ref_id = ?
End
      not_arrived_msg_ids.each do |not_arrived_msg_id|
        update_parent(not_arrived_msg_id)
      end

      not_arrived_ref_ids = Array.new
      stmt = @adapter.prepare(<<'End')
SELECT * FROM message WHERE msg_id = ?
End
      ref_ids = ancestors(msg_id)
      ref_ids.dup.each do |ref_id|
        stmt.execute(ref_id)
        if stmt.fetch
          ref_ids -= ancestors(ref_id)
          break if ref_ids.length <= 1
        else
          not_arrived_ref_ids << ref_id
        end
      end
      if ref_ids.length <= 1 || not_arrived_ref_ids.empty?
        @adapter.execute(<<'End', msg_id)
DELETE FROM not_arrived WHERE msg_id = ?
End
      end
      if ref_ids.length <= 1
        parent = ref_ids.length == 1 ? ref_ids[0] : msg_id
        @adapter.execute(<<'End', msg_id, parent)
INSERT INTO parent VALUES (?, ?)
End
      else
        not_arrived_ref_ids.each do |not_arrived_ref_id|
          unless @adapter.select_one(<<'End', msg_id, not_arrived_ref_id)
SELECT * FROM not_arrived WHERE msg_id = ? AND ref_id = ?
End
            @adapter.execute(<<'End', msg_id, not_arrived_ref_id)
INSERT INTO not_arrived VALUES (?, ?)
End
          end
        end
      end
    end

    def update_message(number,
                       msg_id,
                       digest,
                       date,
                       nov,
                       ref_ids,
                       header)
      stmt = @adapter.prepare(<<'End')
INSERT INTO message VALUES (?, ?, ?, ?, ?)
End
      stmt.execute(number,
                   msg_id,
                   digest,
                   date.dup.utc.xmlschema,
                   nov.to_s)

      stmt = @adapter.prepare(<<'End')
INSERT INTO header VALUES (?, ?, ?)
End
      header.each do |name, body|
        stmt.execute(number, name.downcase, body) unless body.empty?
      end

      stmt = @adapter.prepare(<<'End')
INSERT INTO reference VALUES (?, ?)
End
      ref_ids.each do |ref_id|
        stmt.execute(msg_id, ref_id)
      end
      update_parent(msg_id)
      update_disjoint(number, msg_id)
    end

    def update_disjoint(number, msg_id)
      connected = Array.new
      (ancestors(msg_id) | descendants(msg_id)).each do |connected_id|
        connected |= numbers_by_msg_id(connected_id)
      end
      connected.dup.each do |x|
        row = @adapter.select_one(<<'End', x)
SELECT parent FROM disjoint WHERE number = ?
End
        if row
          rows = @adapter.select_all(<<'End', row[0].to_i)
SELECT number FROM disjoint WHERE parent = ?
End
          connected |= rows.collect {|row| row[0].to_i}
        end
      end

      disjoint = Disjoint.new
      disjoint.make_set(number)
      connected.each do |x|
        disjoint.make_set(x)
      end
      connected.each do |y|
        disjoint.union_sets(number, y)
      end
      disjoint.to_a.each do |x, rank, parent|
        row = @adapter.select_one(<<'End', x)
SELECT * FROM disjoint WHERE number = ?
End
        if row
          stmt = @adapter.prepare(<<'End')
UPDATE disjoint SET rank = ?, parent = ? WHERE number = ?
End
          stmt.execute(rank, parent, x)
        else
          stmt = @adapter.prepare(<<'End')
INSERT INTO disjoint VALUES (?, ?, ?)
End
          stmt.execute(x, rank, parent)
        end
      end
    end
  end
end
