# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'test/unit'
require 'pathname'
require 'fileutils'
require 'msgcab/database'
require 'msgcab/nov'

include MsgCab

class TestDatabase < Test::Unit::TestCase
  def setup
    path = Pathname.new(__FILE__).dirname + 'test-database0.yml'
    Config.load(path)
    @db_path = Pathname.new(__FILE__).dirname + 'db/database.sqlite'
  end

  def teardown
    Database.disconnect
    FileUtils.rm_rf(@db_path.dirname.to_s)
  end

  def test_s_instance_dbi
    assert_nothing_raised do
      assert_equal('MsgCab::DBI', Database.instance.adapter.class.name)
    end
  end

  def test_load_schema
    tables = Array.new
    (Pathname.new(__FILE__).dirname + '../msgcab/schema.sql').open do |file|
      loop do
        line = file.gets
        break unless line
        tables << $1.downcase if line =~ /^CREATE\s+TABLE\s+(\S+)/i
      end
    end
    assert_equal(Database.instance.adapter.tables.sort,
                 tables.sort)
  end

  Fields0 = [
    0,                          # number
    'none',                     # subject
    'nobody',                   # from
    Time.at(0).rfc2822,         # date
    '<aaa@bbb>',                # message_id
    '<ccc@ddd> <eee@fff> <ggg@hhh>', # references
    1024,                       # chars
    76                          # lines
  ]

  Fields1 = [
    1,                          # number
    'none',                     # subject
    'nobody',                   # from
    Time.at(0).rfc2822,         # date
    '<aaa@bbb>',                # message_id
    '<ccc@ddd> <eee@fff> <ggg@hhh>', # references
    1024,                       # chars
    76                          # lines
  ]

  def test_nov
    str = Fields0.join("\t")
    nov = Nov.parse(str)
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO message VALUES (?, ?, ?, ?, ?)
End
    stmt.execute(nov.number,
                 nov.message_id,
                 'digest',
                 nov.date.dup.utc.xmlschema,
                 nov.to_s)
    nov = Database.instance.nov(nov.number)
    assert_instance_of(Nov, nov)
    assert_equal(str, nov.to_s)
  end

  def test_folder
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO filter VALUES (?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0, 'foo', 2, (time - 1).xmlschema)
    stmt.execute(1, 'foo', 5, time.xmlschema)
    folder = Database.instance.folder('foo')
    assert_instance_of(OpenStruct, folder)
    assert_equal('foo', folder.name)
    assert_equal(2, folder.min)
    assert_equal(5, folder.max)
    assert_instance_of(Time, folder.time)
    assert_equal(time.xmlschema, folder.time.xmlschema)
  end

  def test_folders
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO filter VALUES (?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0, 'foo', 2, time.xmlschema)
    stmt.execute(1, 'bar', 5, time.xmlschema)
    folders = Database.instance.folders
    assert_equal(['bar', 'foo'], folders.collect {|folder| folder.name}.sort)
  end

  def test_max
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO message VALUES (?, ?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0,
                 '<aaa@bbb>',
                 'digest',
                 time.xmlschema,
                 '')
    stmt.execute(1,
                 '<aaa@bbb>',
                 'digest',
                 time.xmlschema,
                 '')
    assert_equal(1, Database.instance.max)
  end

  def test_recent_summary
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO message VALUES (?, ?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0,
                 '<aaa@bbb>',
                 'digest',
                 time.xmlschema,
                 Fields0.join("\t"))
    stmt.execute(1,
                 '<aaa@bbb>',
                 'digest',
                 time.xmlschema,
                 Fields1.join("\t"))
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO filter VALUES (?, ?, ?, ?)
End
    stmt.execute(0, 'foo', 2, time.xmlschema)
    stmt.execute(1, 'foo', 5, time.xmlschema)
    summary = Database.instance.recent_summary('foo', 10)
    assert_equal(2, summary.size)
    assert_equal(1, summary[0].number)
    assert_equal(0, summary[1].number)
    summary = Database.instance.recent_summary('foo', 1)
    assert_equal(1, summary.size)
  end

  def test_number_summary
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO message VALUES (?, ?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0,
                 '<aaa@bbb>',
                 'digest',
                 time.xmlschema,
                 Fields0.join("\t"))
    stmt.execute(1,
                 '<aaa@bbb>',
                 'digest',
                 time.xmlschema,
                 Fields1.join("\t"))
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO filter VALUES (?, ?, ?, ?)
End
    stmt.execute(0, 'foo', 2, time.xmlschema)
    stmt.execute(1, 'foo', 5, time.xmlschema)
    summary = Database.instance.number_summary('foo', 0, 10)
    assert_equal(2, summary.size)
    assert_equal(0, summary[0].number)
    assert_equal(1, summary[1].number)
    summary = Database.instance.recent_summary('foo', 1)
    assert_equal(1, summary.size)
    assert_equal(1, summary[0].number)
  end

  def test_message
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO message VALUES (?, ?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0,
                 '<aaa@bbb>',
                 'digest',
                 time.xmlschema,
                 Fields0.join("\t"))
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO filter VALUES (?, ?, ?, ?)
End
    stmt.execute(0, 'foo', 2, time.xmlschema)
    message = Database.instance.message(0)
    assert_instance_of(OpenStruct, message)
    assert_equal('foo', message.folder)
    assert_equal(2, message.folder_number)
    assert_equal(0, message.number)
    assert_equal('<aaa@bbb>', message.msg_id)
    assert_equal('digest', message.digest)
    assert_equal(time.xmlschema, message.date.xmlschema)
    assert_equal(Fields0.join("\t"), message.nov.to_s)
  end

  def test_from_folder_number
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO filter VALUES (?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0, 'foo', 2, time.xmlschema)
    assert_equal(0, Database.instance.from_folder_number('foo', 2))
  end

  def test_to_folder_number
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO filter VALUES (?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0, 'foo', 2, time.xmlschema)
    assert_equal(['foo', 2], Database.instance.to_folder_number(0))
  end

  def test_numbers
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO filter VALUES (?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0, 'foo', 1, time.xmlschema)
    stmt.execute(1, 'foo', 3, time.xmlschema)
    stmt.execute(2, 'foo', 5, time.xmlschema)
    assert_equal([1, 3, 5], Database.instance.numbers('foo', 1, 10).sort)
  end

  def test_numbers_by_msg_id
    stmt = Database.instance.adapter.prepare(<<'End')
INSERT INTO message VALUES (?, ?, ?, ?, ?)
End
    time = Time.now.utc
    stmt.execute(0,
                 '<aaa@bbb>',
                 'digest',
                 time.xmlschema,
                 Fields0.join("\t"))
    stmt.execute(1,
                 '<aaa@bbb>',
                 'digest',
                 time.xmlschema,
                 Fields1.join("\t"))
    assert_equal(2, Database.instance.numbers_by_msg_id('<aaa@bbb>').size)
  end
end
