# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'GD'
require 'face/uncompface'
require 'msgcab/callback'

class FacePlugin < MsgCab::Plugin
  def start_plugin
    MsgCab::Callback.add_callback(:view_initialize, :add_face_anchor) do |view|
      view.add_anchor(:entity_header, FaceAnchor)
    end

    class << attributes[:controller]
      define_rule(:face,
                  %r|\A#{FolderPat}/face/(\d+)\.(\d+)\.png\z|,
                  :folder_name, :folder_number, :face_number)

      def do_face(webapp, folder_name, folder_number, face_number)
        number = database.from_folder_number(folder_name, folder_number)
        unless number
          raise InvalidRequest, "no such message: #{folder_name}:#{folder_number}"
        end
        faces = collect_faces(MsgCab::Entity.parse(mailtree.fetch(number)))
        if faces.empty?
          raise InvalidRequest, "no face header: #{folder_name}:#{folder_number}"
        end

        face_number = face_number.to_i
        if face_number < faces.length
          case faces[face_number].name
          when /\AX-Face\z/
            png = decode_x_face(faces[face_number].body)
          when /\AFace\z/
            png = decode_face(faces[face_number].body)
          end
        else
          raise InvalidRequest, "no such face header"
        end
        webapp << png
      end

      def collect_faces(entity)
        entity.header_fields('x-face', 'face')
      end

      def decode_face(s)
        s.gsub(/[ \t\n]/, '').unpack('m*')[0]
      end

      def decode_x_face(s)
        xface = Compface.uncompface(s)
        image = GD::Image.new(48, 48)
        black = image.colorAllocate(0, 0, 0)
        white = image.colorAllocate(255, 255, 255)
        image.transparent(white)
        48.times do |i|
          48.times do |j|
            image.line(j, i, j + 1, i + 1,
                       xface[i * 48 + j] == 0 ? white : black)
          end
        end
        image.pngStr
      end
    end
  end

  class FaceAnchor < MsgCab::WebApp::Anchor
    include HTree.compile_template(<<'End')
<div _template="replace">
  <span _iter="view.entity.header_fields('x-face', 'face').each_with_index//field,index">
    <img _attr_src='view.uri("/#{view.attributes[:folder].name}/face/#{view.attributes[:number]}.#{index}.png")'
        _attr_alt="field.name"/>
  </span>
  <br/>
  <span _call="next_anchor.replace"></span>
</div>
End
  end
end
