# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'msgcab/entity'

class MsgCab::WebApp::EntityView < MsgCab::WebApp::View
  include MsgCab
  include MsgCab::WebApp

  def initialize(webapp, entity)
    @entity = entity
    super(webapp)
    if renderer_template = self.class.renderer_template(entity)
      # Define "render_entity_body" template.
      load_template(renderer_template)
      # Define "render_entity" template.
      load_template('_entity.html')
      @body_visible = true
    end
    if entity.content_type == 'message/rfc822'
      @body_entity_view = self.class.new(webapp, Entity.parse(entity.decode))
      @body_entity_view.load_template('_entity.html')
    end
  end
  attr_reader :entity

  def self.renderer_template(entity)
    if entity.multipart?
      '_multipart.html'
    elsif entity.content_type == 'application/octet-stream' &&
        entity.header_param('content-type', 'type') == 'patch'
      '_text.html'
    else
      RendererTemplates[entity.content_type]
    end
  end

  def each
    @entity.each do |child|
      view = self.class.new(webapp, child)
      view.attributes = attributes.dup
      yield view
    end
  end

  def cid
    "1.#{@entity.cid.collect {|i| i.succ}.join('.')}"
  end

  def body_visible?
    @body_visible
  end

  def range
    page_max = Config['webapp', 'page_max'] || 100
    min = attributes[:number] / page_max * page_max + 1
    Range.new(min, min + page_max - 1)
  end

  def decode_body
    if attributes.key?(:folder)
      from_charset = Config['charset', attributes[:folder].name]
    else
      from_charset = Config['charset']
    end
    str = @entity.decode_body(MsgCab.kcode_charset, from_charset)
    str.rstrip!
    str
  end

  def decode_header_fields(*names)
    @entity.header_fields(*names).collect {|field|
      Field.new(field.name, EWord::Decoder.decode(field.body, encoding))
    }
  end

  def decode_all_header_fields
    names = Config['webapp', 'header_field_names'] || HeaderFieldNames
    extra_names = Config['webapp', 'extra_header_field_names']
    if extra_names
      names |= extra_names
    end
    decode_header_fields(*names)
  end

  def button
    filename = entity.header_param('content-disposition', 'filename')
    if filename
      '%s %s <%s>' % [cid, filename, entity.content_type]
    else
      '%s <%s>' % [cid, entity.content_type]
    end
  end

  HeaderFieldNames = ['from', 'subject', 'date']

  RendererTemplates = {
    'text/plain' 	=> '_text.html',
    'text/richtext' 	=> '_text.html',
    'text/enriched' 	=> '_text.html',
    'text/x-patch' 	=> '_text.html',
    'message/rfc822'	=> '_message.html',
    'application/emacs-lisp' 	=> '_text.html'
  }
end
