# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'open3'

class EstraierPlugin < MsgCab::Plugin
  def start_plugin
    MsgCab::Callback.add_callback(:view_initialize, :add_estraier_anchor) do
      |view|
      view.add_anchor(:header, EstraierAnchor)
    end

    class << attributes[:controller]
      define_rule(:search,
                  %r|\A#{FolderPat}/search\z|,
                  :folder_name)

      def do_search(webapp, folder_name)
        q = webapp.query_html_get_application_x_www_form_urlencoded
        q = q['q'].untaint
        est_path = MsgCab::Config.absolute_path(MsgCab::Config['estraier', 'estraier_index_path'] ||
                                          './est')
        estxview_path = MsgCab::Config['estraier', 'estxview_path']
        if estxview_path
          estxview_path = MsgCab::Config.absolute_path(estxview_path)
        else
          estxview_path = 'estxview'
        end
        estxview_opts = MsgCab::Config['estraier', 'estxview_opts'] || Array.new
        res, err = Open3.popen3(*[estxview_path,
                                  estxview_opts.collect {|opt| opt.untaint},
                                  '-nk', '-nt', '-ni',
                                  "#{est_path}/#{folder_name.untaint}", q].flatten) {
          |pw, pr, pe|
          [pr.read, pe.read]
        }
        
        if err =~ /\A#{Regexp.escape(estxview_path)}: */
          raise MsgCab::WebApp::InvalidRequest, $' if res.empty? # '
        end

        view = MsgCab::WebApp::View.new(webapp)
        summary = Array.new

        t = HTree(res.untaint)
        t.traverse_element('document') do |e|
          uri = e.find_element('uri').extract_text.to_s
          number = mailtree.path_to_number(Pathname.new(uri))
          folder_number = database.to_folder_number(folder_name, number)
          message = database.message(folder_name, folder_number, number)
          message.nov = message.nov.decode(view.encoding)
          summary << message
        end

	view.attributes[:folder] = database.folder(folder_name)
        view.attributes[:summary] = summary
        view.attributes[:banner] =
          summary.empty? ? "#{folder_name}:no match for \"#{q}\"" : 
          "#{folder_name}:#{summary.length} matches for \"#{q}\""
        view.output_template('summary.html')
      end
    end
  end

end

class EstraierAnchor < MsgCab::WebApp::Anchor
  def initialize(chain, view)
    super(chain, view)
  end

  include HTree.compile_template(<<'End')
<div _template="replace">
  <span _if="view.attributes[:folder]">
    <form _attr_action='view.uri("/#{view.attributes[:folder].name}/search")' accept-charset="UTF-8">
      <span _text='"Search in #{view.attributes[:folder].name}:"></span>
      <input type="text" size="16" name="q"/>
      <input type="submit" value="Search"/>
    </form>
  </span>
  <span _call="next_anchor.replace"></span>
</div>
End
end
