# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'time'
require 'msgcab/eword'
require 'delegate'
require 'msgcab/encode'

module MsgCab
  class Nov
    def initialize(number, subject, from, date, message_id,
                   references, chars, lines, extra = [])
      @number = number
      @subject = subject
      @from = from
      @date = date
      @message_id = message_id
      @references = references
      @chars = chars
      @lines = lines
      @extra = extra
    end
    attr_accessor :number, :subject, :from, :date, :message_id,
    :references, :chars, :lines, :extra

    def self.parse(s)
      fields = s.sub(/\t+\z/, '').split("\t", 9).collect {|field|
        field == "" ? nil : field
      }
      fields[0] = fields[0].to_i
      fields[3] = Time.rfc2822(fields[3]) rescue Time.parse(fields[3])
      unless fields[5]
        fields[5] = Array.new
      else
        fields[5] = fields[5].split(/ +/)
      end
      fields[6] = fields[6].to_i
      if fields[7]
        fields[7] = fields[7].to_i
      end
      if fields[8]
        extra = fields[8].split("\t").collect {|field|
          if field =~ /\A([^\x0-\x20:\x80-\xFF]+): */
            [$1, $']	# '
          end
        }
        extra.compact!
      else
        extra = Array.new
      end
      fields[8] = extra
      Nov.new(*fields)
    end

    def to_s
      [
        number,
        subject,
        from,
        date ? date.rfc2822 : '',
        message_id,
        references.join(' '),
        chars || 0,
        lines || 0,
        *extra.collect {|name, body| "#{name}: #{body}"}
      ].collect! {|field| field.to_s.gsub(/[\t\n]/, ' ')}.join("\t")
    end

    def decode(encoding = MsgCab.kcode_charset)
      DecodedNov.new(self, encoding)
    end
  end

  class DecodedNov < DelegateClass(Nov)
    def initialize(nov, encoding)
      @encoding = encoding
      super(nov)
    end

    def date
      __getobj__.date.strftime('%Y-%m-%d')
    end

    def from
      EWord::Decoder.decode(__getobj__.from || '(nobody)', @encoding)
    end

    def subject
      EWord::Decoder.decode(__getobj__.subject || '(none)', @encoding)
    end

    def canonical_subject
      MsgCab.canonicalize_subject(subject)
    end

    def canonical_from
      MsgCab.canonicalize_from(from)
    end
  end

  def canonicalize_subject(s)
    s.sub!(/\A\s+/, '')
    s.sub!(/\s+\z/, '')
    s.gsub!(/\s+/, ' ')
    loop do
      break if !s.gsub!(/\A(?:(?:re(?:\[\d+\])?|forward|fwd?):\s*)+/i, '') &&
        !s.gsub!(/\A\[.+?\]\s*/i, '')
    end
    s.gsub!(/\s*\(\s*(?:was\s+|re:).*/i, '')
    s
  end

  def canonicalize_from(s)
    s.sub!(/\A\s+/, '')
    s.sub!(/\s+\z/, '')
    s.gsub!(/\s+/, ' ')
    case s
    when /<([^>]+)>\z/
      b = $~.begin(0)
      if s =~ /\A\"(.+)\"/
        $1
      elsif b > 0
        r = s[0 .. b - 1].strip
      else
        s[1 .. -2]
      end
    when /\((.+)\)/
      $1
    else
      s
    end
  end

  module_function :canonicalize_subject
  module_function :canonicalize_from
end
