# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'yaml'
require 'pathname'

module MsgCab
  class Config
    def initialize(path)
      @path = path.expand_path.untaint
      @mtime = Time.at(0)
      @hash = nil
    end

    attr_reader :path

    def [](*keys)
      if @path.exist? && @path.mtime > @mtime
        @hash = @path.open {|file| YAML.load(file)}
        @mtime = @path.mtime
      end
      return nil unless @hash
      value = @hash[keys.shift].untaint
      keys.each do |key|
        break unless value
        value = value[key].untaint
      end
      value
    end

    def relative_path(path)
      pathname = Pathname.new(path)
      if pathname.absolute?
        pathname.relative_path_from(@path.dirname)
      else
        (@path.dirname + pathname).relative_path_from(@path.dirname)
      end
    end

    def absolute_path(path)
      pathname = Pathname.new(path)
      if pathname.absolute?
        pathname
      else
        (@path.dirname + pathname).expand_path
      end
    end

    class << self
      @@instance = nil
      def load(path)
        unless @@instance && @@instance.path == path.expand_path
          @@instance = new(path)
        end
        @@instance
      end

      def [](*keys)
        @@instance[*keys]
      end

      def relative_path(path)
        @@instance.relative_path(path)
      end

      def absolute_path(path)
        @@instance.absolute_path(path)
      end
    end
  end
end
