# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'msgcab/cli/common'
require 'time'
require 'msgcab/progressbar'
require 'msgcab/logger'

module MsgCab
  module CLI
    class Property < Common
      include Logging

      def load_rule(name)
        path = Config.absolute_path(Config['update_dir'] || './update') + "#{name}.rb"
        load(path.to_s)
        rule_class = self.class.const_get("#{name.split(/[-_]+/).collect {|word| word.capitalize}.join}Rule")
        rule_class.new
      end

      def update(props, numbers)
        update_numbers = Hash.new {|hash, key| numbers}
        if numbers.empty?
          props.each do |prop|
            if prop == 'filter'
              update_numbers[prop] = CLI.database.numbers_to_update_filter
            else
              update_numbers[prop] = CLI.database.numbers_to_update_property(prop)
            end
          end
        end
        total = props.inject(0) {|memo, prop| memo + update_numbers[prop].length}
        pbar = ProgressBar.new('Property', total) unless opts[:quiet]
        props.each do |prop|
          begin
            rule = load_rule(prop)
          rescue Exception => e
            log(2, "Failed to load #{prop} rule: #{e}")
            next
          end
          update_numbers[prop].each do |number|
            result = rule.apply(number)
            if result
              if prop == 'filter'
                folder_name, folder_number = result.split(':')
                CLI.database.update_filter(number, folder_name, folder_number.to_i)
              else
                CLI.database.update_property(number, prop, result)
              end
              log(2, "#{number} <<< #{prop}")
              log(3, result[/\A.{60}/])
            end
            pbar.inc unless opts[:quiet]
          end
          run_callback(:after_update, prop, numbers)
        end
        pbar.finish unless opts[:quiet]
      end

      def list(props, numbers)
        numbers.each do |number|
          props.each do |prop|
            if prop == 'filter'
              filters = CLI.database.filters_by_number(number)
              filters.collect! {|filter| filter[1 .. -2].join(':')}
              value = filters.join(',')
            else
              value = CLI.database.property(number, prop)
            end
            $stdout.puts("#{number}:#{prop} = #{value}")
          end
        end
      end
    end
  end
end

module MsgCab
  class Database
    def numbers_to_update_filter
      @adapter.select_all(<<'End').collect {|row| row[0].to_i}
SELECT number FROM message WHERE number NOT IN (SELECT number FROM filter)
End
    end

    def update_filter(number, folder, folder_number)
      @adapter.execute(<<'End', number)
DELETE FROM filter WHERE number = ?
End
      @adapter.execute(<<'End', number, folder, folder_number, Time.now.utc.xmlschema)
INSERT INTO filter VALUES (?, ?, ?, ?)
End
    end

    def numbers_to_update_property(name)
      @adapter.select_all(<<'End', name).collect {|row| row[0].to_i}
SELECT number FROM message WHERE number NOT IN (SELECT number FROM property WHERE name = ?)
End
    end

    def update_property(number, name, value)
      @adapter.execute(<<'End', number, name)
DELETE FROM property WHERE number = ? AND name = ?
End
      @adapter.execute(<<'End', number, name, value)
INSERT INTO property VALUES (?, ?, ?)
End
    end
  end
end
