/***************************************************************
 *   Message Passing Interface for Simula+ Object Libraries    *
 *                  declaration for mpi send                   *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.4.4	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2009,2010 COLLARD Christophe
 * copyright © 2009,2010 Centre National de la Recherche Scientifique
 * copyright © 2009,2010 Arts et Métiers ParisTech
 * copyright © 2009,2010 Laboratoire de Physique et Mécanique des Matériaux
 ***************************************************************/

/*! \namespace mpisol
    \brief Message Passing Interface for Simula+ Object Libraries
*/

/*! \defgroup send MPI send library for MOL++

    \brief MPI send library for MOL++ \n

    \htmlonly 
    <FONT color="#838383">

    send belongs to Message Passing Interface for Simula+ Object Libraries (MPISOL++) </br>
    MPISOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly 2009, 2010 Christophe COLLARD \n
	     copyright \htmlonly &#169; \endhtmlonly 2009, 2010 Centre National de la Recherche Scientifique \n
	     copyright \htmlonly &#169; 2009, 2010 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2009, 2010 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux \endhtmlonly
    \version 2.4.4
    \date 2009-2010
    \bug none
    \warning none
*/

/*@{*/

#ifndef __cplusplus
#error Must use C++ for the type send
#endif

#ifndef _send_h
#define _send_h

#if !defined (__MPI_H)
#include <mpi.h>
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__SYMMATRIX_H)
#include "MOL++/symmatrix.h"
#endif

#if !defined(__TENSOR2_H)
#include "MOL++/tensors2.h"
#endif

#if !defined(__SYMTENSOR2_H)
#include "MOL++/symtensors2.h"
#endif

#if !defined(__TENSOR3_H)
#include "MOL++/tensors3.h"
#endif

#if !defined(__TENSOR4_H)
#include "MOL++/tensors4.h"
#endif

#if !defined(__SYMTENSOR4_H)
#include "MOL++/symtensors4.h"
#endif

#if !defined(__POLYNOMS_H)
#include "MOL++/polynoms.h"
#endif

using namespace std;
using namespace mol;

namespace mpisol
{


/*!
  \ingroup send
  \brief mpi send for scalar

  \param s scalar to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)
*/

//-------------------------------------------------------------
  template <class Tf> void send (Tf& s, int to_thread, int tag)
//-------------------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  Tf* elt;
  elt = &s;
  MPI::Datatype element_type = MPI::BYTE.Create_hvector(1, sizeof(Tf), 1);
  element_type.Commit();
  MPI::COMM_WORLD.Send (elt, 1, element_type, to_thread, tag);
  element_type.Free();
}


/*!
  \ingroup send
  \brief mpi send for vector

  \param v vector to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::vector "vector"
*/

//-------------------------------------------------------------------
template <class Tf> void send (vector<Tf>& v, int to_thread, int tag)
//-------------------------------------------------------------------
{
  assert (v.dim());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  // send space dimension
  int size[1];
  size[0] = v.dim();
  MPI::COMM_WORLD.Send (size, 1, MPI::INT, to_thread, tag);

  // send vector
  MPI::Datatype vector_type = MPI::BYTE.Create_hvector(1, v.dim()*sizeof(Tf), 1); // number of bytes : v.size * sizeof(Tf)
  vector_type.Commit();
  MPI::COMM_WORLD.Send(v(), 1, vector_type, to_thread, tag);
  vector_type.Free();
}


/*!
  \ingroup send
  \brief mpi send for matrix

  \param mat matrix to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::matrix "matrix"
*/

//---------------------------------------------------------------------
template <class Tf> void send (matrix<Tf>& mat, int to_thread, int tag)
//---------------------------------------------------------------------
{
  assert (mat.Rows() && mat.Columns());

  // send space dimension
  int size[2];
  size[0] = mat.Rows();
  size[1] = mat.Columns();
  MPI::COMM_WORLD.Send (size, 2, MPI::INT, to_thread, tag);

  // send matrix
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  for (int i=1; i<=mat.Rows(); i++)
    send (mat[i], to_thread, tag+i);
}


/*!
  \ingroup send
  \brief mpi send for symmatrix

  \param smat symmetric matrix to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::symmatrix "symmatrix"
*/

//--------------------------------------------------------------------------
 template <class Tf> void send (symmatrix<Tf>& smat, int to_thread, int tag)
//--------------------------------------------------------------------------
{
  assert (smat.Rows() && smat.Columns());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  // send space dimension
  int size[2];
  size[0] = smat.Rows();
  size[1] = smat.Columns();
  MPI::COMM_WORLD.Send (size, 2, MPI::INT, to_thread, tag);

  // send symmatrix
  for (int i=1; i<=smat.Rows(); i++)
    send (smat[i], to_thread, tag-i);
}


/*!
  \ingroup send
  \brief mpi send for tensor2

  \param tsr \f$ 2^\text{nd} \f$ order tensor to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::tensor2 "tensor2"
*/

//-----------------------------------------------------------------------
 template <class Tf> void send (tensor2<Tf>& tsr, int to_thread, int tag)
//-----------------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  // send space dimension
  int size[2];
  size[0] = tsr.dim1();
  size[1] = tsr.dim2();
  MPI::COMM_WORLD.Send (size, 2, MPI::INT, to_thread, tag);

  // send tensor2
  for (int i=1; i<=tsr.dim1(); i++)
    send (tsr[i], to_thread, tag-i);
}


/*!
  \ingroup send
  \brief mpi send for symtensor2

  \param tsr symmetric \f$ 2^\text{nd} \f$ order tensor to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::symtensor2 "symtensor2"
*/

//--------------------------------------------------------------------------
 template <class Tf> void send (symtensor2<Tf>& tsr, int to_thread, int tag)
//--------------------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  // send space dimension
  int size[2];
  size[0] = tsr.dim1();
  size[1] = tsr.dim2();
  MPI::COMM_WORLD.Send (size, 2, MPI::INT, to_thread, tag);

  // send symtensor2
  for (int i=1; i<=tsr.dim1(); i++)
    send (tsr[i], to_thread, tag-i);
}


/*!
  \ingroup send
  \brief mpi send for tensor3

  \param tsr \f$ 3^\text{rd} \f$ order tensor to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::tensor3 "tensor3"
*/

//-----------------------------------------------------------------------
 template <class Tf> void send (tensor3<Tf>& tsr, int to_thread, int tag)
//-----------------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2() && tsr.dim3());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  // send space dimension
  int size[3];
  size[0] = tsr.dim1();
  size[1] = tsr.dim2();
  size[2] = tsr.dim3();
  MPI::COMM_WORLD.Send (size, 3, MPI::INT, to_thread, tag);

  // send tensor3
  for (int i=1; i<=tsr.dim1(); i++)
    send (tsr[i], to_thread, tag-i);
}


/*!
  \ingroup send
  \brief mpi send for tensor4

  \param tsr \f$ 4^\text{th} \f$ order tensor to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::tensor4 "tensor4"
*/

//-----------------------------------------------------------------------
 template <class Tf> void send (tensor4<Tf>& tsr, int to_thread, int tag)
//-----------------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2() && tsr.dim3() && tsr.dim4());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  // send space dimension
  int size[4];
  size[0] = tsr.dim1();
  size[1] = tsr.dim2();
  size[2] = tsr.dim3();
  size[3] = tsr.dim4();
  MPI::COMM_WORLD.Send (size, 4, MPI::INT, to_thread, tag);

  // send tensor4
  for (int i=1; i<=tsr.dim1(); i++)
    send (tsr[i], to_thread, tag-i);
}


/*!
  \ingroup send
  \brief mpi send for symtensor4

  \param tsr symmetric \f$ 4^\text{th} \f$ order tensor to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::symtensor4 "symtensor4"
*/

//--------------------------------------------------------------------------
 template <class Tf> void send (symtensor4<Tf>& tsr, int to_thread, int tag)
//--------------------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim3());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  // send space dimensions
  int size[2];
  size[0] = tsr.dim1();
  size[1] = tsr.dim3();
  MPI::COMM_WORLD.Send (size, 2, MPI::INT, to_thread, tag);

  // send symtensor4
  for (int i=1; i<=0.5*size[0]*(size[0]+1); i++)
    send (tsr[i], to_thread, tag-i);
}


/*!
  \ingroup send
  \brief mpi send for polynom

  \param Px polynom to send
  \param thread sending thread
  \param tag tag value (choose the same tag value to send and receive your objet)

  \sa \ref mol::polynom "polynom"
*/

//---------------------------------------------------------------------
template <class Tf> void send (polynom<Tf>& Px, int to_thread, int tag)
//---------------------------------------------------------------------
{
  int size[1];
  size[0] = Px.degree() + 1;
  assert (size[0]);

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (to_thread>=0 && to_thread<nb_threads);

  // send the number of polynom coefficients (degree + 1)
  MPI::COMM_WORLD.Send (size, 1, MPI::INT, to_thread, tag);

  // send vector
  MPI::Datatype polynom_type = MPI::BYTE.Create_hvector(1, size[0]*sizeof(Tf), 1); // number of bytes : v.size * sizeof(Tf)
  polynom_type.Commit();
  MPI::COMM_WORLD.Send(Px(), 1, polynom_type, to_thread, tag);
  polynom_type.Free();
}


}


#endif


/*@}*/
