/***************************************************************
 *   Message Passing Interface for Simula+ Object Libraries    *
 *       class broadcast : declaration for mpi broadcast       *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.4.4	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2006,2009,2010 COLLARD Christophe
 * copyright © 2006,2009,2010 Centre National de la Recherche Scientifique
 * copyright © 2006,2009,2010 Arts et Métiers ParisTech
 * copyright © 2006,2009,2010 Laboratoire de Physique et Mécanique des Matériaux
 ***************************************************************/

/*! \namespace mpisol
    \brief Message Passing Interface for Simula+ Object Libraries
*/

/*! \defgroup broadcast MPI broadcast library for MOL++

    \brief MPI broadcast library for MOL++ \n

    \htmlonly 
    <FONT color="#838383">

    broadcast belongs to Message Passing Interface for Simula+ Object Libraries (MPISOL++) </br>
    MPISOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly 2006, 2009, 2010 Christophe COLLARD \n
	     copyright \htmlonly &#169; \endhtmlonly 2006, 2009, 2010 Centre National de la Recherche Scientifique \n
	     copyright \htmlonly &#169; 2006, 2009, 2010 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2006, 2009, 2010 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux \endhtmlonly
    \version 2.4.4
    \date 2006-2010
    \bug none
    \warning none
*/

/*@{*/

#ifndef __cplusplus
#error Must use C++ for the type broadcast
#endif

#ifndef _broadcast_h
#define _broadcast_h

#if !defined (__MPI_H)
#include <mpi.h>
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__SYMMATRIX_H)
#include "MOL++/symmatrix.h"
#endif

#if !defined(__TENSOR2_H)
#include "MOL++/tensors2.h"
#endif

#if !defined(__SYMTENSOR2_H)
#include "MOL++/symtensors2.h"
#endif

#if !defined(__TENSOR3_H)
#include "MOL++/tensors3.h"
#endif

#if !defined(__TENSOR4_H)
#include "MOL++/tensors4.h"
#endif

#if !defined(__SYMTENSOR4_H)
#include "MOL++/symtensors4.h"
#endif

#if !defined(__POLYNOMS_H)
#include "MOL++/polynoms.h"
#endif

using namespace std;
using namespace mol;

namespace mpisol
{

/*!
  \ingroup broadcast
  \brief mpi broadcast for scalar

  \param s scalar to broadcast
  \param thread broadcasting thread
*/

//----------------------------------------------------
template <class Tf> void broadcast (Tf& s, int thread)
//----------------------------------------------------
{
  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  Tf* elt;
  elt = &s;
  MPI::Datatype element_type = MPI::BYTE.Create_hvector(1, sizeof(Tf), 1);
  element_type.Commit();
  MPI::COMM_WORLD.Bcast (elt, 1, element_type, thread);
  element_type.Free();
}


/*!
  \ingroup broadcast
  \brief mpi broadcast for vector

  \param v vector to broadcast
  \param thread broadcasting thread

  \sa \ref mol::vector "vector"
*/

//------------------------------------------------------------
template <class Tf> void broadcast (vector<Tf>& v, int thread)
//------------------------------------------------------------
{
  int myrank = MPI::COMM_WORLD.Get_rank();
  if (myrank == thread) assert (v.dim());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  // broadcast space dimension
  int size[1];
  if (myrank == thread)
    size[0] = v.dim();
  MPI::COMM_WORLD.Bcast (size, 1, MPI::INT, thread);
  if (!v.dim()) v = size[0]; // allocates memory
  else assert (v.dim() == size[0]);

  // broadcast vector
  MPI::Datatype vector_type = MPI::BYTE.Create_hvector(1, v.dim()*sizeof(Tf), 1); // number of bytes : v.size * sizeof(Tf)
  vector_type.Commit();
  MPI::COMM_WORLD.Bcast(v(), 1, vector_type, thread);
  vector_type.Free();
}


/*!
  \ingroup broadcast
  \brief mpi broadcast for matrix

  \param v matrix to broadcast
  \param thread broadcasting thread

  \sa \ref mol::matrix "matrix"
*/

//--------------------------------------------------------------
template <class Tf> void broadcast (matrix<Tf>& mat, int thread)
//--------------------------------------------------------------
{
  int myrank = MPI::COMM_WORLD.Get_rank();
  if (myrank == thread) assert (mat.Rows() && mat.Columns());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  // broadcast space dimension
  int size[2];
  if (myrank == thread)
    { size[0] = mat.Rows();
      size[1] = mat.Columns();
    }
  MPI::COMM_WORLD.Bcast (size, 2, MPI::INT, thread);
  if (!mat.Rows()) mat.assign (size[0], size[1]); // allocates memory
  else assert (mat.Rows() == size[0] && mat.Columns() == size[1]);

  // broadcast matrix
  for (int i=1; i<=mat.Rows(); i++)
    broadcast (mat[i], thread);
}


/*!
  \ingroup broadcast
  \brief mpi broadcast for symmatrix

  \param v vector to broadcast
  \param thread broadcasting thread

  \sa \ref mol::symmatrix "symmatrix"
*/

//------------------------------------------------------------------
template <class Tf> void broadcast (symmatrix<Tf>& smat, int thread)
//------------------------------------------------------------------
{
  int myrank = MPI::COMM_WORLD.Get_rank();
  if (myrank == thread) assert (smat.Rows() && smat.Columns());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  // broadcast space dimension
  int size[2];
  if (myrank == thread)
    { size[0] = smat.Rows();
      size[1] = smat.Columns();
    }
  MPI::COMM_WORLD.Bcast (size, 2, MPI::INT, thread);
  if (!smat.Rows()) smat.assign (size[0], size[1]); // allocates memory
  else assert (smat.Rows() == size[0] && smat.Columns() == size[1]);

  // broadcast symmatrix
  for (int i=1; i<=smat.Rows(); i++)
    broadcast (smat[i], thread);
}


/*!
  \ingroup broadcast
  \brief mpi broadcast for tensor2

  \param tsr \f$ 2^\text{nd} \f$ order tensor to broadcast
  \param thread broadcasting thread

  \sa \ref mol::tensor2 "tensor2"
*/

//---------------------------------------------------------------
template <class Tf> void broadcast (tensor2<Tf>& tsr, int thread)
//---------------------------------------------------------------
{
  int myrank = MPI::COMM_WORLD.Get_rank();
  if (myrank == thread) assert (tsr.dim1() && tsr.dim2());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  // broadcast space dimension
  int size[2];
  if (myrank == thread)
    { size[0] = tsr.dim1();
      size[1] = tsr.dim2();
    }
  MPI::COMM_WORLD.Bcast (size, 2, MPI::INT, thread);
  if (!tsr.dim1()) tsr.assign (size[0], size[1]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim2() == size[1]);

  // broadcast tensor2
  for (int i=1; i<=tsr.dim1(); i++)
    broadcast (tsr[i], thread);
}


/*!
  \ingroup broadcast
  \brief mpi broadcast for symtensor2

  \param tsr symmetrix \f$ 2^\text{nd} \f$ order tensor to broadcast
  \param thread broadcasting thread

  \sa \ref mol::symtensor2 "symtensor2"
*/

//------------------------------------------------------------------
template <class Tf> void broadcast (symtensor2<Tf>& tsr, int thread)
//------------------------------------------------------------------
{
  int myrank = MPI::COMM_WORLD.Get_rank();
  if (myrank == thread) assert (tsr.dim1() && tsr.dim2());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  // broadcast space dimension
  int size[2];
  if (myrank == thread)
    { size[0] = tsr.dim1();
      size[1] = tsr.dim2();
    }
  MPI::COMM_WORLD.Bcast (size, 2, MPI::INT, thread);
  if (!tsr.dim1()) tsr.assign (size[0], size[1]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim2() == size[1]);

  // broadcast symtensor2
  for (int i=1; i<=tsr.dim1(); i++)
    broadcast (tsr[i], thread);
}


/*!
  \ingroup broadcast
  \brief mpi broadcast for tensor3

  \param tsr \f$ 3^\text{rd} \f$ order tensor to broadcast
  \param thread broadcasting thread

  \sa \ref mol::tensor3 "tensor3"
*/

//---------------------------------------------------------------
template <class Tf> void broadcast (tensor3<Tf>& tsr, int thread)
//---------------------------------------------------------------
{
  int myrank = MPI::COMM_WORLD.Get_rank();
  if (myrank == thread) assert (tsr.dim1() && tsr.dim2() && tsr.dim3());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  // broadcast space dimension
  int size[3];
  if (myrank == thread)
    { size[0] = tsr.dim1();
      size[1] = tsr.dim2();
      size[2] = tsr.dim3();
    }
  MPI::COMM_WORLD.Bcast (size, 3, MPI::INT, thread);
  if (!tsr.dim1()) tsr.assign (size[0], size[1], size[2]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim2() == size[1] && tsr.dim3() == size[2]);

  // broadcast tensor3
  for (int i=1; i<=tsr.dim1(); i++)
    broadcast (tsr[i], thread);
}


/*!
  \ingroup broadcast
  \brief mpi broadcast for tensor4

  \param tsr \f$ 4^\text{th} \f$ order tensor to broadcast
  \param thread broadcasting thread

  \sa \ref mol::tensor4 "tensor4"
*/

//---------------------------------------------------------------
template <class Tf> void broadcast (tensor4<Tf>& tsr, int thread)
//---------------------------------------------------------------
{
  int myrank = MPI::COMM_WORLD.Get_rank();
  if (myrank == thread) assert (tsr.dim1() && tsr.dim2() && tsr.dim3() && tsr.dim4());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  // broadcast space dimension
  int size[4];
  if (myrank == thread)
    { size[0] = tsr.dim1();
      size[1] = tsr.dim2();
      size[2] = tsr.dim3();
      size[3] = tsr.dim4();
    }
  MPI::COMM_WORLD.Bcast (size, 4, MPI::INT, thread);
  if (!tsr.dim1()) tsr.assign (size[0], size[1], size[2], size[3]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim2() == size[1] && tsr.dim3() == size[2] && tsr.dim4() == size[3]);

  // broadcast tensor4
  for (int i=1; i<=tsr.dim1(); i++)
    broadcast (tsr[i], thread);
}


/*!
  \ingroup broadcast
  \brief mpi broadcast for symtensor4

  \param tsr symmetrix \f$ 4^\text{th} \f$ order tensor to broadcast
  \param thread broadcasting thread

  \sa \ref mol::symtensor4 "symtensor4"
*/

//------------------------------------------------------------------
template <class Tf> void broadcast (symtensor4<Tf>& tsr, int thread)
//------------------------------------------------------------------
{
  int myrank = MPI::COMM_WORLD.Get_rank();
  if (myrank == thread) assert (tsr.dim1() && tsr.dim3());

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  // broadcast space dimension
  int size[2];
  if (myrank == thread)
    { size[0] = tsr.dim1();
      size[1] = tsr.dim3();
    }
  MPI::COMM_WORLD.Bcast (size, 2, MPI::INT, thread);
  if (!tsr.dim1()) tsr.assign (size[0], size[0], size[1], size[1]); // allocates memory
  else assert (tsr.dim1() == size[0] && tsr.dim3() == size[1]);

  // broadcast symtensor4
  for (int i=1; i<=0.5*size[0]*(size[0]+1); i++)
    broadcast (tsr[i], thread);
}


/*!
  \ingroup broadcast
  \brief mpi broadcast for polynom

  \param Px polynom to broadcast
  \param thread broadcasting thread

  \sa \ref mol::polynom "polynom"
*/

//--------------------------------------------------------------
template <class Tf> void broadcast (polynom<Tf>& Px, int thread)
//--------------------------------------------------------------
{
  int myrank = MPI::COMM_WORLD.Get_rank();
  int size[1];
  size[0] = Px.degree() + 1;
  if (myrank == thread) assert (size);

  int nb_threads = MPI::COMM_WORLD.Get_size();
  assert (thread>=0 && thread<nb_threads);

  // broadcast the number of polynom coefficients (degree + 1)
  MPI::COMM_WORLD.Bcast (size, 1, MPI::INT, thread);
  if (Px.degree() < 0) Px = size[0] - 1; // allocates memory
  else assert (Px.degree() == size[0]-1);

  // broadcast polynom
  MPI::Datatype polynom_type = MPI::BYTE.Create_hvector(1, size[0]*sizeof(Tf), 1); // number of bytes : v.size * sizeof(Tf)
  polynom_type.Commit();
  MPI::COMM_WORLD.Bcast(Px(), 1, polynom_type, thread);
  polynom_type.Free();
}


}


#endif


/*@}*/
