/******************************************************************************
 * Copyright 2003, 2004, 2005 Alex Karev.
 *
 * This file is part of Modb library.
 * 
 * Modb library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * Modb library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Modb library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 ******************************************************************************/
/*
 *
 * Initially this was two days hack of rs485 library, 
 * with hope to make it MODBAS aware. For now it has punky console based interface.
 * Device selection for rs485 library also supported. 
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <signal.h>
#include <sys/reboot.h>

#include <linux/kd.h>

#include "rs.h"
#include "modb.h"

#define N_ELS(a) (sizeof (a) / sizeof (a)[0])

#define SET_APPLETS(a,arg,op)               \
    do {                                    \
        (a) op app         = (arg);         \
        (a) op app_items   = N_ELS(arg);    \
    } while(0)


#define CONSOLE_INPUT_BUFFER_SIZE 16

extern int print_stat;

static uint16_t get_from_console_uint16_hex( const char *prompt,...);
static unsigned int get_from_console_uint( const char *prompt,...);
static int get_from_console_yes_no( const char *prompt,...);

static int run_modbus_test(unsigned int);
static unsigned int select_device(void);
unsigned int select_port_config(int *baud, int *data, int *parity, int *stopb);

static void clrscr(void);
static void sigint(int  num);

static Port port;

typedef struct aparg {
        Port *port;
        FILE *out;
        
        struct applet *app;
        int app_items;

        int mode;
} Applet;

struct applet {
        const char *descr;
        void *(*fun)(Applet *arg);
};


static void *applet_info(Applet *arg);
static void *applet_view_help(Applet *arg);
static void *applet_poweroff(Applet *arg);
static void *applet_ascii_rhr(Applet *arg);
static void *applet_ascii_wmr(Applet *arg);
static void *applet_ascii_rsi(Applet *arg);
static void *applet_run_tests(Applet *arg);
static void *applet_view_stat(Applet *arg);
static void *applet_switch_mode(Applet *arg);

static void *applet_master_simulator(Applet *arg);
static void *applet_slave_simulator( Applet *arg);
static void *applet_doza_poll(Applet *arg);



struct applet applets[] = 
{
        {"About",                               applet_info},
        {"Help",                                applet_view_help},
        {"Read Holding Registers     [03h]",    applet_ascii_rhr},
        {"Write Multiple Registers   [10h]",    applet_ascii_wmr},
        {"Report Slave ID            [11h]",    applet_ascii_rsi},
        {"Run predefined tests",                applet_run_tests},
        {"View statistics",                     applet_view_stat},
        {"Switch transmission mode",            applet_switch_mode},
        {"Exit (poweroff)",                     applet_poweroff}
};

struct applet test_applets[] = 
{
        {"Run master write/read test",                              applet_master_simulator},
        {"Run \"DOZA\" poll test",                                  applet_doza_poll},
        {"Run slave simulator",                                     applet_slave_simulator},
        {"Return to main menu",                                     NULL}
};

void *
applet_parse_menus(Applet *arg)
{
        int 
                ret = 0, 
                i, 
                sel;

        while (1) {

                fputc('\n', stdout);
                for(i = 0; i < arg->app_items; i++) {
                        fprintf(stdout, "  %3d %s\n", i + 1, arg->app[i].descr);
                }

                sel = get_from_console_uint("\n[%s]Select action (1-%d) [1]:",
                                (arg->mode == MODB_ASCII_MODE) ? "ASCII" : "RTU", i) - 1;
                if ( (sel < 0) || (sel >= arg->app_items) )
                        sel = 0;

                if (arg->app[sel].fun != NULL) {
                        
                       if ((ret = (int) arg->app[sel].fun(arg))) {
                               fputs("Action execution failed!", stderr);
                               fprintf(stderr,"\n\tAction - %s\n\t%s\n", 
                                               arg->app[sel].descr, strerror(errno) );
                               break;
                       }

                } else {
                        break;
                }
	}

        return (NULL);
}

void *
applet_run_tests(Applet *arg)
{
        /* Update global state */
        SET_APPLETS(arg,test_applets,->);
        applet_parse_menus(arg);
        SET_APPLETS(arg,applets,->);

        return (NULL);
}

void *
applet_slave_simulator(Applet *arg)
{
    extern int exit_simulator;
    Port *p = arg->port;
    unsigned int dsta = 0;
    int ret;

    dsta = get_from_console_uint("Device address (1-247) [1]:");
    if ((dsta < 1) || (dsta > 247) ) dsta = 1;
 
    exit_simulator = 0;
    fputs("Running simulator. Press Ctrl-c to stop...\n", stdout);
    fflush(stdout);
       
    ret = modb_slave_simulator(p, (uint8_t) dsta, arg->mode );

    return ((void *) 1);
}

#define DOZA_POLL_REGS  42

void *
applet_doza_poll(Applet *arg)
{
    extern int exit_simulator;
    Port *p = arg->port;
    unsigned int 
            dsta = 0,
            quantity;
    uint16_t afrom = 0;
    int ret;

#if 0
    int mt;
    mt = get_from_console_yes_no("Poll using two serial lines simultaneous (yes/no)[no]:");
#endif

    dsta = get_from_console_uint("Device address(1-247) [1]:");   
    if ((dsta < 1) || (dsta > 247) ) dsta = 1;

    afrom = get_from_console_uint16_hex(
                    "Starting address (0000h-FFFFh) [0000h]:");   
    quantity = get_from_console_uint(
                    "Quantity of registers to test (1-%d) [42]:", 
                    DOZA_POLL_REGS);
    if ( (quantity < 1) || (quantity > MODB_MAX_REGISTERS_TO_WRITE) ) 
            quantity = DOZA_POLL_REGS;

    exit_simulator = 0;
    
    fputs("Running \"DOZA\" poll test. Press Ctrl-c to view session statistics...\n", stdout);
    fflush(stdout);
       
    ret = modb_doza_read_test(p, 
                    (uint8_t) dsta, afrom, quantity, arg->mode);

    return ((void *) 1);
}

void *
applet_master_simulator(Applet *arg)
{
    extern int exit_simulator;
    Port *p = arg->port;
    unsigned int 
            dsta1 = 0,
            dsta2 = 0,
            quantity;
    uint16_t afrom = 0;
    int ret;

    fputs("Input devices addresses range.\n", stdout);
 
    dsta1 = get_from_console_uint("start from device (1-247) [1]:");   
    if ((dsta1 < 1) || (dsta1 > 247) ) dsta1 = 1;

    if (dsta1 < 247) {
        dsta2 = get_from_console_uint("stop at device (%d-247) [%d]:", dsta1, dsta1);
        if ((dsta2 < 1) || (dsta2 > 247) ) dsta2 = 1;
        if (dsta2 < dsta1) dsta2 = dsta1;
    } else {
            dsta2 = dsta1;
    }


    afrom = get_from_console_uint16_hex(
                    "Starting address (0000h-FFFFh) [0000h]:");   
    quantity = get_from_console_uint(
                    "Quantity of registers to test (1-%d) [1]:", 
                    MODB_MAX_REGISTERS_TO_WRITE);
    if ( (quantity < 1) || (quantity > MODB_MAX_REGISTERS_TO_WRITE) ) quantity = 1;

    exit_simulator = 0;
    fputs("Running regression master test. Press Ctrl-c to stop...\n", stdout);
    fflush(stdout);
       
    ret = modb_master_test(p, 
                    (uint8_t) dsta1, (uint8_t) dsta2, 
                    afrom, quantity, arg->mode);

    return ((void *) 1);
}

void *
applet_poweroff(Applet *arg)
{
        if (!get_from_console_yes_no(
                "Are you sure, you want to exit? (yes/no) [no]:"))
              return (NULL);
       
        sync(); sleep(1); sync();

        if (reboot(RB_POWER_OFF)) {
                perror("applet_poweroff");
                exit(1);
        }

        return ( (void *) (1) );
}

void*
applet_info(Applet *arg)
{
    FILE *out;

    out =  (arg == NULL) ? stdout : arg->out;

    fprintf(out, 
                    "MODBUS serial tests. INEUM (C) %s %s\n\n", __DATE__, __TIME__);

    (void) fflush(stdout);
    return (NULL);
}

void *
applet_ascii_rhr(Applet *arg)
{
    Port *p = arg->port;
    unsigned int dsta = 0, w = 0;
    int ret, nr;
    uint16_t regs[MODB_MAX_REGISTERS_TO_READ], sb = 0;

#if 0
    puts("Running applet_ascii_rhr...");
#endif

    dsta = get_from_console_uint("Device address (1-247) [1]:");
    if ((dsta < 1) || (dsta > 247) ) dsta = 1;
    
    sb = get_from_console_uint16_hex("Starting address (0000h-FFFFh) [0000]:");
    w = get_from_console_uint(
                    "Quantity of registers to read (1-%d) [1]:", 
                    MODB_MAX_REGISTERS_TO_READ);
    if ( (w < 1) || (w > 125) ) w = 1;
  
    
    if ((ret = modb_test_fun_rhr(p, 
                                    (uint8_t) dsta, 
                                    sb, w, 
                                    regs, sizeof regs, 
                                    arg->mode)) == -1) {
        return ((void *) 1);
    } else if (ret == 0) 
            return (NULL); 

    nr = ret / 2;
    fprintf(stderr, "Query result for device %d, bytes count %d\n",
			dsta,
			ret);
	
    fprintf(stderr, "Dump for %d register%s\n", 
			nr, 
			(nr > 1) ? "s:" : ":" );
    DUMP_BUFFER_BY_WORD(regs, ret);

    return (NULL);
}

void *
applet_ascii_wmr(Applet *arg)
{
    Port *p = arg->port;
    unsigned int dsta = 0, w = 0;
    int ret, i;
    uint16_t regs[MODB_MAX_REGISTERS_TO_WRITE] = {0}, sb = 0;

#if 0
    puts("Running applet_ascii_wmr...");
#endif

    dsta = get_from_console_uint("Device address (1-247) [1]:");
    if ((dsta < 1) || (dsta > 247) ) dsta = 1;
    
    sb = get_from_console_uint16_hex("Starting address (0000h-FFFFh) [0000h]:");

    w = get_from_console_uint(
                    "Quantity of registers to write (1-%d) [1]:", 
                    MODB_MAX_REGISTERS_TO_WRITE);
    if ( (w < 1) || (w > MODB_MAX_REGISTERS_TO_WRITE) ) w = 1;
  
    fputs("Enter value for register(s) (0000h-FFFFh) [0000h]\n ADDR   VALUE\n", stdout);
    for(i = 0; i < w; i++) 
        regs[i] = get_from_console_uint16_hex("[%04Xh]:", i + sb);   
    
    if (ret = modb_test_fun_wmr(p, (uint8_t) dsta, sb, regs, w, arg->mode) == -1) 
            return ((void *) 1);
    else if (ret == 0) 
            return (NULL);


    fprintf(stderr, "Successfully written %d register%s"
		    " to device %d at address %04xh\n",
                    w, (w > 1) ? "s" : "",
                    dsta, sb );
  
    return (NULL);
}

void *
applet_ascii_rsi(Applet *arg)
{
    Port *p = arg->port;
    unsigned int dsta = 0;
    int ret;

#if 0
    puts("Running applet_ascii_rsi...");
#endif

    dsta = get_from_console_uint("Device address (1-247) [1]:");
    if ((dsta < 1) || (dsta > 247) ) dsta = 1;
    
    ret = modb_test_fun_rsi(p, (uint8_t) dsta, arg->mode );
   
    return (ret == -1) ? ((void *) 1) : (NULL);
}

void *
applet_switch_mode(Applet *arg) 
{
        arg->mode ^= 1;
        rs_switch_modbus_mode(arg->port, (arg->mode)? RS_FLAG_RTU_MODBUS : RS_FLAG_ASCII_MODBUS );
        return (NULL);
}

void *
applet_view_stat(Applet *arg)
{
        fprintf(arg->out, "Transmission mode: %s\n", 
                        (arg->mode == MODB_ASCII_MODE) ? "ASCII" : "RTU" );
        rs_print_stat(arg->port, arg->out);
        return (NULL);
}

void *
applet_view_help(Applet *arg)
{
        FILE *out = arg->out;

        
        fputc('\n',out);
        fputs(
                "       Selection of menu item can be preformed by entering it index\n"
                "  number in command prompt.\n"
                "       Default values for each command are represented in square\n"
                "  parentheses. Values ranges are printed in round parentheses.\n"
                "  To return back to main menu while running slave simulator or master\n"
                "  read/write tests press C-c.\n",
                out);
        
        return (NULL);
}



/*******************************************************************************
 *
 *      get_from_console_*      functions family
 * 
 ******************************************************************************/
unsigned int
get_from_console_uint( const char *prompt,...)
{
        char inb[ CONSOLE_INPUT_BUFFER_SIZE] = {0};
        va_list va;
       
        va_start(va,prompt);
        vfprintf(stdout, prompt,va);
        va_end(va);

        fflush(stdout);
        while (fgets(inb, sizeof inb, stdin) == NULL)
            ;

        return (unsigned int) atoi(inb);
}


int
get_from_console_yes_no( const char *prompt,...) 
{
        char inb[ CONSOLE_INPUT_BUFFER_SIZE ] = {0};
        va_list va;

        va_start(va,prompt);
        vfprintf(stdout, prompt,va);
        va_end(va);
        fflush(stdout);

        while (fgets(inb, sizeof inb, stdin) == NULL)
                ;
       
        if (inb[0] == '\n')     
                return (0);

        if (!strncasecmp(inb, "yes\n", sizeof inb)) {
                return (1);
        } else if (!strncasecmp(inb, "no\n", sizeof inb)) {
                return (0);
        }

        return (0);
}

uint16_t
get_from_console_uint16_hex( const char *prompt,...)
{
        char inb[ CONSOLE_INPUT_BUFFER_SIZE ] = {0};
        uint16_t val = 0;
        va_list va;

        va_start(va,prompt);
        vfprintf(stdout, prompt,va);
        va_end(va);
        fflush(stdout);

        while (fgets(inb, sizeof inb, stdin) == NULL)
            ;

        (void) sscanf(inb, "%04hx\n", &val);
        return val;
}



/*******************************************************************************
 *
 *      main with friends    
 *
 ******************************************************************************/
void
clrscr()
{
        fputs("\033c",stdout);
        fflush(stdout);
}

unsigned int
select_device()
{
       unsigned int id;

     
       rs_print_port_list(stdout);
       id = get_from_console_uint( "\nSome devices may not work on this platform.\n"
                               "Select serial device (1-%d) [1]:",
                       rs_device_array_size() );
       if ((id < 1) || (id > rs_device_array_size()) ) id = 1;

       return (id - 1);
}

unsigned int
select_port_config(int *baud, int *data, int *parity, int *stopb)
{
       unsigned int id;

     
       rs_print_port_predef_conf(stdout);
       fprintf(stdout, "  %2d  Input port parameters from console\n", rs_conf_array_size() + 1);

       id = get_from_console_uint( "\nSome configutations may not work on this platform.\n"
                               "Select serial device configutation (1-%d) [1]:",
                       rs_conf_array_size() + 1);
       if ((id < 1) || (id > rs_conf_array_size() + 1)) id = 1;

       /*
       if (id == rs_conf_array_size() + 1) {

       } else {
       */
               *baud    = rs_get_predef_baud(id - 1);
               *data    = rs_get_predef_data(id - 1);
               *parity  = rs_get_predef_parity(id - 1);
               *stopb   = rs_get_predef_stopb(id - 1);
               /*
       }
       */

       return (id - 1);
}


void
sigint(int  num)
{
        extern int exit_simulator;
 
#if 0
        fputs("Get signal!\n", stderr);
#endif       

        switch(num) {
            case SIGINT:
                    print_stat      = 1;
                    break;
            case SIGQUIT:
                    exit_simulator  = 1; 
                    exit(1);
                    break;
            case SIGTSTP:
                    break;
            default:
                    break;
        }
}

int
run_modbus_test(unsigned int port_num)
{
        int 
                ret,
                data,
                baud,
                parity,
                stopb;
        Applet arg;
 
#if 0
        int i;

        for (i = 0; i < NSIG;i++ ) {
           if (ioctl(STDIN_FILENO, KDSIGACCEPT, i + 1)) {
                fputs("Console not supported\n", stderr);
                return(-1);
            }
        }
#endif

#if 0
        if (ioctl(STDIN_FILENO, KDSIGACCEPT, SIGWINCH)) {
                perror("SIGWINCH");
                return(-1);
        }
        
        if (ioctl(STDIN_FILENO, KDSIGACCEPT, SIGINT)) {
                perror("SIGINT");
                return(-1);
        }
        
        if (ioctl(STDIN_FILENO, KDSIGACCEPT, SIGQUIT)) {
               perror("SIGINT");
               return(-1);
        }

	if (ioctl(STDIN_FILENO, KDSIGACCEPT, SIGTSTP)) {
               perror("SIGINT");
               return(-1);
        }
#endif

#if 1
        if (signal(SIGINT, sigint) == SIG_ERR) {
                perror("signal: SIGINT");
                return (-1);
        }

       	if (signal(SIGQUIT, sigint) == SIG_ERR) {
                perror("signal: SIGQUIT");
                return (-1);
        }
        
        if (signal(SIGTSTP, sigint) == SIG_ERR) {
                perror("signal: SIGTSTP");
                return (-1);
        }
#endif

#if 0
        sched_setcheduler
#endif
        select_port_config( &baud, &data, &parity, &stopb);
        baud    = B9600;
        data    = CS7;
        parity  = 0;
        stopb   = 2;
        if ( rs_open(&port, port_num, baud, data, parity, stopb,  RS_FLAG_ASCII_MODBUS ) == -1 ) {
		fprintf( stderr, "can't open com port %d\n", port_num );
		perror("open");
		return (-2);
	}

        arg.port    = &port;
        arg.out     = stdout;
        arg.mode    = MODB_ASCII_MODE;
 
        SET_APPLETS(arg,applets,.);
        while(1) {
            ret = (int) applet_parse_menus(&arg);
            if (ret) break;
        }

	(void) rs_close(&port);

        while(1) {
                fputs("Turn off or reboot computer now\n", stderr);
                sleep(1);
        }

        return (ret);
}



int 
main(int argc, char *argv[])
{
       clrscr();
       (void) applet_info(NULL);

       return run_modbus_test( select_device());
}

/** vim: set expandtab ts=8 sw=8: */

