/*****************************************************************************
 *
 * Copyright (C) 2005 Alex Karev.
 *
 * This file is part of KP modules library.
 * 
 ****** vim: set ts=4 sw=4 et: ***********************************************/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include <sys/stat.h>
#include <sys/types.h>

#include <unistd.h>
#include <errno.h>


#include "dbapi.h"



#define DB_CACHE_SIZE_KB 	4 * 1024  /* 4 Mb */
#define DB_FILE_MODE		0660

#define DB_DEF_DIR			"data"
#define DB_DEF_TAGDB_NAME		"tags.db"
#define DB_DEF_DSDB_NAME		"ds.db"


#define RETURN_VAL_IF_FAIL(exp,val) if (!(exp)) return (val)


static int db_open_tagdb(DB_ENV *env, DB **dbp, const char *name, int flags);
static int db_open_dsdb(DB_ENV *env, DB *dbp, DB **sdbp, const char *name, int flags);
static int getds(DB *dbp, const DBT *pkey, const DBT *pdata, DBT *skey);

/*
static int db_print_tag_record(Tag *t, DataSource *d, FILE *f);
*/

/**
 * Draw fancy dots while doing recovery ;-)
 */
static void
env_feedback(DB_ENV *env, int opcode, int pct) 
{
	/**
	 *  TODO: 
	 *  	Why not to use Berkeley DB functions instead of fputs?
	 *  	Make more intelegent use of opcode and pct variables.
	 */
	fputs(".", stderr);
}

static int
env_dir_create( const char *dir_name )
{
	struct stat ss;

	if ( 0 == stat( dir_name, &ss) ) /** It exist! */
		return (0);

	if ( mkdir(dir_name, S_IRWXU | S_IRWXG ) != 0 ) { /** mkdir failed. Somthing wrong with permissions? */
		fprintf(stderr, "mkdir: dir name %s, error: %s\n", dir_name, strerror(errno) );
		return (-1);
	}


	return (0);
}


int 
env_close(DB_ENV *env, int flags)
{
	DBMS *dbms;
	
	RETURN_VAL_IF_FAIL(env != NULL, -1);

	
	
	dbms = env->app_private;
	
	dbms->tagdb->close(dbms->tagdb, 0);
	dbms->dsdb->close(dbms->dsdb, 0);
	
	free(dbms);

	
	return env->close(env, 0);
}

int
env_open( DB_ENV **penv, const char *home_dir, int flags)
{
	DB_ENV *env;
	DBMS *dbms;
	int ret;
	char *pdir;

	/* Sanity check */
	RETURN_VAL_IF_FAIL(penv != NULL, -1);


	if (home_dir == NULL || !strlen(home_dir))
		pdir = DB_DEF_DIR;
	else 
		pdir = (char *) home_dir;


	if (env_dir_create(pdir)) /* errno is set */
		return (-1);
	
	
	if ((ret = db_env_create(&env, 0)) != 0)
		return (-1);
	

	
	/* Options setup first */
	if ( (ret = env->set_cachesize( env, 
					0, DB_CACHE_SIZE_KB * 1024, 0)) != 0)
		goto err;

	if ( (ret = env->set_feedback(env, env_feedback)) )
		goto err;

	/** Some speedup */
	if ((ret = env->set_flags(env, DB_TXN_NOSYNC, 1)))
		goto err;
	

	/* Init transaction environment */
	if ((ret = env->open(env, pdir, 
				DB_CREATE | DB_INIT_MPOOL | DB_INIT_LOCK |
				DB_INIT_LOG | DB_INIT_TXN |
			   	DB_THREAD | flags,
				DB_FILE_MODE)) != 0)
		goto err;



	/* Create database private structure */
	dbms = calloc( 1, sizeof(DBMS) );
	
	if (dbms == NULL) goto err;


	if (db_open_tagdb(env, &dbms->tagdb, DB_DEF_TAGDB_NAME, flags))
		goto err;
	
	if (db_open_dsdb(env, dbms->tagdb, &dbms->dsdb, DB_DEF_DSDB_NAME, flags))
		goto err;
	
	env->app_private = dbms;


	*penv = env;
	return (0);


err:
	env->err( env, ret, "%s", __func__);
	env->close(env, 0);
	return (-1);
}

int 
db_update_ds(DB_ENV *env, DataSource *ds, Tag *t)
{
	DBMS *dbms;
	DB *pdb, *sdb;
	DBT key, pkey, data;
	DBC *scur;
	DB_TXN *txn;
	int ret;


	/* Sanity checks */
	RETURN_VAL_IF_FAIL(env != NULL && ds != NULL && t != NULL, -1);

	dbms = env->app_private;

	if (dbms == NULL) return (-1);

	pdb = dbms->tagdb;
	sdb = dbms->dsdb;



	txn = NULL;
	if (ret = env->txn_begin(env, NULL, &txn, 0))
		goto err1;

	if (ret = sdb->cursor(sdb, txn, &scur, 0))
		goto err1;


	memset(&key,  0, sizeof key);
	memset(&pkey, 0, sizeof pkey);
	memset(&data, 0, sizeof data);

	key.data = ds;
	key.size = sizeof (DataSource);


	fprintf(stderr, "Looking up for module # %d channel # %d\n", ds->modid, ds->chid);
	ret = scur->c_pget(scur, &key, &pkey, &data, DB_SET);

	while(!ret) {
	
		fprintf(stderr, "Updating tag: %s\n", (char *) pkey.data);
		(void) memcpy(data.data, t, sizeof (Tag));
		if (ret = pdb->put(pdb, txn, &pkey, &data, 0) != 0)
			break;

		/* Next iteration. */
		ret = scur->c_pget(scur, &key, &pkey, &data, DB_NEXT_DUP);
	}


	
	if (ret != DB_NOTFOUND)
		goto err2;

	scur->c_close(scur);
	if (txn->commit(txn, 0))
		goto err3;

	return (0);


err2:	if (scur != NULL)
		(void) scur->c_close(scur);
	fputs("err2\n", stderr);
err1:	if (txn != NULL)
		(void) txn->abort(txn);
	fputs("err1\n", stderr);
	
err3:	(void) env->err(env, ret, "%s", __func__);
	fputs("err3\n", stderr);
	return (-1);
}


int 
db_create_tag_rec(DB_ENV *env, const char *tagname, int type, DataSource *ds)
{
	DBMS *dbms;
	DB *pdb;
	DBT key, data;
	int res;

	uint8_t record[ sizeof (Tag) + sizeof (DataSource) ];
	Tag t = { .type = type };


	/* Sanity checks */
	RETURN_VAL_IF_FAIL(env != NULL && ds != NULL && tagname != NULL, -1);

	dbms = env->app_private;

	if (dbms == NULL) return (-1);

	pdb = dbms->tagdb;

	memset(&key,  0, sizeof key);
	memset(&data, 0, sizeof data);
	
	
	key.data = (char *) tagname;
	key.size = strlen(tagname) + 1;

	
	data.data = record;
	data.size = sizeof record;

	
	memcpy(record, &t, sizeof t);
	memcpy(record + sizeof t, ds, sizeof (DataSource) );


	if (res = pdb->put(pdb, NULL, &key, &data, DB_AUTO_COMMIT)) {
		(void) env->err(env, res, "%s", __func__);
		return (-1);
	}
		
	return (0);
}


int 
db_update_tag_rec(DB_ENV *env, const char *tagname, Tag *t, DataSource *ds)
{
	DBMS *dbms;
	DB *pdb;
	DBT key, data;
	int res;

	uint8_t record[ sizeof (Tag) + sizeof (DataSource) ];


	/* Sanity checks */
	RETURN_VAL_IF_FAIL(env != NULL && t != NULL && ds != NULL && tagname != NULL, -1);

	dbms = env->app_private;

	if (dbms == NULL) return (-1);

	pdb = dbms->tagdb;

	memset(&key,  0, sizeof key);
	memset(&data, 0, sizeof data);


	key.data = (char *) tagname;
	key.size = strlen(tagname) + 1;


	data.data = record;
	data.size = sizeof record;
	data.flags = DB_DBT_USERMEM;

	
	memcpy(record, t, sizeof (Tag));
	memcpy(record + sizeof (Tag), ds, sizeof (DataSource) );


	if (res = pdb->put(pdb, NULL, &key, &data, DB_AUTO_COMMIT)) {
		(void) env->err(env, res, "%s", __func__);
		return (-1);
	}
		
	return (0);
}

int 
db_select_tag_rec(DB_ENV *env, const char *tagname, Tag *t, DataSource *ds)
{
	DBMS *dbms;
	DB *pdb;
	DBT key, data;
	int res;

	uint8_t record[ sizeof (Tag) + sizeof (DataSource) ];


	/* Sanity checks */
	RETURN_VAL_IF_FAIL(env != NULL && t != NULL && ds != NULL && tagname != NULL, -1);

	dbms = env->app_private;

	if (dbms == NULL) return (-1);

	pdb = dbms->tagdb;

	memset(&key,  0, sizeof key);
	memset(&data, 0, sizeof data);


	key.data = (char *) tagname;
	key.size = strlen(tagname) + 1;


	data.data = record;
	data.size = sizeof record;
	data.flags = DB_DBT_USERMEM;

	
	if (res = pdb->get(pdb, NULL, &key, &data, DB_AUTO_COMMIT)) {
		(void) env->err(env, res, "%s", __func__);
		return (-1);
	}
		
	
	(void) memcpy(t, record, sizeof (Tag));
	(void) memcpy(ds, record + sizeof (Tag), sizeof (DataSource) );
	
	return (0);
}

/***** 	Static interfaces  ***************************************************/



int
db_open_tagdb(DB_ENV *env, DB **dbp, const char *name, int flags)
{
	int ret; 
	char *pname;
	DB *db;


	RETURN_VAL_IF_FAIL(env != NULL && dbp != NULL, -1);



	if (ret = db_create(&db, env, 0))
		goto err;


		
	pname = (char *) ((name == NULL) ? DB_DEF_TAGDB_NAME : name);
	
	if (ret = db->open(db, NULL, pname, NULL, 
				DB_BTREE, 
				flags | DB_CREATE | DB_THREAD | DB_AUTO_COMMIT, 
				DB_FILE_MODE)) {

		db->close(db, 0);
		goto err;
	}


	*dbp = db;
	return (0);

err:
	env->err(env, ret, "%s", __func__);
	return (-1);
}


int
db_open_dsdb(DB_ENV *env, DB *dbp, DB **sdbp, const char *name, int flags)
{
	int ret; 
	char *pname;
	DB *db;


	RETURN_VAL_IF_FAIL(env != NULL && dbp != NULL, -1);



	if (ret = db_create(&db, env, 0))
		goto err;


	if ((ret = db->set_flags(db, DB_DUP | DB_DUPSORT)))
		goto err;

		
	pname = (char *) ((name == NULL) ? DB_DEF_DSDB_NAME : name);
	
	if (ret = db->open(db, NULL, pname, NULL, 
				DB_BTREE, 
				flags | DB_CREATE | DB_THREAD | DB_AUTO_COMMIT, 
				DB_FILE_MODE)) {
		db->close(db, 0);
		goto err;

	}

	
	/* Associate the secondary with the primary. */
	if ((ret = db->associate(dbp, NULL, db, getds, DB_CREATE | DB_AUTO_COMMIT))) {
		db->close(db, 0);
		goto err;
	}

	
	*sdbp = db;
	return (0);

err:
	env->err(env, ret, "%s", __func__);
	return (-1);
}

/*
 * getname -- extracts a secondary key (the last name) from a primary
 * 	key/data pair
 */
static int
getds(dbp, pkey, pdata, skey)
	DB *dbp;
	const DBT *pkey, *pdata;
	DBT *skey;
{
	/*
	 * Since the secondary key is a simple structure member of the
	 * record, we don't have to do anything fancy to return it.  If
	 * we have composite keys that need to be constructed from the
	 * record, rather than simply pointing into it, then the user's
	 * function might need to allocate space and copy data.  In
	 * this case, the DB_DBT_APPMALLOC flag should be set in the
	 * secondary key DBT.
	 */
	memset(skey, 0, sizeof(DBT));

	skey->data = ((unsigned char *) pdata->data) + sizeof (Tag);
	skey->size = sizeof(DataSource);

	return (0);
}

#if 0
int
db_print_tag_record(Tag *t, DataSource *d, FILE *f)
{
	
	fprintf(f, "DataSource module: %d, channel: %d\n", d->modid, d->chid );

	return (-1);
}
#endif


int 
db_print_ds_dep(DB_ENV *env, FILE *ostrem)
{
	return (-1);
}


int 
db_get_tag_val(DB_ENV *env, const char *tagname, Tag *t, int flags)
{
	DBMS *dbms;
	DB *pdb;
	DBT key, data;
	int res;

	uint8_t record[ sizeof (Tag) + sizeof (DataSource) ] = {0};


	/* Sanity checks */
	RETURN_VAL_IF_FAIL(env != NULL && t != NULL && tagname != NULL, -1);

	dbms = env->app_private;

	if (dbms == NULL) return (-1);

	pdb = dbms->tagdb;

	memset(&key,  0, sizeof key);
	memset(&data, 0, sizeof data);
	
	
	key.data = (char *) tagname;
	key.size = strlen(tagname) + 1;

	
	data.data 	= record;
	data.size 	= sizeof record;
	data.flags	= DB_DBT_USERMEM;


	if (res = pdb->get(pdb, NULL, &key, &data, DB_AUTO_COMMIT)) {
		(void) env->err(env, res, "%s", __func__);
		return (-1);
	}
		
	(void) memcpy(t, data.data, sizeof (Tag));
	
	return (0);
}

int 
db_set_tag_val(DB_ENV *env, const char *tagname, Tag *t, int flags)
{
	DBMS *dbms;
	DB *pdb;
	DBT key, data;
	int res;


	uint8_t record[ sizeof (Tag) + sizeof (DataSource) ] = {0};


	/* Sanity checks */
	RETURN_VAL_IF_FAIL(env != NULL && t != NULL && tagname != NULL, -1);

	dbms = env->app_private;

	if (dbms == NULL) return (-1);

	pdb = dbms->tagdb;


	(void) memset(&key,  0, sizeof key);
	(void) memset(&data, 0, sizeof data);
	
	
	key.data = (char *) tagname;
	key.size = strlen(tagname) + 1;

	
	data.data 	= record;
	data.size 	= sizeof record;
	data.flags	= DB_DBT_USERMEM;


	if (res = pdb->get(pdb, NULL, &key, &data, DB_AUTO_COMMIT)) 
		goto err;
		
	
	(void) memcpy(data.data, t, sizeof (Tag) );

	if (res = pdb->put(pdb, NULL, &key, &data, DB_AUTO_COMMIT))
		goto err;

	
	return (0);
err:
	(void) env->err(env, res, "%s", __func__);
	return (-1);
}


