// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \file list.cc C++ API <i>mobius.pod.list</i> class wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <pymobius.h>
#include <pyobject.h>
#include "list.h"
#include <mobius/exception.inc>
#include <stdexcept>
#include "data.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create mobius.pod.list value from Python object
//! \param py_value Python object
//! \return mobius.pod.list value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::pod::list
pymobius_pod_list_from_pyobject (PyObject *py_value)
{
  mobius::pod::list l;

  if (PyList_Check (py_value))
    {
      Py_ssize_t siz = PyList_Size (py_value);

      for (Py_ssize_t i = 0;i < siz;i++)
        {
          PyObject *item = PyList_GetItem (py_value, i);
          if (!item)
            throw std::runtime_error (mobius::MOBIUS_EXCEPTION_MSG ("could not get item"));

          l.append (pymobius_pod_data_from_pyobject (item));
        }
    }

  else if (PyTuple_Check (py_value))
    {
      Py_ssize_t siz = PyTuple_Size (py_value);

      for (Py_ssize_t i = 0;i < siz;i++)
        {
          PyObject *item = PyTuple_GetItem (py_value, i);
          if (!item)
            throw std::runtime_error (mobius::MOBIUS_EXCEPTION_MSG ("could not get item"));
          l.append (pymobius_pod_data_from_pyobject (item));
        }
    }

  else if (PySet_Check (py_value))
    {
      mobius::py::pyobject iter = PyObject_GetIter (py_value);
      if (!iter)
        throw std::invalid_argument (mobius::MOBIUS_EXCEPTION_MSG ("could not iterate through set"));
      
      mobius::py::pyobject item = PyIter_Next (iter);
      while (item)
        {
          l.append (pymobius_pod_data_from_pyobject (item));
          item = PyIter_Next (iter);
        }
    }

  else
    throw std::invalid_argument (mobius::MOBIUS_EXCEPTION_MSG ("could not convert value to mobius.pod.list"));
  
  return l;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create Python object from mobius.pod.list value
//! \param value mobius.pod.list value
//! \return New Python object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
pymobius_pod_list_to_pyobject (const mobius::pod::list& value)
{
  return mobius::py::pylist_from_cpp_container (
           value,
           pymobius_pod_data_to_pyobject
         );
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create pure Python object from mobius.pod.list value
//! \param value mobius.pod.list value
//! \return New Python object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
pymobius_pod_list_to_python (const mobius::pod::list& value)
{
  return mobius::py::pylist_from_cpp_container (
           value,
           pymobius_pod_data_to_python
         );
}
