# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2018 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius.forensics.vfs
import pymobius.forensics.registry.installed_programs
from pymobius.forensics.registry import *
import parser_library_dat
import parser_profile_xml
import parser_sd_files
import parser_searches_dat

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# All Date/Times are stored in Coordinated Universal Time (UTC).
# @see https://msdn.microsoft.com/pt-br/library/windows/desktop/ms724397(v=vs.85).aspx
#
# Version examined: Shareaza 2.7.10.2
#
# Shareaza main files:
#   1. Searches.dat - Stores search history and search results
#   2. Library1.dat and Library2.dat - Stores configuration for local folders and files
#   3. *.sd files - Download control files (one per downloading file)
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generic dataholder
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class dataholder (object):
  pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief iterate through Shareaza AppData folders
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def iter_shareaza_folders (vfs):
  for root in vfs:
    users_folder = root.get_child_by_name ('Users', False)

    if not users_folder:
      users_folder = root.get_child_by_name ('Documents and Settings', False)

    if users_folder:
      for entry in users_folder.get_children ():
        if entry.is_folder ():
          username = entry.name
          shareaza_folder = entry.get_child_by_path ('AppData/Roaming/Shareaza', False)

          if shareaza_folder:
            yield username, entry, shareaza_folder

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief forensics: P2P Shareaza
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Ant (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, item):
    self.__item = item
    self.__data = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_data (self):
    if not self.__data:
      self.__retrieve_data ()

    return self.__data

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief retrieve data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_data (self):

    # create model
    self.__data = dataholder ()
    self.__data.application = dataholder ()
    self.__data.application.id = 'shareaza'
    self.__data.application.name = 'Shareaza'
    self.__data.application.versions = set ()

    self.__data.accounts = []
    self.__data.searches = []
    self.__data.local_files = []
    self.__data.remote_files = []

    # retrieve data
    self.__retrieve_registry_data ()
    self.__retrieve_app_data ()
    self.__normalize_data ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief retrieve registry data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_registry_data (self):
    ant = pymobius.forensics.registry.main.Ant (self.__item)

    for registry in ant.get_data ():
      for program in pymobius.forensics.registry.installed_programs.get (registry):
        if program.display_name.startswith ('Shareaza '):
          self.__data.application.versions.add (program.version)

    # Shareaza folders
    #for username, key in pymobius.forensics.registry.iter_hkey_users (registry):
    #  shareaza_key = key.get_key_by_path ('Software\\Shareaza\\Shareaza')

    #  if shareaza_key:
    #    user_path = get_data_as_string (shareaza_key.get_data_by_name ('UserPath'))
    #    collection_path = get_data_as_string (shareaza_key.get_data_by_path ('Downloads\\CollectionPath'))
    #    complete_path = get_data_as_string (shareaza_key.get_data_by_path ('Downloads\\CompletePath'))
    #    incomplete_path = get_data_as_string (shareaza_key.get_data_by_path ('Downloads\\IncompletePath'))
    #    torrent_path = get_data_as_string (shareaza_key.get_data_by_path ('Downloads\\TorrentPath'))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief retrieve data from disk files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_app_data (self):

    # get item VFS
    vfs = pymobius.forensics.vfs.get_item_vfs (self.__item)
    if not vfs:
      raise Exception, "Datasource is not available"

    # scan VFS root entries
    for username, user_folder, shareaza_folder in iter_shareaza_folders (vfs):
      data_folder = shareaza_folder.get_child_by_name ('Data', False)
      incomplete_folder = user_folder.get_child_by_path ('AppData/Local/Shareaza/Incomplete', False)

      if data_folder:
        parser_searches_dat.retrieve (self.__item, self.__data, data_folder, username)
        parser_library_dat.retrieve (self.__item, self.__data, data_folder, username)
        parser_profile_xml.retrieve (self.__item, self.__data, data_folder, username)

      if incomplete_folder:
        parser_sd_files.retrieve (self.__item, self.__data, incomplete_folder, username)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief normalize retrieved data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __normalize_data (self):

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # sort and remove duplicated remote files
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    remote_files = [ (f.timestamp, f) for f in self.__data.remote_files ]
    timestamp, peer_ip, size, name = None, None, None, None
    self.__data.remote_files = []

    for timestamp, f in sorted (remote_files):
      if (timestamp, peer_ip, size, name) != (f.timestamp, f.peer.ip, f.size, f.name):
        self.__data.remote_files.append (f)
        timestamp, peer_ip, size, name = f.timestamp, f.peer.ip, f.size, f.name
