// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "data_encoder.h"

namespace mobius
{
namespace encoder
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param writer writer object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
data_encoder::data_encoder (mobius::io::writer writer)
  : writer_ (writer)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief encode uint8
//! \param value uint8_t value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_encoder::encode_uint8 (std::uint8_t value)
{
  writer_.write ({value});
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief encode uint16_t little endian
//! \param value uint16_t value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_encoder::encode_uint16_le (std::uint16_t value)
{
  writer_.write ({std::uint8_t (value),
                  std::uint8_t (value >> 8)
                 });
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief encode uint32_t little endian
//! \param value uint32_t value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_encoder::encode_uint32_le (std::uint32_t value)
{
  writer_.write ({std::uint8_t (value),
                  std::uint8_t (value >> 8),
                  std::uint8_t (value >> 16),
                  std::uint8_t (value >> 24)
                 });
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief encode uint64_t little endian
//! \param value uint64_t value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_encoder::encode_uint64_le (std::uint64_t value)
{
  writer_.write ({std::uint8_t (value),
                  std::uint8_t (value >> 8),
                  std::uint8_t (value >> 16),
                  std::uint8_t (value >> 24),
                  std::uint8_t (value >> 32),
                  std::uint8_t (value >> 40),
                  std::uint8_t (value >> 48),
                  std::uint8_t (value >> 56)
                 });
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief encode string by size
//! \param value string
//! \param size size in bytes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_encoder::encode_string_by_size (const std::string& value, std::size_t size)
{
  if (value.length () <= size)
    {
      writer_.write (value);

      mobius::bytearray pad (size - value.length ());
      pad.fill (0);

      writer_.write (pad);
    }

  else
    writer_.write (value.substr (0, size));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief encode string by size
//! \param value string
//! \param size size in bytes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_encoder::encode_bytearray (const mobius::bytearray& data)
{
  writer_.write (data);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief write siz bytes to output
//! \param siz size in bytes
//! \param value value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_encoder::fill (std::size_t siz, std::uint8_t value)
{
  constexpr std::size_t BLOCK_SIZE = 32768;

  if (siz >= BLOCK_SIZE)
    {
      mobius::bytearray data (BLOCK_SIZE);
      data.fill (value);

      while (siz >= BLOCK_SIZE)
        {
          writer_.write (data);
          siz -= BLOCK_SIZE;
        }
    }

  if (siz > 0)
    {
      mobius::bytearray data (siz);
      data.fill (value);
      writer_.write (data);
    }
}

} // namespace encoder
} // namespace mobius
