# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius.app.chromium
import pymobius.app.gecko
import pymobius.app.internet_explorer
import mobius

ANT_ID = 'bookmarked-urls'
ANT_NAME = 'Bookmarked URLs'
ANT_VERSION = '1.0'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generic bookmark class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Bookmark (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Ant: URL Bookmarks
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Ant (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, item):
    self.id = ANT_ID
    self.name = ANT_NAME
    self.version = ANT_VERSION
    self.__item = item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Run ant
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def run (self):
    if not self.__item.datasource:
      return

    self.__entries = []

    self.__retrieve_chromium ()
    self.__retrieve_gecko ()
    self.__retrieve_internet_explorer ()

    self.__save_data ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Chromium based browsers
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_chromium (self):
    model = pymobius.app.chromium.model (self.__item)
    
    for profile in model.get_profiles ():
      for entry in profile.get_bookmarks ():
        bookmark = Bookmark ()
        bookmark.url = entry.url
        bookmark.name = entry.name
        bookmark.creation_time = entry.creation_time
        bookmark.folder = entry.folder
        bookmark.username = entry.username
        bookmark.profile = profile
        
        bookmark.metadata = mobius.pod.map ()
        bookmark.metadata.set ('source', entry.path)
        bookmark.metadata.set ('id', entry.id)
        bookmark.metadata.set ('last_visited_time', entry.last_visited_time)
        
        self.__entries.append (bookmark)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Gecko based browsers
  # @see http://doxygen.db48x.net/mozilla/html/interfacensIDownloadManager.html
  # @todo Move Unicode convertion to profile class
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_gecko (self):
    model = pymobius.app.gecko.model (self.__item)
    
    for profile in model.get_profiles ():
      for entry in profile.get_bookmarks ():
        bookmark = Bookmark ()
        bookmark.url = entry.url
        bookmark.name = entry.name
        bookmark.creation_time = entry.creation_time
        bookmark.folder = entry.folder
        bookmark.username = entry.username
        bookmark.profile = profile
        
        bookmark.metadata = mobius.pod.map ()
        bookmark.metadata.set ('source', entry.path)
        bookmark.metadata.set ('id', entry.id)
        bookmark.metadata.set ('last_modification_time', entry.last_modification_time)
        
        self.__entries.append (bookmark)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Internet Explorer browser
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_internet_explorer (self):
    model = pymobius.app.internet_explorer.model (self.__item)
    
    for profile in model.get_profiles ():
      for entry in profile.get_favorites ():
        bookmark = Bookmark ()
        bookmark.url = entry.url
        bookmark.name = entry.name
        bookmark.creation_time = entry.creation_time
        bookmark.folder = entry.folder
        bookmark.username = profile.username
        bookmark.profile = profile
        
        bookmark.metadata = mobius.pod.map ()
        bookmark.metadata.set ('source', entry.path)
        bookmark.metadata.set ('last_modification_time', entry.last_modification_time)
        
        self.__entries.append (bookmark)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save data into model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __save_data (self):
    case = self.__item.case
    transaction = case.new_transaction ()

    # remove old data
    self.__item.remove_bookmarked_urls ()

    # save data
    profiles = {}

    for b in self.__entries:
      bookmark = self.__item.new_bookmarked_url (b.url)
      
      if b.name:
        bookmark.name = b.name
      
      if b.creation_time:
        bookmark.creation_time = b.creation_time

      if b.folder:
        bookmark.folder = b.folder
              
      if b.username:
        bookmark.username = b.username

      bookmark.metadata = b.metadata

      # create profile, if necessary
      p = profiles.get (id (b.profile))

      if not p:
        app = case.new_application (b.profile.app_id, b.profile.app_name)

        p = self.__item.new_profile (b.profile.app_id, b.profile.path)
        p.id = b.profile.name
        p.username = b.profile.username
        
        if b.profile.creation_time:
          p.creation_time = b.profile.creation_time
        profiles[id (b.profile)] = p

      bookmark.profile = p

    # set ant run
    self.__item.set_ant (ANT_ID, ANT_NAME, ANT_VERSION)
    transaction.commit ()
