// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "item.h"
#include "ant.h"
#include "application.h"
#include "case.h"
#include "cookie.h"
#include "password.h"
#include "password_hash.h"
#include "profile.h"
#include <mobius/exception.inc>
#include <mobius/core/category_manager.h>
#include <stdexcept>
#include <cstddef>

namespace mobius
{
namespace model
{
namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Expand value mask, using item attributes
//! \param value_mask value mask
//! \param item case item
//! \return expanded string value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
expand_value_mask (const std::string& value_mask, mobius::model::item item)
{
  std::size_t len = value_mask.length ();
  std::size_t pos = 0;
  std::string out;

  while (pos < len)
    {
      auto start_pos = value_mask.find ("${", pos);

      if (start_pos == std::string::npos)
        {
          out += value_mask.substr (pos);
          pos = len;
        }

      else
        {
          out += value_mask.substr (pos, start_pos - pos);
          pos = start_pos;

          auto end_pos = value_mask.find ("}", start_pos + 2);

          if (end_pos == std::string::npos)
            {
              out += value_mask.substr (pos);
              pos = len;
            }

          else
            {
              std::string var = value_mask.substr (start_pos + 2, end_pos - start_pos - 2);
              out += item.get_attribute (var);
              pos = end_pos + 1;
            }
        }
    }

  return out;
}

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief item implementation class
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class item::impl
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Constructors
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  impl (const impl&) = delete;
  impl (impl&&) = delete;
  explicit impl (const mobius::model::Case&);
  impl (const mobius::model::Case&, uid_type);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Operators
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  impl& operator= (const impl&) = delete;
  impl& operator= (impl&&) = delete;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // function prototypes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  item new_child (const std::string&, int);
  void remove ();
  void move (int, const item&);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get uid
  //! \return uid
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  uid_type
  get_uid () const
  {
    return uid_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get category
  //! \return category
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::string
  get_category () const
  {
    _load_data ();
    return category_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get case
  //! \return case
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  Case
  get_case () const
  {
    return case_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get database
  //! \return database
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::database::database
  get_database () const
  {
    return case_.get_database ();
  }

private:
  //! \brief Case object
  Case case_;

  //! \brief Unique ID
  uid_type uid_ = -1;

  //! \brief Category
  mutable std::string category_;

  //! \brief Data loaded flag
  mutable bool data_loaded_ = false;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Helper functions
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  void _load_data () const;
  int _reserve_index (int) const;
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param c case object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item::impl::impl (const mobius::model::Case& c)
  : case_ (c)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param c case object
//! \param uid Unique ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item::impl::impl (const mobius::model::Case& c, uid_type uid)
  : case_ (c), uid_ (uid)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new child item
//! \param category item's category
//! \param idx child position, starting in 1 or -1 for last position
//! \return new item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item
item::impl::new_child (const std::string& category, int idx)
{
  if (uid_ == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = case_.get_database ();

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // get item index
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  idx = _reserve_index (idx);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // create item
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto stmt = db.new_statement (
                "INSERT INTO item "
                     "VALUES (NULL, ?, ?, ?, DATETIME ('NOW'))");

  stmt.bind (1, uid_);
  stmt.bind (2, idx);
  stmt.bind (3, category);
  stmt.execute ();

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // return item
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto uid = db.get_last_insert_row_id ();
  return item (case_, uid);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::impl::remove ()
{
  if (uid_ == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  if (uid_ == 1)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Cannot remove root item"));

  auto db = case_.get_database ();

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // get item index
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto stmt = db.new_statement (
                "SELECT idx, parent_uid "
                "FROM item "
                "WHERE uid = ?");

  stmt.bind (1, uid_);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // if item exists, delete it
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  if (stmt.fetch_row ())
    {
      auto idx = stmt.get_column_int (0);
      auto parent_uid = stmt.get_column_int64 (1);

      // delete item
      auto stmt = db.new_statement (
                    "DELETE FROM item "
                          "WHERE uid = ?");

      stmt.bind (1, uid_);
      stmt.execute ();

      // update idx for remaining items
      stmt = db.new_statement (
               "UPDATE item "
                  "SET idx = idx - 1 "
                "WHERE parent_uid = ? "
                  "AND idx > ?");

      stmt.bind (1, parent_uid);
      stmt.bind (2, idx);
      stmt.execute ();
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // reset attributes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  uid_ = -1;
  category_.clear ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Move item
//! \param idx position before which the item will be inserted
//! \param parent parent item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::impl::move (int idx, const item& parent)
{
  if (uid_ == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  if (uid_ == 1)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Cannot move root item"));

  if (!parent)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("New parent cannot be null"));

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // get current idx and parent
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto db = case_.get_database ();

  auto stmt = db.new_statement (
                "SELECT idx, parent_uid "
                "FROM item "
                "WHERE uid = ?");

  stmt.bind (1, uid_);

  if (!stmt.fetch_row ())
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item not found"));

  auto old_idx = stmt.get_column_int (0);
  auto old_parent_uid = stmt.get_column_int64 (1);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // open slot into new parent for item
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  idx = parent.impl_->_reserve_index (idx);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // set item's idx and parent
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  stmt = db.new_statement (
           "UPDATE item "
              "SET parent_uid = ?, "
                  "idx = ? "
            "WHERE uid = ?");

  stmt.bind (1, parent.get_uid ());
  stmt.bind (2, idx);
  stmt.bind (3, uid_);
  stmt.execute ();

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // close slot on old parent
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  stmt = db.new_statement (
           "UPDATE item "
              "SET idx = idx - 1 "
            "WHERE parent_uid = ? "
              "AND idx > ?");

  stmt.bind (1, old_parent_uid);
  stmt.bind (2, old_idx);
  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Load data on demand
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::impl::_load_data () const
{
  if (data_loaded_)
    return;

  // load data
  auto db = case_.get_database ();

  auto stmt = db.new_statement (
                "SELECT category "
                  "FROM item "
                 "WHERE uid = ?");

  stmt.bind (1, uid_);

  if (stmt.fetch_row ())
    category_ = stmt.get_column_string (0);

  data_loaded_ = true;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Reserve one slot for a child at position idx
//! \param idx index, starting in 1 or -1 for last position
//! \return index reserved
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int
item::impl::_reserve_index (int idx) const
{
  if (uid_ == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Get last idx
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto db = case_.get_database ();

  auto stmt = db.new_statement (
                "SELECT MAX (idx) "
                  "FROM item "
                 "WHERE parent_uid = ?");

  stmt.bind (1, uid_);

  int max_idx = 0;

  if (stmt.fetch_row ())
    max_idx = stmt.get_column_int (0);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Reserve slot, if necessary
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  if (idx == -1)
    idx = max_idx + 1;

  else if (idx < 1 || idx > max_idx + 1)
    throw std::out_of_range (MOBIUS_EXCEPTION_MSG ("Index out of range"));

  else
    {
      auto stmt = db.new_statement (
                    "UPDATE item "
                       "SET idx = idx + 1 "
                     "WHERE parent_uid = ? "
                       "AND idx >= ?");

      stmt.bind (1, uid_);
      stmt.bind (2, idx);
      stmt.execute ();
    }

  return idx;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param c case object
//! \param uid Unique ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item::item (const mobius::model::Case& c, uid_type uid)
  : impl_ (std::make_shared <impl> (c, uid))
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get uid
//! \return uid
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item::uid_type
item::get_uid () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  return impl_->get_uid ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get category
//! \return category
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
item::get_category () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  return impl_->get_category ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get case
//! \return case
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
Case
item::get_case () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  return impl_->get_case ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get number of children
//! \return Number of children
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int
item::get_child_count () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // run query
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT count (*) "
                  "FROM item "
                 "WHERE parent_uid = ?");

  stmt.bind (1, impl_->get_uid ());

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // get result
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  int count = 0;

  if (stmt.fetch_row ())
    count = stmt.get_column_int (0);

  return count;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get children items
//! \return children
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <item>
item::get_children () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // run query
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                  "FROM item "
                 "WHERE parent_uid = ? "
              "ORDER BY idx");

  stmt.bind (1, impl_->get_uid ());

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // fill vector
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto c = impl_->get_case ();
  std::vector <item> items;

  while (stmt.fetch_row ())
    {
      auto uid = stmt.get_column_int64 (0);
      items.emplace_back (c, uid);
    }

  return items;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get parent item
//! \return parent, if exists
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item
item::get_parent () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // get parent_uid
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT parent_uid "
                  "FROM item "
                 "WHERE uid = ?");

  stmt.bind (1, impl_->get_uid ());

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // if item exists, get parent
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  item parent;

  if (stmt.fetch_row ())
    {
      if (!stmt.is_column_null (0))
        {
          auto c = impl_->get_case ();
          parent = c.get_item_by_uid (stmt.get_column_int64 (0));
        }
    }

  return parent;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new child item
//! \param category item's category
//! \param idx child position, starting in 1 or -1 for last position
//! \return new item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item
item::new_child (const std::string& category, int idx)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  return impl_->new_child (category, idx);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::remove ()
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  impl_->remove ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Move item
//! \param idx new index
//! \param parent parent item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::move (int idx, const item& parent)
{
  impl_->move (idx, parent);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check if attribute exists
//! \param id attribute ID
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
item::has_attribute (const std::string& id) const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT * "
                  "FROM attribute "
                 "WHERE item_uid = ? "
                   "AND id = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, id);

  return bool (stmt.fetch_row ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get attribute value
//! \param id attribute ID
//! \return attribute value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
item::get_attribute (const std::string& id) const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT value "
                  "FROM attribute "
                 "WHERE item_uid = ? "
                   "AND id = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, id);

  std::string value;

  if (stmt.fetch_row ())
    value = stmt.get_column_string (0);

  return value;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set attribute value
//! \param id attribute ID
//! \param value attribute value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::set_attribute (const std::string& id, const std::string& value)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();
  mobius::database::statement stmt;

  if (has_attribute (id))
    {
      stmt = db.new_statement (
               "UPDATE attribute "
                  "SET value = ? "
                "WHERE item_uid = ? "
                  "AND id = ?");

      stmt.bind (1, value);
      stmt.bind (2, impl_->get_uid ());
      stmt.bind (3, id);
    }

  else
    {
      stmt = db.new_statement (
               "INSERT INTO attribute "
                    "VALUES (NULL, ?, ?, ?)");

      stmt.bind (1, impl_->get_uid ());
      stmt.bind (2, id);
      stmt.bind (3, value);
    }

  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove attribute
//! \param id attribute ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::remove_attribute (const std::string& id)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "DELETE FROM attribute "
                      "WHERE item_uid = ? "
                        "AND id = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, id);
  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get attributes
//! \return map containing attributes' IDs and values
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::map <std::string, std::string>
item::get_attributes () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT id, value "
                  "FROM attribute "
                 "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());

  std::map <std::string, std::string> attributes;

  while (stmt.fetch_row ())
    {
      auto id = stmt.get_column_string (0);
      auto value = stmt.get_column_string (1);
      attributes[id] = value;
    }

  return attributes;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check if ANT has been executed
//! \param id ANT ID
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
item::has_ant (const std::string& id) const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT * "
                  "FROM ant "
                 "WHERE item_uid = ? "
                   "AND id = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, id);

  return bool (stmt.fetch_row ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set ANT
//! \param id ANT ID
//! \param name ANT name
//! \param version ANT version
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::set_ant (
  const std::string& id,
  const std::string& name,
  const std::string& version
)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();
  mobius::database::statement stmt;

  if (has_ant (id))
    {
      stmt = db.new_statement (
               "UPDATE ant "
                  "SET name = ?, "
                      "version = ?, "
                      "last_execution_time = DATETIME ('now') "
                "WHERE item_uid = ? "
                  "AND id = ?");

      stmt.bind (1, name);
      stmt.bind (2, version);
      stmt.bind (3, impl_->get_uid ());
      stmt.bind (4, id);
    }

  else
    {
      stmt = db.new_statement (
               "INSERT INTO ant "
                    "VALUES (NULL, ?, ?, ?, ?, DATETIME ('now'))");

      stmt.bind (1, impl_->get_uid ());
      stmt.bind (2, id);
      stmt.bind (3, name);
      stmt.bind (4, version);
    }

  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Reset ANT
//! \param id ANT ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::reset_ant (const std::string& id)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "DELETE FROM ant "
                      "WHERE item_uid = ? "
                        "AND id = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, id);
  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get executed ANTs
//! \return Executed ANTs
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <ant>
item::get_ants () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                  "FROM ant "
                 "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());

  std::vector <ant> ants;

  while (stmt.fetch_row ())
    {
      auto uid = stmt.get_column_int64 (0);
      ants.emplace_back (*this, uid);
    }

  return ants;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove ANTs
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::remove_ants ()
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Item is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "DELETE FROM ant "
                "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new password
//! \param type Type
//! \param value Value
//! \param description Description
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
password
item::new_password (
  const std::string& type,
  const std::string& value,
  const std::string& description)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "INSERT INTO password "
                "VALUES (NULL, ?, ?, ?, ?)");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, type);
  stmt.bind (3, value);
  stmt.bind (4, description);
  stmt.execute ();

  auto uid = db.get_last_insert_row_id ();
  return password (impl_->get_case (), uid);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get passwords
//! \return Passwords
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <password>
item::get_passwords () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                "FROM password "
                "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());

  std::vector <password> passwords;

  while (stmt.fetch_row ())
    {
      auto uid = stmt.get_column_int64 (0);
      passwords.emplace_back (impl_->get_case (), uid);
    }

  return passwords;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove passwords
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::remove_passwords ()
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "DELETE FROM password "
                "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new password hash
//! \param type Type
//! \param value Value
//! \param description Description
//! \return Password hash
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
password_hash
item::new_password_hash (
  const std::string& type,
  const std::string& value,
  const std::string& description)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "INSERT INTO password_hash "
                "VALUES (NULL, ?, ?, ?, ?, NULL)");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, type);
  stmt.bind (3, value);
  stmt.bind (4, description);
  stmt.execute ();

  auto uid = db.get_last_insert_row_id ();
  return password_hash (impl_->get_case (), uid);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get password hashes
//! \return Password hashes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <password_hash>
item::get_password_hashes () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                  "FROM password_hash "
                 "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());

  std::vector <password_hash> password_hashes;

  while (stmt.fetch_row ())
    {
      auto uid = stmt.get_column_int64 (0);
      password_hashes.emplace_back (impl_->get_case (), uid);
    }

  return password_hashes;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove password hashes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::remove_password_hashes ()
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "DELETE FROM password_hash "
                "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new cookie
//! \param name Name
//! \param value Value
//! \param is_encrypted Is value encrypted?
//! \return Cookie
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
cookie
item::new_cookie (
  const std::string& name,
  const std::string& value,
  bool is_encrypted)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "INSERT INTO cookie "
                     "VALUES (NULL, ?, ?, ?, NULL, NULL, NULL, NULL, "
		             "NULL, 0, ?, NULL)");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, name);
  stmt.bind (3, value);
  stmt.bind (4, is_encrypted);
  stmt.execute ();

  auto uid = db.get_last_insert_row_id ();
  return cookie (*this, uid);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get cookies
//! \return Cookies
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <cookie>
item::get_cookies () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                  "FROM cookie "
                 "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());

  std::vector <cookie> cookies;

  while (stmt.fetch_row ())
    {
      auto uid = stmt.get_column_int64 (0);
      cookies.emplace_back (*this, uid);
    }

  return cookies;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove cookies
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::remove_cookies ()
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "DELETE FROM cookie "
                 "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new profile
//! \param app_id Application ID
//! \param path Path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
profile
item::new_profile (
  const std::string& app_id,
  const std::string& path)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // get application UID
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                  "FROM application "
                 "WHERE id = ?");

  stmt.bind (1, app_id);

  if (!stmt.fetch_row ())
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Invalid app_id"));

  auto app_uid = stmt.get_column_int64 (0);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Check if profile already exists
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  stmt = db.new_statement (
                  "SELECT uid "
                    "FROM profile "
                   "WHERE item_uid = ? "
                     "AND path = ? "
                     "AND application_uid = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, path);
  stmt.bind (3, app_uid);

  if (stmt.fetch_row ())
    {
      auto uid = stmt.get_column_int64 (0);
      return profile (*this, uid);
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // create profile
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  stmt = db.new_statement (
                "INSERT INTO profile "
                     "VALUES (NULL, ?, NULL, ?, NULL, NULL, ?)");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, path);
  stmt.bind (3, app_uid);
  stmt.execute ();

  auto uid = db.get_last_insert_row_id ();
  return profile (*this, uid);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get profiles
//! \return Pprofiles
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <profile>
item::get_profiles () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                  "FROM profile "
                 "WHERE item_uid = ?");

  stmt.bind (1, impl_->get_uid ());

  std::vector <profile> profiles;

  while (stmt.fetch_row ())
    {
      auto uid = stmt.get_column_int64 (0);
      profiles.emplace_back (*this, uid);
    }

  return profiles;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove profiles by application ID
//! \param app_id Application ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::remove_profiles_by_app_id (const std::string& app_id)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("item object is null"));

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // get application UID
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                  "FROM application "
                 "WHERE id = ?");

  stmt.bind (1, app_id);

  if (!stmt.fetch_row ())
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Invalid app_id"));

  auto app_uid = stmt.get_column_int64 (0);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // delete profiles
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  stmt = db.new_statement (
                "DELETE FROM profile "
                      "WHERE item_uid = ? "
                        "AND application_uid = ?");

  stmt.bind (1, impl_->get_uid ());
  stmt.bind (2, app_uid);
  stmt.execute ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Expand ${} masks into attribute values
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
item::expand_masks ()
{
  mobius::core::category_manager category_manager;
  auto category = category_manager.get_category (get_category ());

  if (!category)
    return;

  for (const auto& attr : category.get_attributes ())
    {
      const std::string value_mask = attr.get_value_mask ();

      if (!value_mask.empty ())
        {
          std::string value = expand_value_mask (value_mask, *this);
          set_attribute (attr.get_id (), value);
        }
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get data path
//! \param rpath relative path
//! \return fullpath
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
item::get_data_path (const std::string& rpath) const
{
  char s_uid[16];
  sprintf (s_uid, "%04ld", get_uid ());

  auto c = get_case ();
  return c.get_path ("data/" + std::string (s_uid) + '/' + rpath);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create data path
//! \param rpath relative path
//! \return fullpath
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
item::create_data_path (const std::string& rpath) const
{
  char s_uid[16];
  sprintf (s_uid, "%04ld", get_uid ());

  auto c = get_case ();
  return c.create_path ("data/" + std::string (s_uid) + '/' + rpath);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if two items are equal
//! \param a item a
//! \param b item b
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
operator== (const item& a, const item& b)
{
  return a.get_uid () == b.get_uid ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if two items are different
//! \param a item a
//! \param b item b
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
operator!= (const item& a, const item& b)
{
  return a.get_uid () != b.get_uid ();
}

} // namespace model
} // namespace mobius
