// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "hmac.h"

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief initialize object
//! \param key cryptographic key
//! \param hash_id hash function class ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
hmac::hmac (const bytearray& key, const std::string& hash_id)
  : h_inner_ (hash (hash_id)),
    h_outer_ (hash (hash_id))
{
  std::size_t block_size = h_inner_.get_block_size ();

  bytearray K (block_size);
  K.fill (0);

  // keys longer than H::block_size are first hashed using H (RFC 2104 - section 3)
  if (key.size () > block_size)
    {
      hash h (hash_id);
      h.update (key);
      auto digest = h.get_digest ();
      std::copy (digest.begin (), digest.end (), K.begin ());
    }

  else
    {
      std::copy (key.begin (), key.end (), K.begin ());
    }

  // initialize inner hash object (RFC 2104 - section 2)
  bytearray ipad (block_size);
  ipad.fill (0x36);
  h_inner_.update (K ^ ipad);

  // initialize outer hash object (RFC 2104 - section 2)
  bytearray opad (block_size);
  opad.fill (0x5c);
  h_outer_.update (K ^ opad);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get digest
//! \return digest
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
hmac::get_digest ()
{
  h_outer_.update (h_inner_.get_digest ());
  return h_outer_.get_digest ();
}

} // namespace crypt
} // namespace mobius
