// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/decoder/mfc_decoder.h>
#include <mobius/datetime/conv_unix_timestamp.h>
#include <mobius/exception.inc>
#include <stdexcept>

namespace mobius
{
namespace decoder
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param reader reader object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mfc_decoder::mfc_decoder (mobius::io::reader reader)
 : decoder_ (reader)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode bool
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
mfc_decoder::get_bool ()
{
  return decoder_.get_uint32_le () == 1;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode WORD (16 bits)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint16_t
mfc_decoder::get_word ()
{
  return decoder_.get_uint16_le ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode DWORD (32 bits)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
mfc_decoder::get_dword ()
{
  return decoder_.get_uint32_le ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode QWORD (64 bits)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint64_t
mfc_decoder::get_qword ()
{
  return decoder_.get_uint64_le ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode int (32 bits)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int32_t
mfc_decoder::get_int ()
{
  return decoder_.get_int32_le ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode count
//! \see http://computer-programming-forum.com/82-mfc/1758299f7763d979.htm
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
mfc_decoder::get_count ()
{
  std::uint32_t count = decoder_.get_uint16_le ();
  
  if (count == 0xffff)
    count = decoder_.get_uint32_le ();
  
  return count;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode GUID
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
mfc_decoder::get_guid ()
{
  return decoder_.get_guid ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode data as hexadecimal string
//! \param siz size in bytes
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
mfc_decoder::get_hex_string (size_t siz)
{
  bool is_valid = decoder_.get_uint32_le () == 1;
  std::string str;
  
  if (is_valid)
    str = decoder_.get_bytearray_by_size (siz).to_hexstring ();
  
  return str;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode data
//! \param siz size in bytes
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::bytearray
mfc_decoder::get_data (size_t siz)
{
  return decoder_.get_bytearray_by_size (siz);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode CString
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
mfc_decoder::get_string ()
{
  std::uint32_t size = decoder_.get_uint8 ();
  bool b_utf16 = false;
  
  if (size == 0xff)
    {
      size = decoder_.get_uint16_le ();
      if (size == 0xfffe)
        {
          b_utf16 = true;
          size = decoder_.get_uint8 ();
          
          if (size == 0xff)
            size = decoder_.get_uint16_le ();
        }

      if (size == 0xffff)
        size = decoder_.get_uint32_le ();
    }

  std::string r;

  if (b_utf16)
    r = decoder_.get_string_by_size (size * 2, "UTF-16");
  else
    r = decoder_.get_string_by_size (size, "ASCII");

  return r;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode CTime
//! \see https://msdn.microsoft.com/en-us/library/b6989cds.aspx
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::datetime::datetime
mfc_decoder::get_ctime ()
{
  std::uint32_t size = decoder_.get_uint8 ();
  
  if (size != 0x0a)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("unknown time encoding"));
  
  decoder_.get_uint16_le ();	// dummy1
  decoder_.get_int8 ();		// ndst
  std::uint64_t timestamp = decoder_.get_uint64_le ();

  return mobius::datetime::datetime_from_unix_timestamp (timestamp);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode NT time
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::datetime::datetime
mfc_decoder::get_nt_time ()
{
  return decoder_.get_nt_datetime ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode IP v4
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
mfc_decoder::get_ipv4 ()
{
  return decoder_.get_ipv4 ();
}

} // namespace decoder
} // namespace mobius
