// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/io/folder_impl_local.h>
#include <mobius/exception_posix.inc>
#include <mobius/uri.h>
#include <sys/types.h>
#include <dirent.h>
#include <memory>

namespace mobius
{
namespace io
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief walker for local folders
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class local_walker : public folder_impl_base::walker
{
public:
  explicit local_walker (const std::string&);
  virtual const std::string get () override;

private:
  std::shared_ptr <DIR> dir_p_;
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief initialize object
//! \param path path to local folder
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
local_walker::local_walker (const std::string& path)
{
  DIR *dir_p = opendir (path.c_str ());
  if (!dir_p)
    throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);

  dir_p_ = std::shared_ptr <DIR> (dir_p, closedir);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get folder resource
//! \return resource name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
local_walker::get ()
{
  std::string value;

  dirent entry;
  dirent *result;

  if (readdir_r (dir_p_.get (), &entry, &result))
    throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);

  while (value.empty () && result)
    {
      std::string filename = result->d_name;

      if (filename != "." && filename != "..")
        value = filename;

      else if (readdir_r (dir_p_.get (), &entry, &result))
        throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);
    }

  return value;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief initialize object
//! \param url local folder's URL
//! \param url URL
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
folder_impl_local::folder_impl_local (const std::string& url)
{
  mobius::uri u (url);
  path_ = u.get_path ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new walker
//! \return walker for folder
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::shared_ptr <folder_impl_base::walker>
folder_impl_local::new_walker () const
{
  return std::make_shared <local_walker> (path_);
}

} // namespace io
} // namespace mobius
