#!/usr/bin/env python

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# ICQ 4.1 (ICQLite)
# Arquivos de Programas/ICQLite/<ID>.fb -> feedbag (buddy list)
# Arquivos de Programas/ICQLite/ICQLite.exe -> executavel
# 88189260fff03baac505d9d6071e9fe1  ICQLite.exe (ICQ 4.1)
# Documents and Settings/<user>/Meus documentos/Dados de aplicativos/ICQLite
#    ---> Bartcache/<ID>
#         ---> 1/* -> jpeg e gif ????
#         ---> 8/* -> xml ???
#         ---> Temp/*.tmp -> jpeg e gif ???
#         ---> cl/*.dat -> ???
#    ---> HistoryDB/<ID>/<BUDDY_ID>/*.xml  -> mensagens
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @todo Eliminar selecao de colunas
# @todo SPACE -> toggle
# @todo verificar erro quando 'open' diretorio inicial
# @todo Suporte a mais de um TO
# @todo Botao de 'find'
# @todo Opcao para remover mensagens duplicadas
# @todo Controle de multiplas janelas
# @todo Opcao 'close' para janela
# @todo Escolher icones melhores
# @todo Permitir configuracao de locale
# @todo Processar imagem raw (fat32, por enquanto)
# @TODO Habilitar opcoes somente quando tiver mensagens (zoom, select all, ...)
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import os.path
import optparse
import re
import datetime
import ConfigParser
import gobject
import gtk
import pango
import libxml2
import mobius.application
import mobius.config
import mobius.decoder.aol_feedbag
import mobius.decoder.icqlite_xml
import mobius.decoder.icq2003b_uin
import mobius.ui.base_window
import mobius.ui.manager

MESSAGE_EXP = re.compile ('.*?/ICQLite/HistoryDB/([0-9]+)/([0-9]+)/.*')
BASE_DATETIME = datetime.datetime (1970, 1, 1)
(MSG_TOGGLE, MSG_TIME, MSG_FROM, MSG_TO, MSG_TEXT, MSG_OBJ) = range (6)

#-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Account
#-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Account (object):
  def __init__ (self, uin):
    self.uin = uin
    self.alias = '%d' % uin
    self.nickname = ''
    self.firstname = ''
    self.lastname = ''
    self.email = ''
    self.buddy_lists = {}
    self.message_files = []

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add buddy list
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_buddy_list (self, path, version):
    fp = open (path)
    data = fp.read ()
    fp.close ()

    if mobius.decoder.aol_feedbag.signature (data):
      self.buddy_lists[version] = BuddyList (path, version)

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add message file
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_message_file (self, path, version):
    fp = open (path)
    data = fp.read ()
    fp.close ()

    if mobius.decoder.icqlite_xml.signature (data):
      self.message_files.append (MessageFile (path, version))

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add uin file
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_uin_file (self, path, version):
    fp = open (path)
    data = fp.read ()
    fp.close ()

    if mobius.decoder.icq2003b_uin.signature (data):
      decoder = mobius.decoder.icq2003b_uin.Decoder ()
      model = decoder.decode (data)
      self.nickname = model.nickname
      self.firstname = model.firstname
      self.lastname = model.lastname
      self.email = model.email

      if model.firstname:
        self.alias = '%s %s' % (model.firstname, model.lastname)
      elif model.nickname:
        self.alias = model.nickname
      else:
        self.alias = '%d' % self.uin

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get messages
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_messages (self):
    decoder = mobius.decoder.icqlite_xml.Decoder ()

    for message_file in self.message_files:
      data = open (message_file.path).read ()
      model = decoder.decode (data)
      peer_uin = int (os.path.basename (message_file.path).split ('-')[0])
      buddy_list = self.get_buddy_list (message_file.version)
 
      for message in model.messages:
        msg = Message ()
        msg.text = message.text
        msg.time = message.time

        if message.incoming:
          msg.from_uin = peer_uin
          msg.from_alias = buddy_list.get_alias (msg.from_uin)
          msg.to_uin = self.uin
          msg.to_alias = self.alias
        else:
          msg.from_uin = self.uin
          msg.from_alias = self.alias
          msg.to_uin = peer_uin
          msg.to_alias = buddy_list.get_alias (msg.to_uin)

        yield msg

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get buddy list suitable for ICQ version
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_buddy_list (self, version):
    buddy_list = self.buddy_lists.get (version)
    return buddy_list
      
#-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Buddylist
#-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class BuddyList (object):
  def __init__ (self, path, version):
    self.path = path
    self.version = version
    self.model = None

  def get_alias (self, uin):
    if not self.model:
      data = open (self.path).read ()
      decoder = mobius.decoder.aol_feedbag.Decoder ()
      self.model = decoder.decode (data)

    buddy = self.model.buddies.get (uin)
    if buddy:
      return buddy.alias
    else:
      return '%d' % uin

#-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief MessageFile
#-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class MessageFile (object):
  def __init__ (self, path, version):
    self.path = path
    self.version = version

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generic message
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Message (object):
  def __init__ (self):
    self.time = BASE_DATETIME
    self.from_id = -1
    self.from_alias = ''
    self.to_id = -1
    self.to_alias = ''
    self.text = ''

  def __cmp__ (self, o):
    if self.time < o.time:
      return -1
    elif self.time > o.time:
      return 1
    return 0

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief ICQ Model
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class ICQModel (object):

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize model
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.accounts = {}

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get account
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_account (self, uin):
    return self.accounts.setdefault (uin, Account (uin))

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load cache
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self):
    doc = libxml2.parseFile ('mobius_mail.cache')
    node = doc.getRootElement ()

    # parse elements
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'account':
        self.account[int (node.prop ('uin'))] = self.load_account (node)

      node = node.next

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load account
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_account (self, node):
    uin = int (node.prop ('uin'))
    account = Account (uin)

    # parse elements
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'buddylist':
        account.add_buddy_list (node.prop ('path'), node.prop ('version'))

      elif node.type == 'element' and node.name == 'messagefile':
        account.add_message_file (node.prop ('path'), node.prop ('version'))

      node = node.next

    self.account[uin] = account

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save cache
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save (self):
    doc = libxml2.newDoc ('1.0')
    root = libxml2.newNode ('cache')
    doc.addChild (root)

    for account in self.accounts.itervalues ():
      node = self.save_account (account)
      root.addChild (node)

    doc.saveFormatFileEnc ('mobius_icq.cache', 'utf-8', 1)
    doc.freeDoc ()

  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save account
  #-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_account (self, account):
    node = libxml2.newNode ('account')
    node.setProp ('uin', str (account.uin))

    for buddy_list in account.buddy_lists.itervalues ():
      cnode = libxml2.newNode ('buddylist')
      cnode.setProp ('path', buddy_list.path)
      cnode.setProp ('version', buddy_list.version)
      node.addChild (cnode)

    for message_file in account.message_files:
      cnode = libxml2.newNode ('messagefile')
      cnode.setProp ('path', message_file.path)
      cnode.setProp ('version', message_file.version)
      node.addChild (cnode)

    return node

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Report viewer
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class ReportWindow (gtk.Window):

  def __init__ (self, *args):
    gtk.Window.__init__ (self, *args)
    self.set_default_size (400, 570)
    self.set_title ('Report')

    # accel_group
    accel_group = gtk.AccelGroup ()
    self.add_accel_group (accel_group)

    # textview
    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    self.add (sw)

    self.textview = gtk.TextView ()
    self.textview.show ()
    sw.add (self.textview)

    buffer = self.textview.get_buffer ()
    tag = buffer.create_tag ('day')
    tag.set_property ('foreground', '#00F')
    tag = buffer.create_tag ('sender')
    tag.set_property ('weight', pango.WEIGHT_BOLD)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add day
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_date (self, date):
    buffer = self.textview.get_buffer ()
    
    #@todo Other languages and locales
    if buffer.get_line_count () > 1:
      text = '\n'
    else:
      text = ''

    text+= '%d' % date.day

    if date.day == 1:
      text += 'o'

    text += date.strftime (' de %B de %Y\n')
    buffer.insert_with_tags_by_name (buffer.get_end_iter (), text, 'day')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add message
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_message (self, msg):
    buffer = self.textview.get_buffer ()

    #@todo Other formats
    alias = msg.from_alias or str (msg.from_id)
    text = msg.text.replace ('\n', '\n\t')
    buffer.insert_with_tags_by_name (buffer.get_end_iter (), alias, 'sender')
    buffer.insert (buffer.get_end_iter (), ' (%s): %s\n' % (msg.time.strftime ('%H:%M'), text))

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Instant messenger viewer
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class MobiusICQViewer (mobius.ui.base_window.MobiusBaseWindow):

  def __init__ (self, *args):
    mobius.ui.base_window.MobiusBaseWindow.__init__ (self, *args)
    self.set_default_size (720, 570)
    self.set_title ()

    item = gtk.MenuItem ('_Edit')
    item.show ()
    self.menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    item = gtk.ImageMenuItem (gtk.STOCK_SELECT_ALL, self.accel_group)
    item.connect ("activate", self.on_select_all)
    item.show ()
    menu.append (item)

    item = gtk.MenuItem ('_Unselect all')
    item.connect ("activate", self.on_unselect_all)
    item.show ()
    menu.append (item)

    # toolbar
    button = gtk.ToolButton (gtk.STOCK_ZOOM_IN)
    button.connect ("clicked", self.on_open_view)
    button.show ()
    button.set_tooltip (self.tooltips, "Dialogy zoom")
    self.toolbar.insert (button, 1)

    button = gtk.ToolButton (gtk.STOCK_INDENT)
    button.connect ("clicked", self.on_report_view)
    button.show ()
    button.set_tooltip (self.tooltips, "Generate report")
    self.toolbar.insert (button, 2)

    # icq list
    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    self.vbox.pack_start (sw)

    model = gtk.ListStore (bool, str, str, str, str, object)

    self.treeview = gtk.TreeView (model)
    self.treeview.set_headers_visible (True)
    self.treeview.set_search_column (MSG_TEXT)
    self.treeview.set_search_equal_func (self.treeview_search)
    self.treeview.set_rules_hint (True)
    sw.add (self.treeview)
    self.treeview.show ()

    renderer = gtk.CellRendererToggle ()
    renderer.set_property ('activatable', True)
    renderer.connect ('toggled', self.on_toggle_selection)
    tvcolumn = gtk.TreeViewColumn (u'\u2714')
    tvcolumn.set_sort_column_id (MSG_TOGGLE)
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'active', MSG_TOGGLE)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Date/time')
    tvcolumn.set_sort_column_id (MSG_TIME)
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', MSG_TIME)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Sender')
    tvcolumn.set_sort_column_id (MSG_FROM)
    tvcolumn.set_resizable (True)
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', MSG_FROM)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Receiver')
    tvcolumn.set_sort_column_id (MSG_TO)
    tvcolumn.set_resizable (True)
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', MSG_TO)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Text')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', MSG_TEXT)
    tvcolumn.set_resizable (True)
    self.treeview.append_column (tvcolumn)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Populate
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def populate (self, account):
    self.set_title ('%d' % account.uin)

    # populate liststore
    liststore = self.treeview.get_model ()

    for msg in account.get_messages ():
      liststore.append ((False, msg.time, msg.from_alias, msg.to_alias, msg.text, msg))

    self.status_label.set_text ('Messages: %d' % len (liststore))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set title
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_title (self, subtitle=None):
    title = 'Mobius ICQ Viewer v%s' % mobius.config.APP_VERSION
    if subtitle:
      title += ' [%s]' % subtitle

    gtk.Window.set_title (self, title)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Treeview search
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def treeview_search (self, model, column, key, iter, *args):
    return key.lower () not in model.get_value (iter, column).lower ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Close window
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_close_window (self, widget, *args):
    return self.app.destroy_window (self.id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief toogle selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_toggle_selection (self, renderer, path, *args):
    model = self.treeview.get_model ()
    iter = model.get_iter (path)
    model[iter][MSG_TOGGLE] = not model[iter][MSG_TOGGLE] 

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Dialogy view
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_open_view (self, *args):
    model, iter = self.treeview.get_selection ().get_selected ()

    if iter:
      msg_from = model.get_value (iter, MSG_FROM)
      msg_to = model.get_value (iter, MSG_TO)
      window = self.app.build_window (MobiusICQViewer)
      window_model = window.treeview.get_model ()

      for data in model:
        if data[MSG_FROM] in (msg_from, msg_to) and data[MSG_TO] in (msg_from, msg_to):
          window_model.append (data)

      window.show ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Report view
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_report_view (self, *args):
    model = self.treeview.get_model ()
    report = ReportWindow ()
    date = None

    for data in model:
      if data[MSG_TOGGLE]:
        msg = data[MSG_OBJ]

        if date != msg.time.date ():
          date = msg.time.date ()
          report.add_date (date)

        report.add_message (msg)

    report.show ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Select all
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_select_all (self, *args):
    model = self.treeview.get_model ()

    for data in model:
      data[MSG_TOGGLE] = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Unselect all
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_unselect_all (self, *args):
    model = self.treeview.get_model ()

    for data in model:
      data[MSG_TOGGLE] = False
 
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief ICQ viewer
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class MobiusICQBrowser (gtk.Window):

  def __init__ (self, app, *args):
    gtk.Window.__init__ (self)
    self.connect ('delete-event', self.on_close_window)
    self.set_default_size (600, 400)
    self.set_title ('Mobius ICQ Viewer v%s' % mobius.config.APP_VERSION)

    # accel_group
    accel_group = gtk.AccelGroup ()
    self.add_accel_group (accel_group)

    # vbox
    vbox = gtk.VBox (False, 1)
    vbox.set_border_width (1)
    self.add (vbox)
    vbox.show ()

    # menubar
    menubar = gtk.MenuBar ()
    menubar.show ()
    vbox.pack_start (menubar, False, False)

    item = gtk.MenuItem ('_File')
    item.show ()
    menubar.append (item)

    file_menu = gtk.Menu ()
    file_menu.show ()
    item.set_submenu (file_menu)

    item = gtk.ImageMenuItem (gtk.STOCK_OPEN, accel_group)
    item.connect ("activate", self.on_open)
    item.show ()
    file_menu.insert (item, 0)

    item = gtk.SeparatorMenuItem ()
    item.show ()
    file_menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_QUIT, accel_group)
    item.connect ("activate", app.on_app_quit)
    item.show ()
    file_menu.append (item)

    item = gtk.MenuItem ('_Help')
    item.show ()
    menubar.append (item)

    help_menu = gtk.Menu ()
    help_menu.show ()
    item.set_submenu (help_menu)

    item = gtk.ImageMenuItem (gtk.STOCK_ABOUT, accel_group)
    item.connect ("activate", self.on_help_about)
    item.show ()
    help_menu.append (item)

    # toolbar
    tooltips = gtk.Tooltips ()

    handlebox = gtk.HandleBox ()
    handlebox.show ()
    vbox.pack_start (handlebox, False, False)

    toolbar = gtk.Toolbar ()
    toolbar.set_style (gtk.TOOLBAR_ICONS)
    toolbar.set_tooltips (True)
    toolbar.show ()
    handlebox.add (toolbar)

    button = gtk.ToolButton (gtk.STOCK_OPEN)
    button.connect ("clicked", self.on_open)
    button.show ()
    button.set_tooltip (tooltips, "Select root directory")
    toolbar.insert (button, 0)

    button = gtk.ToolButton (gtk.STOCK_QUIT)
    button.connect ("clicked", app.on_app_quit)
    button.show ()
    button.set_tooltip (tooltips, "Exit from Mobius")
    toolbar.insert (button, -1)

    # status bar
    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_IN)
    frame.show ()
    vbox.pack_end (frame, False, False)

    self.status_label = gtk.Label ()
    self.status_label.set_alignment (0, -1)
    self.status_label.set_ellipsize (pango.ELLIPSIZE_END)
    self.status_label.show ()
    frame.add (self.status_label)

    # icq list
    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    vbox.pack_start (sw)

    model = gtk.ListStore (int, str, str, str, str, object)

    self.treeview = gtk.TreeView (model)
    self.treeview.set_headers_visible (True)
    self.treeview.connect ('row-activated', self.on_select_account)
    sw.add (self.treeview)
    self.treeview.show ()

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('UIN')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', 0)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Nickname')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', 1)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('First name')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', 2)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Last name')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', 3)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('E-mail')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', 4)
    self.treeview.append_column (tvcolumn)

    # windows data
    self.app = app
    self.id = app.get_next_id ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close window
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_close_window (self, *args):
    return self.app.destroy_window (self.id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call 'about' dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_help_about (self, widget, *args):
    dialog = mobius.ui.about_dialog.Dialog ()
    dialog.run ()
    dialog.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief open directory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_open (self, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Choose folder', parent=self)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)
    fs.set_action (gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER)

    rc = fs.run ()
    path = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    model = self.treeview.get_model ()
    model.clear ()
    self.scan (path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief select account
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_select_account (self, treeview, path, view_column, *args):
    model, iter = treeview.get_selection ().get_selected ()

    if iter:
      uin = model.get_value (iter, 0)
      account = model.get_value (iter, 5)
      window = self.app.build_window (MobiusICQViewer)
      window.populate (account)
      window.show ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Scan ICQ directory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def scan (self, path):
    ui_manager = mobius.ui.manager.UIManager (self)
    ui_manager.flush ()
    pos = len (path) + 1
    model = ICQModel ()

    # scan directory
    for root, dirs, files in os.walk (path):
      self.status_label.set_text ('Scanning: %s' % root[pos:])
      ui_manager.flush ()

      for filename in files:
        fullpath = os.path.join (root, filename)
        extension = os.path.splitext (filename)[1][1:]

        if extension == 'fb' and '/ICQLite/' in fullpath:
          uin = int (os.path.splitext (os.path.basename (fullpath))[0])
          account = model.get_account (uin)
          account.add_buddy_list (fullpath, 'ICQLite')

        elif extension == 'xml' and '/ICQLite/HistoryDB/' in fullpath:
          match = MESSAGE_EXP.match (fullpath)
          uin = int (match.group (1))
          peer_uin = int (match.group (2))
          account = model.get_account (uin)
          account.add_message_file (fullpath, 'ICQLite')

        elif extension == 'uin' and '/ICQ/UIN/' in fullpath:
          uin = int (os.path.splitext (os.path.basename (fullpath))[0])
          account = model.get_account (uin)
          account.add_uin_file (fullpath, 'ICQ2003b')

    model.save ()

    # populate liststore with ICQ accounts
    liststore = self.treeview.get_model ()
    accounts = model.accounts.items ()
    accounts.sort ()

    for uin, account in accounts:
      liststore.append ((uin, account.nickname, account.firstname, account.lastname, account.email, account))

    self.status_label.set_text ('UINs: %d' % len (liststore))

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Process command line
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
app = mobius.application.Application ()
app.build_window (MobiusICQBrowser)
app.start ()
