/* quoting.c - Various routines for working with quoted strings
 *
 * Copyright (C) 2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

/* TODO: unified handling of whitespace (single point of definition),
 * both as string and function?
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdbool.h>		/* Gnulib/C99/POSIX */
#include <string.h>		/* C89 */
#include <stdint.h>		/* Gnulib/C99/POSIX */
#include "xalloc.h"		/* Gnulib */
#include "quoting.h"

/* Quote STRING with double quotes if QUOTED is true, otherwise
 * by escaping whitespace, double quote and backslash.
 */
char *
quote_word(const char *string, bool quoted, bool add_end_quote)
{
    const char *whitespace = " \n\t";
    const char *s;
    char *result;
    char *r;

    if (quoted) {
        result = r = xmalloc(2 * strlen(string) + 3);
        *r++ = '"';
        for (s = string; *s; s++) {
	    if (*s == '"' || *s == '\\')
	        *r++ = '\\';
	    *r++ = *s;
        }
        if (add_end_quote)
            *r++ = '"';
        *r = '\0';
    } else {
        result = r = xmalloc(2 * strlen(string) + 1);
        for (s = string; *s; s++) {
	    if (*s == '"' || *s == '\\' || strchr(whitespace, *s) != NULL)
	        *r++ = '\\';
	    *r++ = *s;
        }
        *r = '\0';
    }

    return result;
}

/* Remove quotes and backslashs from STR, looking no further than
 * MAXEND, or if MAXEND is null, where STR ends. MODE specifies
 * initial quoting - false for no quoting, true for double quotes.
 */
char *
dequote_words(const char *str, bool quoted, const char *maxend)
{
    const char *p;
    char *result;
    char *r;

    if (maxend == NULL)
	maxend = (void *) UINTPTR_MAX;

    result = r = xmalloc(strlen(str) + 1);
    for (p = str; p < maxend && *p != '\0'; p++) {
	if (*p == '\\') {
	    p++;
	    if (p >= maxend || *p == '\0')
		break; /* also discards bogus trailing backslash */
	    *r++ = *p;
	} else if (*p == '"') {
	    quoted = !quoted;
	} else {
	    *r++ = *p;
	}
    }
    *r = '\0';

    return result;
}

/* Find the first character which is part of the first word
 * in STR, i.e. skip past leading whitespace. Look no further
 * than MAXEND or where STR ends. If MAXEND is NULL, stop only
 * where STR ends.
 */
static const char *
find_word_start(const char *str, const char *maxend)
{
    const char *whitespace = " \n\t";

    if (maxend == NULL)
        maxend = (void *) UINTPTR_MAX;

    for (; str < maxend && *str != '\0'; str++) {
        if (strchr(whitespace, *str) == NULL)
            break;
    }

    return str;
}

/* Find the first character which is not part of the first word
 * in STR, but look no further than MAXEND or where STR ends.
 * If MAXEND is NULL, stop only where STR ends.
 */
static const char *
find_word_end(const char *str, const char *maxend)
{
    const char *whitespace = " \n\t";
    bool quoted = false;

    if (maxend == NULL)
        maxend = (void *) UINTPTR_MAX;

    str = find_word_start(str, maxend);
    for (; str < maxend && *str != '\0'; str++) {
        if (*str == '\\') {
            str++;
            if (str >= maxend || *str == '\0')
                break;
        } else if (*str == '"') {
            quoted = !quoted;
        } else if (!quoted && strchr(whitespace, *str) != NULL) {
            break;
        }
    }

    return str;
}

/* Return index of the word at position POS in the string STR.
 * This is also the index of the word that would be at POS -
 * there may not necessarily be a complete word at POS.
 */
int
get_word_index(const char *str, int pos)
{
    const char *maxend = str + pos;
    int index;

    for (index = 0; ; index++) {
        str = find_word_end(str, maxend);
        if (str >= maxend || *str == '\0')
            return index;
    }
}

/* Get the word in STR that would be used for completion if 
 * the user were to press tab at POS. This function may return
 * partial words in STR, or even the empty string.
 */
char *
get_completion_word_dequoted(const char *str, int pos)
{
    const char *maxend = str + pos;
    const char *start = str;
    int index;

    for (index = 0; ; index++) {
        str = find_word_end(str, maxend);
        if (str >= maxend || *str == '\0') {
	    start = find_word_start(start, maxend);
	    if (start >= maxend || *start == '\0')
		return xstrdup("");
	    return dequote_words(start, false, maxend);
	}
	start = str;
    }
}

/* Split input into an array of words. The array will be null-terminated.
 *
 * XXX: this function has not been tested
 */
char **
get_word_array_dequoted(const char *str, const char *strend, int *argc)
{
    size_t array_cur;
    size_t array_max;
    char **array;

    if (strend == NULL)
        strend = (void *) UINTPTR_MAX;

    array_cur = 0;
    array_max = 8;
    array = xmalloc(array_max * sizeof(char *));

    str = find_word_start(str, strend);
    if (str < strend && *str != '\0') {
        while (str < strend && *str != '\0') {
            const char *end;
            end = find_word_end(str, strend);
            if (array_cur >= array_max) {
                array_max *= 2;
                array = xrealloc(array, array_max * sizeof(char *));
            }
            array[array_cur++] = dequote_words(str, false, end);
            if (end >= strend || *end == '\0')
                break;
            str = end;
        }
    }

    if (array_cur >= array_max) {
        array_max *= 2;
        array = xrealloc(array, array_max * sizeof(char *));
    }
    array[array_cur] = NULL;
    return array;
}

/* Return a newly allocated string of COUNT words starting at
 * INDEX in string STR. The returned string with be dequoted.
 */
char *
get_subwords_dequoted(const char *str, const char *strend, int index, size_t count)
{
    const char *end;
    int c;

    /* XXX: get rid of this */
    if (str == NULL)
	return NULL;

    if (strend == NULL)
        strend = (void *) UINTPTR_MAX;

    for (c = 0; c < index; c++) {
        str = find_word_end(str, strend);
        if (str >= strend || *str == '\0')
            return NULL;
    }

    str = find_word_start(str, strend);
    if (str >= strend || *str == '\0')
        return NULL;

    end = str;
    for (c = 0; c < count; c++) {
        end = find_word_end(end, strend);
        if (end >= strend || *end == '\0')
            break;
    }
    
    return dequote_words(str, false, end);
}

/* Determine if character INDEX in STRING is
 * inside a quoted string or right behind a backslash.
 */
int
char_is_quoted(char *string, int index)
{
    bool escaped = false;
    int c;

    for (c = 0; c <= index; c++) {
	if (escaped) {
	    if (c >= index)
		return 1;
	    escaped = false;
	} else if (string[c] == '"') {
	    char quote = string[c];
	    c++;
	    for (; c < index && string[c] != '\0' && string[c] != quote; c++) {
		if (string[c] == '\\' && string[c+1] != '\0')
		    c++;
	    }
	    if (c >= index)
		return 1;
	} else if (string[c] == '\\') {
	    escaped = true;
	}
    }

    return 0;
}
