/* command.c - Command input and basic commands
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <ctype.h>		/* C89 */
#include <string.h>		/* ? */
#include <sys/types.h>		/* ? */
#include <sys/stat.h>		/* ? */
#include <unistd.h>		/* POSIX */
#include <netdb.h>		/* ? */
#include <sys/socket.h>		/* POSIX */
#include <netinet/in.h>		/* ? */
#include <arpa/inet.h>		/* ? */
#include <inttypes.h>		/* ? */
#include "xvasprintf.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#include "quotearg.h"		/* Gnulib */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "common/error.h"
#include "common/strleftcmp.h"
#include "common/intutil.h"
#include "common/minmaxonce.h"
#include "common/strbuf.h"
#include "common/range.h"
#include "common/quoting.h"
#include "common/comparison.h"
#include "common/bksearch.h"
#include "common/swap.h"
#include "microdc.h"

static void cmd_exit(char *cmd, char *args);
static void cmd_say(char *cmd, char *args);
static void cmd_msg(char *cmd, char *args);
static void cmd_raw(char *cmd, char *args);
static void cmd_disconnect(char *cmd, char *args);
static void cmd_connect(char *cmd, char *args);
static void cmd_grantslot(char *cmd, char *args);
static void cmd_browse(char *cmd, char *args);
static void cmd_pwd(char *cmd, char *args);
static void cmd_find(char *cmd, char *args);
static void cmd_ls(char *cmd, char *args);
static void cmd_cd(char *cmd, char *args);
static void cmd_get(char *cmd, char *args);
static void cmd_queue(char *cmd, char *args);
static void cmd_unqueue(char *cmd, char *args);
static void cmd_who(char *cmd, char *args);
static void cmd_transfers(char *cmd, char *args);
static void cmd_cancel(char *cmd, char *args);
static void cmd_search(char *cmd, char *args);
static void cmd_status(char *cmd, char *args);
static void cmd_results(char *cmd, char *args);
static void cmd_unsearch(char *cmd, char *args);
static char *command_completion_generator(const char *base, int state);

typedef struct {
    char *name;
    void (*handler)(char *cmd, char *args);
    DCCompletionFunctionType type;
    void *completor;
} DCCommand;

/* This structure must be sorted by command name, according to strcmp. */
static DCCommand commands[] = {
    { "browse",     	cmd_browse, 	    DC_CPL_SIMPLE, user_or_myself_completion_generator },
    { "cancel",         cmd_cancel,         DC_CPL_SIMPLE, transfer_completion_generator },
    { "cd", 	    	cmd_cd,     	    DC_CPL_FILE, remote_dir_completion_generator },
    { "connect",    	cmd_connect, 	    DC_CPL_NONE, NULL },
    { "disconnect", 	cmd_disconnect,     DC_CPL_NONE, NULL },
    { "exit", 	    	cmd_exit,   	    DC_CPL_NONE, NULL },
    { "find",		cmd_find,	    DC_CPL_FILE, remote_path_completion_generator },
    { "get", 	    	cmd_get,    	    DC_CPL_FILE, remote_path_completion_generator },
    { "grantslot",  	cmd_grantslot,      DC_CPL_SIMPLE, user_completion_generator },
    { "ls", 	    	cmd_ls,     	    DC_CPL_FILE, remote_path_completion_generator },
    { "msg", 	    	cmd_msg,    	    DC_CPL_SIMPLE, user_completion_generator },
    { "pwd", 	    	cmd_pwd,    	    DC_CPL_NONE, NULL },
    { "queue",      	cmd_queue,  	    DC_CPL_SIMPLE, user_with_queue_completion_generator },
    { "raw", 	    	cmd_raw,    	    DC_CPL_NONE, NULL },
    { "results",        cmd_results,        DC_CPL_NONE, NULL },
    { "say", 	    	cmd_say,    	    DC_CPL_CUSTOM, say_user_completion_generator },
    { "search",         cmd_search,         DC_CPL_NONE, NULL },
    { "set",		cmd_set,            DC_CPL_CHAIN, set_completion_selector },
    { "status",		cmd_status,	    DC_CPL_NONE, NULL },
    { "transfers",   	cmd_transfers,	    DC_CPL_NONE, NULL },
    { "unqueue",    	cmd_unqueue, 	    DC_CPL_SIMPLE, user_completion_generator },
    { "unsearch",       cmd_unsearch,       DC_CPL_NONE, NULL },
    { "who", 	    	cmd_who,    	    DC_CPL_SIMPLE, user_completion_generator },
};
static const uint32_t commands_count = sizeof(commands)/sizeof(*commands);

static int
queued_file_cmp(const char *filename, DCQueuedFile *qf)
{
    return strcmp(filename, qf->filename);
}

static DCCommand *
find_command(const char *name)
{
    return (DCCommand *) bksearch(name, commands, commands_count,
                                  sizeof(DCCommand),
                                  offsetof(DCCommand, name),
                                  (comparison_fn_t) strcmp);
}

void *
default_completion_selector(const char *buffer, int start, int end, DCCompletionFunctionType *type)
{
    char *name;
    int c;
    DCCommand *cmd;

    c = get_word_index(buffer, end);
    if (c == 0) {
        *type = DC_CPL_SIMPLE;
        return command_completion_generator;
    }

    name = get_word_dequoted(buffer, 0);
    cmd = find_command(name);
    free(name);
    if (cmd != NULL) {
	*type = cmd->type;
    	return cmd->completor;
    }

    return NULL;
}

static char *
command_completion_generator(const char *base, int state)
{
    return sorted_list_completion_generator(base, state, commands, commands_count, sizeof(DCCommand), offsetof(DCCommand, name));
}

/* Execute a command specification from a complete line.
 */
void
command_execute(char *line)
{
    char *name;
    char *args;
    int i;
    DCCommand *cmd;

    for (; isspace(*line); line++);
    if (!*line || *line == '#')
    	return;

    name = line;
    for (; *line && !isspace(*line); line++);
    args = NULL;
    if (*line) {
    	*line = '\0';
	for (line++; isspace(*line); line++);
	if (*line) {
	    args = line;
    	    i = strlen(args);
	    while (i > 0 && isspace(args[i-1]))
		args[--i] = '\0';
	}
    }

    cmd = find_command(name);
    if (cmd != NULL) {
	cmd->handler(name, args);
    } else {
	screen_putf(_("%s: Unknown command.\n"), quotearg(name));
    }
}

static void
cmd_status(char *cmd, char *args)
{
    char *fmt1;
    HMapIterator it;
    uint32_t c;

    switch (hub_state) {
    case DC_HUB_DISCONNECTED:
	screen_putf(_("Hub state: %s\n"), _("Not connected"));
	break;
    case DC_HUB_CONNECT:
	screen_putf(_("Hub state %s\n"), _("Waiting for complete connection"));
	break;
    case DC_HUB_LOCK:
	screen_putf(_("Hub state: %s\n"), _("Waiting for $Lock"));
	break;
    case DC_HUB_HELLO:
	screen_putf(_("Hub state: %s\n"), _("Waiting for $Hello"));
	break;
    case DC_HUB_LOGGED_IN:
	screen_putf(_("Hub state: %s\n"), _("Logged in"));
	break;
    }

    if (hub_state >= DC_HUB_LOGGED_IN) {
        screen_putf(_("Hub users: %s\n"), uint32_str(hmap_size(hub_users)));
    } else {
        screen_putf(_("Hub users: %s\n"), _("(not logged in)"));
    
    }

    screen_putf(_("Pending user connections:\n"));
    hmap_iterator(pending_userinfo, &it);
    for (c = 0; it.has_next(&it); c++)
        screen_putf("  %s\n", quotearg(it.next(&it)));
    if (c == 0)
        screen_putf(_("  (none)\n"));

    fmt1 = ngettext("byte", "bytes", my_share_size);
    screen_putf(_("Share size: %" PRId64" %s (%" PRId64 " MB).\n"),
                my_share_size, fmt1, my_share_size/(1024*1024));
}

static void
cmd_exit(char *cmd, char *args)
{
    running = false;
}

static void
cmd_say(char *cmd, char *args)
{
    char *msg;

    if (args == NULL) {
    	screen_putf(_("Usage: %s MESSAGE..\n"), cmd);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    msg = escape_message(args);
    hub_putf("<%s> %s|", quotearg_n(0, my_nick), quotearg_n(1, msg)); /* Ignore error */
    free(msg);
}

static void
cmd_msg(char *cmd, char *args)
{
    char *msg;
    char *user;
    DCUserInfo *ui;

    if (args != NULL) {
	user = args;
	for (; *args && !isspace(*args); args++);
	if (!*args) {
	    args = NULL;
    	} else {
	    *args = '\0';
	    for (args++; isspace(*args); args++);
	    if (!*args)
	    	args = NULL;
	}
    }
    if (args == NULL) {
    	screen_putf(_("Usage: %s USER MESSAGE..\n"), cmd);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }

    ui = hmap_get(hub_users, user);
    if (ui == NULL) {
    	screen_putf(_("%s: No such user on this hub\n"), quotearg(user));
	return;
    }

    msg = escape_message(args);
    hub_putf("$To: %s From: %s $<%s> %s|", ui->nick, my_nick, my_nick, msg); /* Ignore error */
    free(msg);
}

static void
cmd_raw(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Usage: %s DATA...\n"), cmd);
	return;
    }
    if (hub_state < DC_HUB_LOCK) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    hub_putf("%s", args);
}

static void
cmd_connect(char *cmd, char *args)
{
    char *portstr;
    struct sockaddr_in addr;

    if (args == NULL) {
    	screen_putf(_("Usage: %s HOST[:PORT]\n"), cmd);
	return;
    }
    if (hub_state != DC_HUB_DISCONNECTED) {
    	screen_putf(_("Already connected or connecting, disconnect first.\n"));
	return;
    }

    portstr = strchr(args, ':');
    if (portstr != NULL) {
    	*portstr = '\0';
    	if (!parse_uint16(portstr+1, &addr.sin_port)) {
	    screen_putf(_("Invalid port number `%s'\n"), quotearg(portstr));
	    return;
	}
	addr.sin_port = htons(addr.sin_port);
    } else {
    	addr.sin_port = htons(DC_HUB_TCP_PORT);
    }

    if (!inet_aton(args, &addr.sin_addr)) {
    	struct hostent *he;

	screen_putf(_("Looking up IP address for %s\n"), quotearg(args));
	he = gethostbyname(args);
	if (he == NULL) {
    	    screen_putf(_("%s: Cannot look up address - %s\n"), quotearg(args), hstrerror(h_errno));
	    return;
	}
    	addr.sin_addr = *(struct in_addr *) he->h_addr;
    }

    addr.sin_family = AF_INET;

    hub_connect(&addr); /* Ignore errors */
}

static void
cmd_disconnect(char *cmd, char *args)
{
    if (hub_state == DC_HUB_DISCONNECTED) {
    	warn(_("Not connected.\n"));
    } else {
    	warn(_("Disconnecting from hub.\n"));
	hub_disconnect();
    }
}

static void
cmd_grantslot(char *cmd, char *args)
{
    DCUserInfo *ui;

    if (args == NULL) {
    	screen_putf(_("Usage: %s USER\n"), cmd);
	return;
    }
    if (hub_state != DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    ui = hmap_get(hub_users, args);
    if (ui == NULL) {
    	screen_putf(_("%s: No such user on this hub\n"), quotearg(args));
	return;
    }
    ui->slot_granted = true;
}

void
browse_none(void)
{	
    /* Clean up previous browse. */
    if (browse_list != NULL) {
	filelist_free(browse_list);
	browse_list = NULL;
	free(browse_path);
	browse_path = NULL;
	free(browse_path_previous);
	browse_path_previous = NULL;
    }
    if (browse_user != NULL) {
	user_info_free(browse_user);
	browse_user = NULL;
    }
    browsing_myself = false;
}

static void
cmd_browse(char *cmd, char *args)
{
    DCUserInfo *ui;
    char *filename;
    struct stat st;

    if (args == NULL) {
    	if (!browsing_myself && browse_user == NULL) {
	    screen_putf(_("Not browsing any user.\n"));
	    return;
	}
	browse_none();
	update_prompt();
	return;
    }

    if (strcmp(my_nick, args) == 0) {
	browse_none();
	browse_list = our_filelist;
	browse_path = xstrdup("/");
	browse_path_previous = NULL;
	browse_user = NULL;
	browsing_myself = true;
	update_prompt();
	return;
    }

    ui = hmap_get(hub_users, args);
    if (ui == NULL) {
        screen_putf(_("%s: No such user on this hub\n"), quotearg(args));
        return;
    }

    filename = xasprintf("%s/%s.MyList.DcLst", listing_dir, ui->nick);
    if (stat(filename, &st) < 0) {
	if (errno != ENOENT) {
            screen_putf(_("%s: Cannot get file status - %s\n"), quotearg(filename), errstr);
            free(filename);
            return;
	}

        free(filename);
	if (ptrv_find(ui->download_queue, "/MyList.DcLst", (comparison_fn_t) queued_file_cmp) < 0) {
	    DCQueuedFile *queued = xmalloc(sizeof(DCQueuedFile));

	    queued->filename = xstrdup("/MyList.DcLst");
	    queued->base_path = xstrdup("/");
	    queued->flag = DC_TF_LIST;
	    ptrv_prepend(ui->download_queue, queued);
	}
    	if (!has_user_conn(ui, DC_DIR_RECEIVE) && ui->conn_count < DC_USER_MAX_CONN)
	    hub_connect_user(ui); /* Ignore errors */
	else
	    screen_putf(_("No free connections. Queued file for download.\n"));

	browse_none();
	browse_user = ui;
	browsing_myself = false;
	ui->refcount++;
	update_prompt();
	return;
    }

    browse_none();
    browse_list = filelist_open(filename);
    browse_path = xstrdup("/");
    browse_path_previous = NULL;
    browse_user = ui;
    browsing_myself = false;
    ui->refcount++;
    free(filename);
    update_prompt();
}

static void
cmd_pwd(char *cmd, char *args)
{
    if (browse_list == NULL) {
	if (browse_user == NULL) {
	    screen_putf(_("Not browsing any user.\n"));
	} else {
	    screen_putf(_("(%s) Waiting for file list.\n"), quotearg(browse_user->nick));
	}
    } else {
	screen_putf(_("(%s) %s\n"),
		quotearg_n(0, browsing_myself ? my_nick : browse_user->nick),
		quotearg_n(1, browse_path));
    }
}

static void
cmd_cd(char *cmd, char *args)
{
    DCFileList *node;
    char *new_path;
    char *path;

    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }

    new_path = get_word_dequoted(args, 0);
    if (new_path == NULL) {
	path = xstrdup("/");
    } else if (strcmp(new_path, "-") == 0) {
	if (browse_path_previous == NULL) {
	    warn(_("No previous path.\n"));
	} else {
	    swap(browse_path, browse_path_previous);
	    update_prompt();
	}
	return;
    } else {
	path = apply_cwd(new_path);
	free(new_path);
    }

    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
    	screen_putf(_("%s: No such file or directory\n"), quotearg(args));
	free(path);
	return;
    }
    if (node->type != DC_TYPE_DIR) {
    	screen_putf(_("%s: Not a directory\n"), quotearg(args));
	free(path);
	return;
    }
    free(browse_path_previous);
    browse_path_previous = browse_path;
    browse_path = filelist_get_path(node);
    update_prompt();
}

static void
cmd_find(char *cmd, char *args)
{
    char *path;
    char *arg;
    DCFileList *node;

    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }
    arg = get_word_dequoted(args, 0);
    if (arg != NULL) {
	path = apply_cwd(arg);
	free(arg);
    } else {
	path = xstrdup(browse_path);
    }

    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
	screen_putf(_("%s: No such file or directory\n"), quotearg(path));
	free(path);
	return;
    }
    filelist_list_recursively(node, "", 1);
    free(path);
}

static void
cmd_ls(char *cmd, char *args)
{
    char *arg;
    char *path = NULL;
    DCFileList *node;
    bool long_mode = false;

    /* XXX: this code can probably be cleaned up a little...
     * split into a list and call getopt?
     */

    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }
    arg = get_word_dequoted(args, 0);
    if (arg != NULL) {
	if (strcmp(arg, "-l") == 0) {
	    long_mode = true;
	    free(arg);
	    arg = get_word_dequoted(args, 1);
	    if (arg != NULL) {
		path = apply_cwd(arg);
		free(arg);
	    }
	} else {
    	    path = apply_cwd(arg);
	    free(arg);
	    arg = get_word_dequoted(args, 1);
	    if (arg != NULL && strcmp(arg, "-l") == 0) {
		long_mode = true;
		free(arg);
	    }
	}
    }
    if (path == NULL)
	path = xstrdup(browse_path);

    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
	screen_putf(_("%s: No such file or directory\n"), quotearg(path));
	free(path);
	return;
    }
    filelist_list(node, long_mode);
    free(path);
}

/* XXX: move to download.c? */
static bool
append_download_dir(DCUserInfo *ui, DCFileList *node, char *base_path, uint64_t *bytes, uint32_t *files)
{
    if (node->type == DC_TYPE_REG) {
	DCQueuedFile *queued;
    	char *path = filelist_get_path(node);

	if (ptrv_find(ui->download_queue, path, (comparison_fn_t) queued_file_cmp) >= 0) {
	    screen_putf(_("Queue already contains this file, ignoring\n"));
	    free(path);
	    return false;
	}

	queued = xmalloc(sizeof(DCQueuedFile));
	queued->filename = path;
	queued->base_path = xstrdup(base_path);
	queued->flag = DC_TF_NORMAL;
	ptrv_append(ui->download_queue, queued);

	(*bytes) += node->u.reg.size;
	(*files) ++;
	return true;
    }
    if (node->type == DC_TYPE_DIR) {
    	HMapIterator it;
	bool s = false;

	hmap_iterator(node->u.dir.children, &it);
	while (it.has_next(&it))
	    s = append_download_dir(ui, it.next(&it), base_path, bytes, files) || s;
	return s;
    }
    return false;
}


static void
cmd_get(char *cmd, char *args)
{
    DCFileList *node;
    char *path;
    char *file_name;
    uint64_t bytes = 0;
    uint32_t files = 0;

    /* Note: This command assumes that we are browsing some user.
     * If we in the future would allow browse-less getting,
     * i.e. get NICK FILE, then we must here add additional checks
     * found in cmd_browse, such as strcmp(my_nick, nick)==0 etc.
     */

    file_name = get_word_dequoted(args, 0);
    if (file_name == NULL) {
    	screen_putf(_("Usage: %s FILE\n"), cmd);
	return;
    }
    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }
    if (browsing_myself) {
	screen_putf(_("Cannot download files from myself.\n"));
	return;
    }

    path = apply_cwd(file_name);
    free(file_name);
    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
	screen_putf(_("%s: No such file or directory\n"), quotearg(path));
	free(path);
	return;
    }
    free(path);

    if (append_download_dir(browse_user, node, browse_path, &bytes, &files)) {
        char *fmt1 = ngettext("byte", "bytes", bytes);
        char *fmt2 = ngettext("file", "files", files);
    	path = filelist_get_path(node);
	screen_putf(_("Downloading %s (%" PRId64 " %s in %d %s)\n"), quotearg(path), bytes, fmt1, files, fmt2);
	free(path);
    } else {
        /* XXX: hmm, shouldn't we return even if queue is non-empty? */
    	if (browse_user->download_queue->cur == 0) {
	    screen_putf(_("No files to download (empty directories).\n"));
	    return;
	}
    }

    if (!has_user_conn(browse_user, DC_DIR_RECEIVE) && browse_user->conn_count < DC_USER_MAX_CONN) {
    	hub_connect_user(browse_user); /* Ignore errors */
    } else {
	screen_putf(
          ngettext("No free connections. Queued file for download.\n",
	  	   "No free connections. Queued files for download.\n", files));
    }
}

static void
cmd_queue(char *cmd, char *args)
{
    uint32_t c;
    DCUserInfo *ui;

    if (args == NULL) {
    	screen_putf(_("Usage: %s USER\n"), cmd);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    ui = hmap_get(hub_users, args);
    if (ui == NULL) {
        screen_putf(_("%s: No such user on this hub\n"), quotearg(args));
	return;
    }
    for (c = 0; c < ui->download_queue->cur; c++) {
	DCQueuedFile *queued = ui->download_queue->buf[c];
	screen_putf(_("%d. [ %s ] %s\n"),
		c+1,
		quotearg_n(0, queued->base_path),
		quotearg_n(1, queued->filename + strlen(queued->base_path)));
    }
}

static void
removed_queued_by_range(uint32_t sp, uint32_t ep, void *userdata)
{
    DCUserInfo *user = userdata;
    uint32_t c;

    for (c = sp-1; c < ep; c++) {
	if (user->download_queue->buf[c] != NULL) {
	    free_queued_file(user->download_queue->buf[c]);
	    user->download_queue->buf[c] = NULL;
	}
    }
}

static void
compact_queue_ptrv(DCUserInfo *user)
{
    int32_t first_free = -1;
    uint32_t c;

    for (c = 0; c < user->download_queue->cur; c++) {
	if (user->download_queue->buf[c] == NULL) {
	    if (first_free == -1)
		first_free = c;
	} else {
	    if (first_free != -1) {
		user->download_queue->buf[first_free] = user->download_queue->buf[c];
		user->download_queue->buf[c] = NULL;
		for (first_free++; first_free < c; first_free++) {
		    if (user->download_queue->buf[first_free] == NULL)
			break;
		}
	    }
	}
    }

    if (first_free != -1)
	user->download_queue->cur = first_free;
}    

static void
cmd_unqueue(char *cmd, char *args)
{
    DCUserInfo *user;
    char *range;

    if (args == NULL) {
    	screen_putf(_("Usage: %s USER [RANGE]\n"), cmd);
	return;
    }

    /* XXX: parse each argument, allow RANGE RANGE2 .. */
    range = strchr(args, ' ');
    if (range != NULL) {
	*range = '\0';
	for (range++; isspace(*range); range++);
	if (*range == '\0')
	    range = "1-";
    } else {
	range = "1-";
    }

    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    user = hmap_get(hub_users, args);
    if (user == NULL) {
        screen_putf(_("%s: No such user on this hub\n"), quotearg(args));
        return;
    }
    if (!foreach_in_range(range, 1, user->download_queue->cur, NULL, NULL)) {
	screen_putf(_("%s: Invalid range, or index out of range (1-%d)\n"), quotearg(range), user->download_queue->cur);
	return;
    }
    foreach_in_range(range, 1, user->download_queue->cur, removed_queued_by_range, user);
    compact_queue_ptrv(user);
}

static int
user_info_compare(const void *i1, const void *i2)
{
    const DCUserInfo *f1 = *(const DCUserInfo **) i1;
    const DCUserInfo *f2 = *(const DCUserInfo **) i2;

    return strcmp(f1->nick, f2->nick);
}

#define IFNULL(a,x) ((a) == NULL ? (x) : (a))

static void
cmd_who(char *cmd, char *args)
{
    uint32_t maxlen;
    HMapIterator it;
    uint32_t c;
    DCUserInfo **items;
    uint32_t count;
    uint32_t cols;
    StrBuf *out;

    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }

    if (args != NULL) {
    	DCUserInfo *ui;

	ui = hmap_get(hub_users, args);
	if (ui == NULL) {
    	    screen_putf(_("%s: No such user on this hub\n"), quotearg(args));
	} else {
	    char *fmt1 = ngettext("byte", "bytes", ui->share_size);
    	    screen_putf(_("Nick: %s\n"), quotearg(ui->nick));
    	    screen_putf(_("Description: %s\n"), quotearg(IFNULL(ui->description, "")));
    	    screen_putf(_("Speed: %s\n"), quotearg(IFNULL(ui->speed, "")));
    	    screen_putf(_("Level: %d\n"), ui->level);
    	    screen_putf(_("E-mail: %s\n"), quotearg(IFNULL(ui->email, "")));
    	    screen_putf(_("Operator: %d\n"), ui->is_operator);
    	    screen_putf(_("Share Size: %" PRId64 " %s (%" PRId64 " MB)\n"),
    	        ui->share_size, fmt1, ui->share_size/(1024*1024));
	}
    	return;
    }

    maxlen = 0;
    for (hmap_iterator(hub_users, &it); it.has_next(&it); ) {
    	DCUserInfo *ui = it.next(&it);
	maxlen = max(maxlen, strlen(quotearg(ui->nick)));
    }

    count = hmap_size(hub_users);
    items = xmalloc(count * sizeof(DCUserInfo *));
    hmap_iterator(hub_users, &it);
    for (c = 0; c < count; c++)
	items[c] = it.next(&it);
    qsort(items, count, sizeof(DCUserInfo *), user_info_compare);

    screen_get_size(NULL, &cols);

    out = strbuf_new();
    for (c = 0; c < count; c++) {
    	DCUserInfo *ui = items[c];
	char *nick = quotearg(ui->nick);

	strbuf_clear(out);
	strbuf_append(out, nick);
	strbuf_append_char_n(out, maxlen+1-strlen(nick), ' ');
	strbuf_appendf(out, "  %7" PRId64 "M", ui->share_size / (1024*1024));
	strbuf_append(out, ui->is_operator ? " op" : "   ");
	if (ui->download_queue->cur > 0)
	    strbuf_appendf(out, " (%3d)", ui->download_queue->cur);
	else
	    strbuf_append(out, "      ");
	strbuf_appendf(out, " %s", quotearg(ui->description ? ui->description : ""));
	if (strbuf_length(out) > cols)
	    strbuf_set_length(out, cols);
	screen_putf("%s\n", strbuf_buffer(out));
    }
    free(items);
    strbuf_free(out);
}

static void
cmd_transfers(char *cmd, char *args)
{
    char *format;
    HMapIterator it;
    uint32_t maxlen = 0;
    time_t now = 0;

    hmap_iterator(user_conns, &it);
    while (it.has_next(&it)) {
    	DCUserConn *uc = it.next(&it);
	maxlen = max(maxlen, strlen(uc->name));
    }

    format = xasprintf("%%-%ds  %%s\n", maxlen);

    if (time(&now) == (time_t) -1) {
    	warn(_("Cannot get current time - %s\n"), errstr);
	return;
    }

    hmap_iterator(user_conns, &it);
    while (it.has_next(&it)) {
    	DCUserConn *uc = it.next(&it);
	char *status;

    	/*if (!get_user_conn_status(uc))
	    continue;*/
	status = user_conn_status_to_string(uc, now);
	screen_putf(format, quotearg(uc->name), status);
	free(status);
    }

    screen_putf(_("Upload slots: %d/%d  Download slots: %d/unlimited\n"), used_ul_slots, my_ul_slots, used_dl_slots);

    free(format);
}

static void
cmd_cancel(char *cmd, char *args)
{
    DCUserConn *uc;

    if (args == NULL) {
    	screen_putf(_("Usage: %s CONNECTION\n"), cmd);
	return;
    }

    uc = hmap_get(user_conns, args);
    if (uc == NULL) {
    	screen_putf(_("%s: No such user connection.\n"), quotearg(args));
	return;
    }

    user_conn_cancel(uc);
}

static void
cmd_search(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Usage: %s STRING...\n"), cmd);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    add_search_request(args); /* Ignore errors */
}

static void
cmd_results(char *cmd, char *args)
{
    uint32_t c;

    if (args == NULL) {
        time_t now;
    
	if (time(&now) == (time_t) -1) {
            warn(_("Cannot get current time - %s\n"), errstr);
            return;
        }

    	for (c = 0; c < our_searches->cur; c++) {
    	    DCSearchRequest *sd = our_searches->buf[c];
    	    char *status;
    	    char *spec;

            spec = search_selection_to_string(&sd->selection);
            status = sd->issue_time + SEARCH_TIME_THRESHOLD <= now
                   ? _("Closed") : _("Open");
	    screen_putf(_("%d. %s (%s) Results: %d\n"), c+1, quotearg(spec),
                   status, sd->responses->cur);
	}
    } else {
    	DCSearchRequest *sd;
	if (!parse_uint32(args, &c) || c == 0 || c-1 >= our_searches->cur) {
    	    screen_putf(_("Invalid search index.\n"));
	    return;
	}
	sd = our_searches->buf[c-1];
	for (c = 0; c < sd->responses->cur; c++) {
	    DCSearchResponse *sr = sd->responses->buf[c];
	    char *n;
	    char *t;

	    n = translate_remote_to_local(sr->filename);
	    if (sr->filetype == DC_TYPE_DIR) /* XXX: put into some function */
	    	t = "/";
	    else
	    	t = "";
	    screen_putf("%d. %s %s%s\n", c+1, quotearg(sr->userinfo->nick), n, t);
	    free(n);
	}
    }
}

static void
cmd_unsearch(char *cmd, char *args)
{
    DCSearchRequest *sd;
    uint32_t index;

    if (args == NULL) {
    	screen_putf(_("Usage: %s INDEX\n"), cmd);
	return;
    }
    if (!parse_uint32(args, &index) || index == 0 || index-1 >= our_searches->cur) {
    	screen_putf(_("Invalid search index.\n"));
	return;
    }
    
    sd = our_searches->buf[index-1];
    ptrv_remove_range(our_searches, index-1, index);
    free_search_request(sd);    
}

void
update_prompt(void)
{
    if (browsing_myself || browse_user != NULL) {
	char *nick = browsing_myself ? my_nick : browse_user->nick;
	
	if (browse_list == NULL) {
	    set_screen_prompt("%s:(%s)> ", PACKAGE, quotearg(nick));
	} else {
	    set_screen_prompt("%s:%s:%s> ", PACKAGE, quotearg_n(0, nick), quotearg_n(1, browse_path));
	}
    } else {
	set_screen_prompt("%s> ", PACKAGE);
    }
}
