/* fs.c - Local and remote file system management
 *
 * Copyright (C) 2004, 2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <config.h>
#include <assert.h>		/* ? */
#include <sys/types.h>		/* ? */
#include <sys/stat.h>		/* ? */
#include <unistd.h>		/* POSIX */
#include <fcntl.h>		/* ? */
#include <stdlib.h>		/* C89 */
#include <dirent.h>		/* ? */
#include <sys/types.h>		/* ? */
#include <inttypes.h>		/* ? */
#include "full-read.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "minmax.h"		/* Gnulib */
#include "full-write.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "quotearg.h"		/* Gnulib */
#include "fnmatch.h"		/* Gnulib */
#include "dirname.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "common/minmaxonce.h"
#include "common/error.h"
#include "common/intutil.h"
#include "common/strbuf.h"
#include "common/strleftcmp.h"
#include "common/substrcmp.h"
#include "common/comparison.h"
#include "microdc.h"

#define MAX_DIR_DEPTH 32
#define IS_OCT_DIGIT(d) ( (d) >= '0' && (d) <= '7' )

typedef struct _DCFileListIterator DCFileListIterator;

struct _DCFileListIterator {
    DCFileList *node;
    HMapIterator it;
    uint32_t c;
};

DCFileList *our_filelist = NULL;

static int
fs_completion_entry_compare(const void *e1, const void *e2)
{  
    const DCCompletionEntry *ce1 = *(const DCCompletionEntry **) e1;
    const DCCompletionEntry *ce2 = *(const DCCompletionEntry **) e2;
    if (ce1->sorting.file_type == ce2->sorting.file_type) {
        char *s1 = xasprintf(ce1->display_fmt, ce1->display); /* XXX: this is really slow */
        char *s2 = xasprintf(ce2->display_fmt, ce2->display); /* XXX: this is really slow */
        int cmp = strcmp(s1, s2);
        free(s1);
        free(s2);
        return cmp;
    }
    return ce1->sorting.file_type - ce2->sorting.file_type;
}   

static DCFileList *
new_file_node(const char *name, DCFileType type, DCFileList *parent)
{
    DCFileList *node;

    node = xmalloc(sizeof(DCFileList));
    node->name = xstrdup(name);
    node->type = type;
    node->parent = parent;
    if (parent != NULL)
        hmap_put(parent->u.dir.children, node->name, node);
    switch (node->type) {
    case DC_TYPE_DIR:
    	node->u.dir.children = hmap_new();
	node->u.dir.totalsize = 0;
	break;
    case DC_TYPE_REG:
    	node->u.reg.size = 0;
    default:
    	break;
    }

    return node;
}

static DCFileList *
get_child_node(DCFileList *node, const char *path)
{
    if (IS_CURRENT_DIR(path))
        return node;
    if (IS_PARENT_DIR(path))
        return node->parent == NULL ? node : node->parent;
    return hmap_get(node->u.dir.children, path);
}

void
filelist_free(DCFileList *node)
{
    switch (node->type) {
    case DC_TYPE_REG:
    	break;
    case DC_TYPE_DIR:
        hmap_foreach_value(node->u.dir.children, filelist_free);
	hmap_free(node->u.dir.children);
    	break;
    }
    free(node->name);
    free(node);
}

DCFileList *
parse_decoded_dclst(char *decoded, uint32_t decoded_len)
{
    DCFileList *node;
    uint32_t c;
    PtrV *dirs;

    dirs = ptrv_new();
    node = new_file_node("", DC_TYPE_DIR, NULL);
    ptrv_append(dirs, node);
    for (c = 0; c < decoded_len; c++) {
    	DCFileList *oldnode;
	char *name;
    	int depth;

	for (; c < decoded_len && decoded[c] == '\n'; c++);
    	depth = 1;
	for (; c < decoded_len && decoded[c] == '\t'; c++)
	    depth++;
	if (c >= decoded_len)
	    break; /* Premature end */
	if (decoded[c] == '\r') {
	    c++; /* skip LF */
	    continue; /* Skipping bad line */
	}

    	name = decoded + c;
    	for (; c < decoded_len && decoded[c] != '\r' && decoded[c] != '|'; c++);
	if (c >= decoded_len)
	    break;

	if (depth < dirs->cur)
	    ptrv_remove_range(dirs, depth, dirs->cur);

	oldnode = dirs->buf[dirs->cur-1];
	if (decoded[c] == '|') {
	    char *sizestr;
	    uint64_t size;

	    decoded[c] = '\0';
    	    sizestr = decoded+c+1;
    	    for (c++; c < decoded_len && decoded[c] != '\r'; c++);
	    if (c >= decoded_len)
	    	break; /* Premature end */
	    decoded[c] = '\0';
    	    c++; /* skip LF */
	    if (!parse_uint64(sizestr, &size))
	    	continue; /* Skipping bad line */

	    node = new_file_node(name, DC_TYPE_REG, oldnode);
	    node->u.reg.size = size;
	} else {
	    decoded[c] = '\0';
    	    c++; /* skip LF */
	    node = new_file_node(name, DC_TYPE_DIR, oldnode);
	}

	if (node->type == DC_TYPE_REG) {
	    DCFileList *t;
	    for (t = oldnode; t != NULL; t = t->parent)
	    	t->u.dir.totalsize += node->u.reg.size;
	}
	if (node->type == DC_TYPE_DIR)
	    ptrv_append(dirs, node);
    }

    node = dirs->buf[0];
    ptrv_free(dirs); /* ignore non-empty */
    return node;
}

DCFileList *
filelist_lookup(DCFileList *node, const char *filename)
{
    const char *end;
    char *name;

    if (*filename != '/')
    	return NULL;
    for (filename++; *filename == '/'; filename++);
    end = strchr(filename, '/');
    if (end == NULL) {
	if (!*filename)
	    return node;
    	end = filename + strlen(filename);
    }
    if (node->type != DC_TYPE_DIR)
    	return NULL;

    name = xstrndup(filename, end-filename); /* XXX: strndupa? */
    node = get_child_node(node, name);
    free(name);
    if (node != NULL)
	return (*end == '\0' ? node : filelist_lookup(node, end));

    return NULL;
}

char *
filelist_get_path_with_trailing_slash(DCFileList *node)
{
    StrBuf *sb;

    sb = strbuf_new();
    while (node->parent != NULL) {
	strbuf_prepend(sb, node->name);
	strbuf_prepend_char(sb, '/');
    	node = node->parent;
    }
    if (node->type == DC_TYPE_DIR)
        strbuf_append_char(sb, '/');

    return strbuf_free_to_string(sb);
}

/* Return the path of the node relative to the share root directory.
 */
char *
filelist_get_path(DCFileList *node)
{
    StrBuf *sb;

    if (node->parent == NULL)
	return xstrdup("/"); /* root */

    sb = strbuf_new();
    while (node->parent != NULL) {
	strbuf_prepend(sb, node->name);
	strbuf_prepend_char(sb, '/');
    	node = node->parent;
    }

    return strbuf_free_to_string(sb);
}

/* Return the physical path of the node, by prepending the share directory path.
 */
static char *
filelist_get_real_path(DCFileList *node)
{
    char *p1;
    char *p2;

    p1 = filelist_get_path(node);
    p2 = catfiles(share_dir, p1+1); /* p1[0] == '/', we don't want that */
    free(p1);
    return p2;
}

/*static int
filelist_completion_compare(const void *i1, const void *i2)
{
    const DCCompletionEntry *ce1 = *(const DCCompletionEntry **) i1;
    const DCCompletionEntry *ce2 = *(const DCCompletionEntry **) i2;
    return strcmp(ce1->input, ce2->input);
}
*/

static int
file_node_compare(const void *i1, const void *i2)
{
    const DCFileList *f1 = *(const DCFileList **) i1;
    const DCFileList *f2 = *(const DCFileList **) i2;

    if (f1->type != f2->type)
	return f1->type - f2->type;

    return strcmp(f1->name, f2->name);
}

static DCFileList **
get_sorted_file_list(DCFileList *node, uint32_t *out_count)
{
    HMapIterator it;
    DCFileList **items;
    uint32_t count;
    uint32_t c;

    assert(node->type == DC_TYPE_DIR);
    count = hmap_size(node->u.dir.children);
    items = xmalloc((count+1) * sizeof(DCFileList *));
    hmap_iterator(node->u.dir.children, &it);
    for (c = 0; c < count; c++)
        items[c] = it.next(&it);
    items[count] = NULL;
    qsort(items, count, sizeof(DCFileList *), file_node_compare);

    if (out_count != NULL)
        *out_count = count;

    return items;
}

void
filelist_list_recursively(DCFileList *node, char *basepath)
{
    if (node->type == DC_TYPE_DIR) {
	DCFileList **items;
	uint32_t c;

        items = get_sorted_file_list(node, NULL);
	for (c = 0; items[c] != NULL; c++) {
	    char *path = catfiles(basepath, items[c]->name);
            filelist_list_recursively(items[c], path);
            free(path);
        }
	free(items);
    } else {
	screen_putf("%7" PRIu64 "M %s\n", (uint64_t) (node->u.reg.size/(1024*1024)), quotearg(basepath)); /* " */
    }
}

void
filelist_list(DCFileList *node, bool long_mode)
{
    uint32_t maxlen;
    uint64_t maxsize;

    if (node->type == DC_TYPE_DIR) {
        HMapIterator it;

	maxlen = 0;
	maxsize = 0;
	hmap_iterator(node->u.dir.children, &it);
	while (it.has_next(&it)) {
    	    DCFileList *subnode = it.next(&it);

	    switch (subnode->type) {
	    case DC_TYPE_REG:
		maxsize = max(maxsize, subnode->u.reg.size);
        	maxlen = max(maxlen, strlen(subnode->name));
        	break;
            case DC_TYPE_DIR:
		maxsize = max(maxsize, subnode->u.dir.totalsize);
        	maxlen = max(maxlen, strlen(subnode->name)+1);
        	break;
            }
	}
    } else {
    	maxsize = node->u.reg.size;
    	maxlen = strlen(node->name);
    }

    if (long_mode) {
        char *format;

        format = xasprintf("%%%d" PRIu64 "M %%s%%s\n", ilog10(max(1, maxsize/(1024*1024))));
	if (node->type == DC_TYPE_DIR) {
	    DCFileList **items;
	    uint32_t c;

            items = get_sorted_file_list(node, NULL);
	    for (c = 0; items[c] != NULL; c++) {
		switch (items[c]->type) {
		case DC_TYPE_REG:
    		    screen_putf(format, (uint64_t) (items[c]->u.reg.size/(1024*1024)), quotearg(items[c]->name), "");
		    break;
		case DC_TYPE_DIR:
    		    screen_putf(format, (uint64_t) (items[c]->u.dir.totalsize/(1024*1024)), quotearg(items[c]->name), "/");
		    break;
		}
	    }
	    free(items);
	} else {
	    screen_putf(format, (uint64_t) (node->u.reg.size/(1024*1024)), quotearg(node->name), "");
	}
	free(format);
    } else {
    	if (node->type == DC_TYPE_DIR) {
	    DCFileList **items;
            int cols;
	    int rows;
	    int row;
            int per_row;
            uint32_t count;

            items = get_sorted_file_list(node, &count);
            screen_get_size(NULL, &cols);
            per_row = MAX(1, (cols+2)/(maxlen+2));
	    rows = (count/per_row) + (count%per_row != 0);

	    for (row = 0; row < rows; row++) {
		uint32_t c;

		for (c = row; c < count; c += rows) {
		    DCFileList *item = items[c];
		    int extlen = 0;
		    int d;
    
		    switch (item->type) {
		    case DC_TYPE_REG:
			extlen = 0;
			screen_putf("%s", quotearg(item->name));
			break;
		    case DC_TYPE_DIR:
			extlen = 1;
			screen_putf("%s/", quotearg(item->name));
			break;
		    }
		    if (c+rows < count) {
			for (d = maxlen-strlen(items[c]->name)-extlen+2; d > 0; d--)
			    screen_putf(" ");
		    }
		}
		screen_putf("\n");
	    }
	    free(items);
	} else {
    	    screen_putf("%s\n", quotearg(node->name));
	}
    }
}

DCFileList *
filelist_open(const char *filename)
{
    struct stat st;
    uint8_t *contents;
    char *decoded;
    uint32_t decoded_len;
    int fd;
    DCFileList *root;
    ssize_t res;

    if (stat(filename, &st) < 0) {
    	screen_putf(_("%s: Cannot get file status - %s\n"), quotearg(filename), errstr);
	return NULL;
    }
    contents = malloc(st.st_size);
    if (contents == NULL) {
    	screen_putf("%s: %s\n", quotearg(filename), errstr);
	return NULL;
    }
    fd = open(filename, O_RDONLY);
    if (fd < 0) {
    	screen_putf(_("%s: Cannot open file for reading - %s\n"), quotearg(filename), errstr);
	return NULL;
    }
    res = full_read(fd, contents, st.st_size);
    if (res < st.st_size) {
    	if (res < 0)
    	    screen_putf(_("%s: Cannot read from file - %s\n"), quotearg(filename), errstr);
	else
	    screen_putf(_("%s: Premature end of file\n"), quotearg(filename));	/* XXX: really: file was truncated? */
	free(contents);
	if (close(fd) < 0)
    	    screen_putf(_("%s: Cannot close file - %s\n"), quotearg(filename), errstr);
	return NULL;
    }
    if (close(fd) < 0)
    	screen_putf(_("%s: Cannot close file - %s\n"), quotearg(filename), errstr);
    decoded = huffman_decode(contents, st.st_size, &decoded_len);
    free(contents);
    if (decoded == NULL) {
    	screen_putf(_("%s: Invalid data, cannot decode\n"), quotearg(filename));
	return NULL;
    }
    root = parse_decoded_dclst(decoded, decoded_len);
    free(decoded);
    return root;
}

static void
dir_to_filelist(DCFileList *parent, const char *path)
{
    struct dirent *ep;
    DIR *dp;

    dp = opendir(path);
    if (dp == NULL) {
    	screen_putf(_("%s: Cannot open directory - %s\n"), quotearg(path), errstr);
	return;
    }
    errno = 0;
    while ((ep = readdir(dp)) != NULL) {
    	struct stat st;
	char *fullname;

    	if (IS_SPECIAL_DIR(ep->d_name))
	    continue;

	/* If we ran into looped symlinked dirs, stat will stop (errno=ELOOP). */

    	fullname = catfiles(path, ep->d_name);
    	if (stat(fullname, &st) < 0) {
	    screen_putf(_("%s: Cannot get file status - %s\n"), quotearg(fullname), errstr);
	    free(fullname);
	    continue;
	}
	if (S_ISDIR(st.st_mode)) {
	    DCFileList *node = new_file_node(ep->d_name, DC_TYPE_DIR, parent);
	    dir_to_filelist(node, fullname);
	    parent->u.dir.totalsize += node->u.dir.totalsize;
	}
	else if (S_ISREG(st.st_mode)) {
	    DCFileList *node = new_file_node(ep->d_name, DC_TYPE_REG, parent);
	    node->u.reg.size = st.st_size;
	    parent->u.dir.totalsize += node->u.reg.size;
	}
	else {
	    screen_putf(_("%s: Not a regular file or directory, ignoring\n"), quotearg(fullname));
	}
	free(fullname);
    }
    if (errno != 0)
    	screen_putf(_("%s: Cannot read directory - %s\n"), quotearg(path), errstr);
    if (closedir(dp) < 0)
    	screen_putf(_("%s: Cannot close directory - %s\n"), quotearg(path), errstr);
}

static void
filelist_to_string(DCFileList *node, StrBuf *sb, int level)
{
    if (level != 0)
    	strbuf_append_char_n(sb, level-1, '\t');

    if (node->type == DC_TYPE_REG) {
   	strbuf_appendf(sb, "%s|%" PRIu64 "\r\n", node->name, node->u.reg.size); /* " joe sh bug */
    } else {
    	HMapIterator it;

    	if (level != 0)
   	    strbuf_appendf(sb, "%s\r\n", node->name);
	hmap_iterator(node->u.dir.children, &it);
	while (it.has_next(&it)) {
	    DCFileList *subnode = it.next(&it);
            filelist_to_string(subnode, sb, level+1);
	}
    }
}

bool
filelist_create(const char *basedir)
{
    StrBuf *sb;
    char *indata;
    char *outdata;
    int fd;
    char *filename;
    uint32_t len;
    struct stat st;
    int i;
    DCFileList *root;

    root = new_file_node("", DC_TYPE_DIR, NULL);

    sb = strbuf_new();
    if (basedir != NULL) {
    	screen_putf(_("Scanning directory %s\n"), quotearg(basedir));
    	dir_to_filelist(root, basedir);
	filelist_to_string(root, sb, 0);
    }

    len = strbuf_length(sb);
    indata = strbuf_free_to_string(sb);
    outdata = huffman_encode((uint8_t *) indata, len, &len);
    free(indata);

    filename = catfiles(listing_dir, "MyList.DcLst");
    mkdirs_for_file(filename, true); /* Ignore errors */
    if (stat(filename, &st) < 0) {
    	if (errno != ENOENT)
	    warn(_("%s: Cannot get file status - %s\n"), filename, errstr);
    } else {
	if (unlink(filename) < 0)
	    warn(_("%s: Cannot remove file - %s\n"), filename, errstr);
    }
    i = ptrv_find(delete_files, filename, (comparison_fn_t) strcmp);
    if (i >= 0)
    	ptrv_remove_range(delete_files, i, i+1);

    fd = open(filename, O_CREAT|O_EXCL|O_WRONLY, 0666);
    if (fd < 0) {
    	screen_putf(_("%s: Cannot open file for writing - %s\n"), quotearg(filename), errstr);
	filelist_free(root);
	free(outdata);
	free(filename);
	return false;
    }
    if (ptrv_find(delete_files, filename, (comparison_fn_t) strcmp) < 0)
    	ptrv_append(delete_files, xstrdup(filename));

    if (full_write(fd, outdata, len) < len) {
    	screen_putf(_("%s: Cannot write to file - %s\n"), quotearg(filename), errstr);
	filelist_free(root); 
	free(outdata);
	free(filename);
	return false;
    }
    if (close(fd) < 0)
    	screen_putf(_("%s: Cannot close file - %s\n"), quotearg(filename), errstr);
    free(filename);
    free(outdata);

    if (our_filelist != NULL)
    	filelist_free(our_filelist);
    our_filelist = root;
    my_share_size = our_filelist->u.dir.totalsize;

    return true;
}

/* Find the physical path of a file that the remote end
 * wants to download from us.
 */
char *
resolve_upload_file(DCUserInfo *ui, const char *name)
{
    DCFileList *node;

    /* Note: 'name' will have been translated to local slashes
     * by a call to translate_remote_to_local prior to calling
     * this function.
     */

    /* Skip leading slashes, all but one */
    if (name[0] != '/')
	return NULL;
    for (; name[1] == '/'; name++);

    if (strcmp(name, "/MyList.DcLst") == 0)
        return catfiles(listing_dir, name);

    if (our_filelist == NULL || share_dir == NULL)
        return NULL;

    node = filelist_lookup(our_filelist, name);
    if (node == NULL)
        return NULL;
    return filelist_get_real_path(node);
}

char *
resolve_download_file(DCUserInfo *ui, DCQueuedFile *queued)
{
    char *filename;
    char *tmp;

    if (queued->flag == DC_TF_LIST) {
        tmp = xasprintf("%s.%s", ui->nick, base_name(queued->filename));
        filename = catfiles(listing_dir, tmp);
        free(tmp);
	ptrv_append(delete_files, xstrdup(filename));
        mkdirs_for_file(filename, true); /* Ignore errors */
    } else {
        tmp = catfiles(download_dir, queued->filename + strlen(queued->base_path));
        filename = xasprintf("%s.part", tmp);
        free(tmp);
        mkdirs_for_file(filename, false); /* Ignore errors */
    }

    return filename;
}

char *
translate_remote_to_local(const char *remotename)
{
    StrBuf *sb;

    sb = strbuf_new();
    strbuf_append_char(sb, '/');
    for (; *remotename != '\0'; remotename++) {
    	if (*remotename == '\\')
	    strbuf_append_char(sb, '/');
	else
	    strbuf_append_char(sb, *remotename);
    }

    return strbuf_free_to_string(sb);
}

char *
translate_local_to_remote(const char *localname)
{
    StrBuf *sb;

    sb = strbuf_new();
    localname++;
    for (; *localname != '\0'; localname++) {
    	if (*localname == '/')
	    strbuf_append_char(sb, '\\');
	else
	    strbuf_append_char(sb, *localname);
    }

    return strbuf_free_to_string(sb);
}

char *
apply_cwd(const char *path)
{
    if (*path == '/') {
    	return xstrdup(path);
    } else {
    	return catfiles(browse_path, path);
    }
}

/* Concatenate two file name components, taking in consideration trailing
 * slash in the first component P1.
 * An empty file name component ("") can be specified as well. It is a
 * special case which is treated as "no directory component".
 * "." is returned if both P1 and P2 are empty file name components.
 * Note that consecutive sequences of slashes ("/") in P1 and P2 are
 * untouched. Only the trailing slash of P1 is considered when concatenating
 * P1 and P2.
 * The returned value should be freed when no longer needed.
 * P1 or P2 may not be NULL.
 */   
char *
concat_filenames(const char *p1, const char *p2)
{
    size_t l1;
    size_t l2;
    char *out;

    if (*p1 == '\0' && *p2 == '\0')
        return xstrdup(".");
    if (*p1 == '\0')
        return xstrdup(p2);
    if (*p2 == '\0')
        return xstrdup(p1);

    l1 = strlen(p1);
    l2 = strlen(p2);
    if (p1[l1-1] == '/')
        l1--;

    out = xmalloc(l1+1+l2+1);
    memcpy(out, p1, l1);
    out[l1] = '/';
    memcpy(out+l1+1, p2, l2+1);

    return out;
}


/* Return true if the string BUF (which may contain quotes and
 * escapes) starts with a slash.
 * This is equivalent to dequoting BUF, and testing if the first
 * character of the result is '/', only that with this function
 * no memory is actually allocated.
 * This function could also be implemented as
 *   return skip_slashes(&buf, &quoted);
 */
bool
has_leading_slash(const char *buf)
{
    for (; buf[0] == '"'; buf++); /* Begins and ends empty strings */
    return buf[0] == '/' || (buf[0] == '\\' && buf[1] == '/');
}

/* Find the first non-slash character, not including
 * quotes. Return true if a slash was found.
 * Update *BUFPTR to point to the first non-slash character.
 * Also update *QUOTEDPTR to reflect quoted state;
 */
/* rename: skip_leading_slashes */
static bool
skip_slashes(char **bufptr, bool *quotedptr)
{
    char *buf = *bufptr;
    bool quoted = *quotedptr;
    bool slash = false;

    for (; ; buf++) {
        if (buf[0] == '"') {
            quoted = !quoted;
        } else if (buf[0] == '/') {
            slash = true;
        } else if (buf[0] == '\\' && buf[1] == '/') {
            slash = true;
            buf++;
        } else {
            break;
        }
    }    

    *bufptr = buf;
    *quotedptr = quoted;
    return slash;
}  


/* Extract the first file name component of *BUFPTR and place the
 * newly allocated string into *OUTPTR. Update *BUFPTR to point  
 * to the first character not in this component (excluding quotes).
 * Also update *QUOTEDPTR. A file component is not supposed to contain
 * slashes, so all leading slashes of *BUFPTR are ignored. Note that  
 * if *BUFPTR is the empty string (possibly after leading slashes have
 * been ignored), then *OUTPTR will be an empty string. You should
 * free *OUTPTR when it is no longer needed.
 *
 * This function combines many operations:
 *  - stop matching when a slash is encountered
 *  - determine if there are unquoted and unescaped wildcards
 *  - escape quoted wildcards for fnmatch, if there were wildcards
 *  - remove quotes and escapes
 */
static bool
dircomp_to_fnmatch_str(char **bufptr, bool *quotedptr, char **outptr)
{
    StrBuf *out;
    char *buf;  
    bool quoted;
    bool wildcards = false;

    skip_slashes(bufptr, quotedptr);
    out = strbuf_new(); 

    quoted = *quotedptr;
    for (buf = *bufptr; *buf != '\0' && *buf != '/'; buf++) {
        if (*buf == '"')
            quoted = !quoted;
        else if (*buf == '\\' && buf[1] != '\0')
            buf++;
        else if ((*buf == '*' || *buf == '?') && !quoted)
            wildcards = true;
    }

    quoted = *quotedptr;
    for (buf = *bufptr; *buf != '\0' && *buf != '/'; buf++) {
        if (*buf == '"') {
            quoted = !quoted;
        } else if (*buf == '\\') {
            buf++;
            if (*buf == '\0')
                break;

            /*if (*buf == 'a') {
                strbuf_append_char(out, '\a');
            } else if (*buf == 'b') {
                strbuf_append_char(out, '\b');
            } else if (*buf == 'f') {
                strbuf_append_char(out, '\f');
            } else if (*buf == 'n') {
                strbuf_append_char(out, '\n');
            } else if (*buf == 'r') {
                strbuf_append_char(out, '\r');
            } else if (*buf == 't') {
                strbuf_append_char(out, '\t');
            } else if (*buf == 'v') {
                strbuf_append_char(out, '\v');
            } else*/ if (IS_OCT_DIGIT(*buf)) {
                int chr = *buf - '0';
                buf++;
                if (*buf != '\0' && IS_OCT_DIGIT(*buf)) {
                    chr = (chr * 8) + (*buf - '0');    
                    buf++;
                    if (*buf != '\0' && IS_OCT_DIGIT(*buf)) {
                        chr = (chr * 8) + (*buf - '0');    
                        buf++;
                    }
                }
                buf--;
		strbuf_append_char(out, chr);
            } else if (*buf == '*' || *buf == '?') {
                if (quoted || !wildcards)
                    strbuf_append_char(out, '\\'); /* escape wildcard for fnmatch */
                strbuf_append_char(out, *buf);
            } else {
                strbuf_append_char(out, *buf);
            }
        } else {
            if (wildcards && quoted && (*buf == '*' || *buf == '?'))
                strbuf_append_char(out, '\\'); /* escape wildcard for fnmatch */
            strbuf_append_char(out, *buf);
        }
    }

    *bufptr = buf;
    *quotedptr = quoted;
    *outptr = strbuf_free_to_string(out);
    
    return wildcards;
}

static void
add_remote_wildcard_result(char *name, DCFileList *node, DCFSCompletionFlags flags, DCCompletionInfo *ci)
{
    DCCompletionEntry *entry = NULL;
    char *input;

    input = filename_quote_string(name, ci->word_full[0] == '"', true);
    if (node->type == DC_TYPE_DIR) {
        if (flags & DC_CPL_DIR)
            entry = new_completion_entry_full(input, name, "%s/", "%s/", false, true);
    } else {
        if (flags & DC_CPL_REG)
            entry = new_completion_entry_full(input, name, "%s", "%s", true, true);
    }
    if (entry != NULL) {
        entry->sorting.file_type = node->type;
        ptrv_append(ci->results, entry);
    } else {
        free(input);
        free(name);
    }
}

/* Note that NAME will never be "/" for this function.
 * It is not possible for a complete operation to have the bare result "/".
 */
static void
add_local_wildcard_result(char *name, DCFSCompletionFlags flags, DCCompletionInfo *ci)
{
    struct stat st;

    if (lstat(name, &st) == 0) {
        DCCompletionEntry *entry = NULL;
        char *input;
        bool dquoted = (ci->word_full[0] == '"');

        input = filename_quote_string(name, dquoted, true);
        if (S_ISDIR(st.st_mode)) {
            if (flags & DC_CPL_DIR) {
                entry = new_completion_entry_full(input, name, "%s/", "%s/", false, true);
                entry->sorting.file_type = DC_TYPE_DIR;
            }
        } else if (S_ISLNK(st.st_mode)) {
            if (stat(name, &st) == 0 && S_ISDIR(st.st_mode)) {
                if (flags & DC_CPL_DIR) {
                    entry = new_completion_entry_full(input, name, "%s", "%s@", false, true);
                    if (dquoted)
                        input[strlen(input)-1] = '\0';
                    if (strcmp(ci->word_full, input) == 0)
                        entry->input_single_fmt = "%s/";
                    if (dquoted)
                        input[strlen(input)] = '"';
                    entry->sorting.file_type = DC_TYPE_DIR;
                }
            } else {
                if (flags & DC_CPL_REG) {
                    entry = new_completion_entry_full(input, name, "%s", "%s@", true, true);
                    entry->sorting.file_type = DC_TYPE_REG;
                }
            }
        } else if (S_ISREG(st.st_mode)) {
            if ((flags & DC_CPL_REG) || (flags & DC_CPL_EXE)) {
                if ((access(name, X_OK) == 0) == ((flags & DC_CPL_EXE) != 0)) {
                    entry = new_completion_entry_full(input, name, "%s", "%s", true, true);
                    entry->sorting.file_type = DC_TYPE_REG;
                } else if ((flags & DC_CPL_EXE) == 0) {
                    entry = new_completion_entry_full(input, name, "%s", "%s*", true, true);
                    entry->sorting.file_type = DC_TYPE_REG;
                }
            }
        }

        if (entry != NULL) {
            ptrv_append(ci->results, entry);
        } else {
            free(input);
            free(name);
        }
    }
}

static void
filelist_iterator(DCFileList *node, DCFileListIterator *it)
{
    hmap_iterator(node->u.dir.children, &it->it);
    it->node = node;
    it->c = 0;
}

static bool
filelist_get_next(DCFileListIterator *it, DCFileList **node, char **name)
{
    it->c++;
    if (it->c == 1) {
        *node = it->node; 
        *name = ".";
        return true;
    }
    if (it->c == 2) {
        *node = it->node->parent == NULL ? it->node : it->node->parent;
        *name = "..";
        return true;
    }
    if (it->it.has_next(&it->it)) {
        *node = it->it.next(&it->it);
        *name = (*node)->name;
        return true;
    }
    return false;
}

void
remote_wildcard_expand(char *matchpath, bool *quotedptr, const char *basedir, DCFileList *basenode, PtrV *results)
{
    char *matchcomp;
    char *fullpath;
    char *nodename;
    DCFileList *node;
    DCFileListIterator it;

    if (dircomp_to_fnmatch_str(&matchpath, quotedptr, &matchcomp)) {
        filelist_iterator(basenode, &it);
        while (filelist_get_next(&it, &node, &nodename)) {
            if (fnmatch(matchcomp, nodename, FNM_PERIOD) == 0) {
                if (*matchpath != '\0') {
                    if (node->type == DC_TYPE_DIR) {
                        fullpath = concat_filenames(basedir, nodename);
                        remote_wildcard_expand(matchpath, quotedptr, fullpath, node, results);
                        free(fullpath);
                    }
                } else {
                    ptrv_append(results, concat_filenames(basedir, nodename));
                    /*ptrv_append(results, node);*/
                }
            }
        }
    } else {
        if (*matchcomp == '\0') {
            ptrv_append(results, concat_filenames(basedir, matchcomp));
        } else {
            node = get_child_node(basenode, matchcomp);
            if (node != NULL) {
                if (*matchpath != '\0') {
                    fullpath = concat_filenames(basedir, matchcomp);
                    remote_wildcard_expand(matchpath, quotedptr, fullpath, node, results);
                    free(fullpath);
                } else {
                    ptrv_append(results, concat_filenames(basedir, matchcomp));
                    /*ptrv_append(results, node);*/
                }
            }
        }
    }
    free(matchcomp);
}

static void
remote_wildcard_complete(char *matchpath, bool *quotedptr, char *basedir, DCFileList *basenode, DCFSCompletionFlags flags, DCCompletionInfo *ci, bool found_wc)
{
    char *matchcomp;
    char *fullpath;
    char *nodename;
    DCFileList *node;
    DCFileListIterator it;

    if (dircomp_to_fnmatch_str(&matchpath, quotedptr, &matchcomp)) {
        filelist_iterator(basenode, &it);
        while (filelist_get_next(&it, &node, &nodename)) {
            if (fnmatch(matchcomp, nodename, FNM_PERIOD) == 0) {
                if (*matchpath != '\0') {
                    if (node->type == DC_TYPE_DIR) {
                        fullpath = concat_filenames(basedir, nodename);
                        remote_wildcard_complete(matchpath, quotedptr, fullpath, node, flags, ci, true);
                        free(fullpath);
                    }
                } else {
                    add_remote_wildcard_result(concat_filenames(basedir, node->name), node, flags, ci);
                }
            }
        }
    } else {
        fullpath = concat_filenames(basedir, matchcomp);
        if (*matchpath != '\0') { /* more components follow after this one? */
            node = get_child_node(basenode, matchcomp);
            if (node != NULL)
                remote_wildcard_complete(matchpath, quotedptr, fullpath, node, flags, ci, found_wc);
        } else {
            /* If the string we are completing had wild cards, attempt to expand the string
             * first rather than generating multiple possible completion results.
             * This behavior is similar to that of GNU readline.
             */
	    if (found_wc) {
	        if (*matchcomp == '\0') { /* completion word ends in slash */
                    add_remote_wildcard_result(concat_filenames(basedir, matchcomp), basenode, flags, ci);
	        } else {
	            node = get_child_node(basenode, matchcomp);
                    if (node != NULL)
                        add_remote_wildcard_result(concat_filenames(basedir, matchcomp), node, flags, ci);
                }
            } else {
                filelist_iterator(basenode, &it);
                while (filelist_get_next(&it, &node, &nodename)) {
                    if ((nodename[0] == '.') != (matchcomp[0] == '.'))
                        continue;
                    if (strleftcmp(matchcomp, nodename) == 0)
                        add_remote_wildcard_result(concat_filenames(basedir, nodename), node, flags, ci);
                }
            }
        }
        free(fullpath);
    }
    free(matchcomp);
}

static void
local_wildcard_complete(char *matchpath, bool *quotedptr, char *basedir, DCFSCompletionFlags flags, DCCompletionInfo *ci, bool found_wc)
{
    char *matchcomp;
    struct stat sb;
    char *fullpath;
    DIR *dh;
    struct dirent *de;

    if (dircomp_to_fnmatch_str(&matchpath, quotedptr, &matchcomp)) {
        dh = opendir(*basedir == '\0' ? "." : basedir);
        if (dh != NULL) {
            while ((de = readdir(dh)) != NULL) {
                if (fnmatch(matchcomp, de->d_name, FNM_PERIOD) == 0) {
                    if (*matchpath != '\0') {
                        fullpath = concat_filenames(basedir, de->d_name);
                        if (stat(fullpath, &sb) == 0 && S_ISDIR(sb.st_mode))
                            local_wildcard_complete(matchpath, quotedptr, fullpath, flags, ci, true);
                        free(fullpath);
                    } else {
                        add_local_wildcard_result(concat_filenames(basedir, de->d_name), flags, ci);
                    }
                }
            }
            closedir(dh);
        }
    } else {
        fullpath = concat_filenames(basedir, matchcomp);
        if (*matchpath != '\0') {
            if (lstat(fullpath, &sb) == 0)
                local_wildcard_complete(matchpath, quotedptr, fullpath, flags, ci, found_wc);
        } else {
            /* If the string we are completing had wild cards, attempt to expand the string
             * first rather than generating multiple possible completion results.
             * This behavior is similar to that of GNU readline.
             */
	    if (found_wc) {
	        if (*matchcomp == '\0') { /* completion word ends in slash */
	            add_local_wildcard_result(concat_filenames(basedir, matchcomp), flags, ci);
                } else if (/*strcmp(basedir, "/") != 0 &&*/ lstat(fullpath, &sb) == 0) {
                    add_local_wildcard_result(concat_filenames(basedir, matchcomp), flags, ci);
                }
            } else {
                dh = opendir(*basedir == '\0' ? "." : basedir);
                if (dh != NULL) {
                    while ((de = readdir(dh)) != NULL) {
                        if ((de->d_name[0] == '.') != (matchcomp[0] == '.'))
                            continue;
                        if (strleftcmp(matchcomp, de->d_name) == 0)
                            add_local_wildcard_result(concat_filenames(basedir, de->d_name), flags, ci);
                    }
                    closedir(dh);
                }
            }
        }
        free(fullpath);
    }
    free(matchcomp);
}

static void
fixup_wildcard_completion_results(DCCompletionInfo *ci)
{
    if (ci->results->cur > 1) {
        DCCompletionEntry *ce;
        char *s1, *s2;
        int c, d;
        int min;

        /* This differs from GNU readline. Readline always displays
         * the base name of every result, even if the directory part
         * differs. Find longest common leading string, then find last
         * '/' and strip everything before and including the slash.
         */
        ce = ci->results->buf[0];
        s1 = xasprintf(ce->display_fmt, ce->display);
        min = strlen(s1);

        for (c = 1; c < ci->results->cur; c++) {
            char c1, c2;

            ce = ci->results->buf[c];
            s2 = xasprintf(ce->display_fmt, ce->display);
            for (d = 0; (c1 = s1[d]) != '\0' && (c2 = s2[d]) != '\0'; d++) {
                if (c1 != c2)
                    break;
            }
            min = MIN(min, d);
            free(s2);
        }

        if (min > 0) {
            s1[min] = '\0';
            s2 = strrchr(s1, '/');
            if (s2 != NULL) {
                min = s2-s1+1;
                for (c = 0; c < ci->results->cur; c++) {
                    ce = ci->results->buf[c];
                    s2 = strdup(ce->display + min);
                    if (ce->display != ce->input)
                        free(ce->display);
                    ce->display = s2;
                }
            }
        }
        free(s1);
 
        ptrv_sort(ci->results, fs_completion_entry_compare);
    }
}

void
local_fs_completion_generator(DCCompletionInfo *ci, DCFSCompletionFlags flags)
{
    bool quoted = false;
    local_wildcard_complete(ci->word_full, &quoted, has_leading_slash(ci->word_full) ? "/" : "", flags, ci, false);
    fixup_wildcard_completion_results(ci);
}

void
local_path_completion_generator(DCCompletionInfo *ci)
{
    local_fs_completion_generator(ci, DC_CPL_REG|DC_CPL_DIR);
}

void
local_dir_completion_generator(DCCompletionInfo *ci)
{
    local_fs_completion_generator(ci, DC_CPL_DIR);
}

void
remote_fs_completion_generator(DCCompletionInfo *ci, DCFSCompletionFlags flags)
{
    bool quoted = false;
    char *basedir;
    DCFileList *basenode;

    if (browse_list == NULL)
        return;
    if (has_leading_slash(ci->word_full)) {
        basenode = filelist_lookup(browse_list, "/");
        basedir = "/";
    } else {
        basenode = filelist_lookup(browse_list, browse_path);
        basedir = "";
    }
    if (basenode != NULL) {
        remote_wildcard_complete(ci->word_full, &quoted, basedir, basenode, flags, ci, false);
        fixup_wildcard_completion_results(ci);
    }
}

void
remote_path_completion_generator(DCCompletionInfo *ci)
{
    return remote_fs_completion_generator(ci, DC_CPL_REG|DC_CPL_DIR);
}

void
remote_dir_completion_generator(DCCompletionInfo *ci)
{
    return remote_fs_completion_generator(ci, DC_CPL_DIR);
}
