/*
 * @(#)TypeConverter.java      1.1             25 October 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.type;

import java.util.Vector;

/**
 * An interface for transfering type information from types to other
 * objects. All methods should at maximum be called once for each
 * type, and the createX() methods are even mutually exclusive,
 * meaning that only one of them may be called.
 *
 * @version  1.1
 * @author   Per Cederberg, per@percederberg.net
 */
public interface TypeConverter {

    /**
     * Called for integer types.
     */
    void createInteger();

    /**
     * Called for integer types that may only have one of an
     * enumerated set of values.
     *
     * @param   labels     a vector with the string labels
     * @param   values     a vector with the integer values
     */
    void createEnumeratedInteger(Vector labels, Vector values);

    /**
     * Called for string types.
     */
    void createString();

    /**
     * Called for vector types.
     *
     * @param   elementType   the vector element type
     */
    void createVector(Type elementType);

    /**
     * Called for composed types.
     *
     * @param   symbols      a vector with the symbols that compose
     *                       the type
     */
    void createCompound(Vector symbols);

    /**
     * Called to report a type name alias for this type. This method
     * is called once for each type symbol referenced from the type.
     * The order will be the reverse, i.e. the inner-most name will be
     * reported first.
     *
     * @param   name         the alias nam
     */
    void transferNameAlias(String name);

    /**
     * Called for integer types having value limits associated with
     * them.
     *
     * @param   minValue     the minimum value
     * @param   maxValue     the maximum value
     */
    void transferValueLimits(int minValue, int maxValue);

    /**
     * Called for types having size limits associated with them.
     * Typically this can be either strings or arrays.
     *
     * @param   minLength     the minimum length, or -1 for none
     * @param   maxLength     the maximum length, or -1 for none
     */
    void transferSizeLimits(int minLength, int maxLength);

    /**
     * Called to set the status of whether or not that data is
     * readable from the data layer.
     *
     * @param   readable    true if readable, false otherwise
     */
    void transferReadable(boolean readable);

    /**
     * Called to set the status of whether or not that data is
     * writable to the data layer.
     *
     * @param   writeable    true if writeable, false otherwise
     */
    void transferWriteable(boolean writeable);

    /**
     * Called for types having a default value.
     *
     * @param   value     a string value or a label, depending on
     *                    context
     */
    void transferDefaultValue(String value);

    /**
     * Called for types having a default value.
     *
     * @param   value      an integer value
     */
    void transferDefaultValue(int value);

}
