/*=======================================================================
 * Version: $Id: html.c,v 1.2 2014/11/13 16:36:40 nroche Exp $
 * Project: MediaTeX
 * Module : html
 *
 * HTML generator

 MediaTex is an Electronic Records Management System
 Copyright (C) 2014  Nicolas Roche

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 =======================================================================*/

#include "alloc.h"
#include "html.h"

#include <ctype.h>

/*=======================================================================
 * Function   : htmlCaps
 * Description: 
 * Synopsis   : 
 * Input      : FILE* fd
 * Output     : TRUE on success
 =======================================================================*/
int
htmlCaps(FILE* fd, char* text)
{
  int rc = FALSE;
  char* string = NULL;
  int l = 0;

  if (text == NULL) goto error;
  logEmit(LOG_DEBUG, "%s", "htmlCaps");

  l = strlen(text);
  if (!(string = malloc(l+1))) {
    logEmit(LOG_ERR, "%s", "malloc fails");
    goto error;
  }
  strncpy(string, text, l);
  string[l] = (char)0;
  
  while (--l >= 0) {
    string[l] = toupper(string[l]);
  }

  if (!fprintf(fd, "<SMALL>%s</SMALL>", string)) 
    goto error;

  rc = TRUE;
 error:
  if(!rc) {
    logEmit(LOG_ERR, "%s", "htmlCaps fails");
  }
  free(string);
  return rc;
}


/*=======================================================================
 * Function   : htmlMainHead
 * Description: 
 * Synopsis   : 
 * Input      : FILE* fd
 * Output     : TRUE on success
 =======================================================================*/
int
htmlMainHeadBasic(FILE* fd, char* title, char* url)
{
  int rc = FALSE;

  logEmit(LOG_DEBUG, "%s", "htmlMainHead");
  
  if (!fprintf(fd, 
	       "<!DOCTYPE HTML PUBLIC '-//W3C//DTD HTML 4.1//EN'>\n"
	       "\n"
	       "<!--Inspired from LaTeX2HTML 2008 (1.71)\n"
	       " Generated by MediaTeX software\n"
	       " Nicolas Roche -->\n"
	       "<HTML>\n"
	       "<HEAD>\n"
	       "<TITLE>%s</TITLE>\n"
	       "<META NAME='description' CONTENT='MediaTeX catalog'>\n"
	       "<META NAME='keywords' CONTENT='%s'>\n"
	       "<META NAME='resource-type' CONTENT='data files'>\n"
	       "<META NAME='distribution' CONTENT='private content'>\n"
	       "\n"
	       "<META HTTP-EQUIV='Content-Type' "
	       "CONTENT='text/html; charset=utf-8'>\n"
	       "<META NAME='Generator' CONTENT='MediaTeX v2013'>\n"
	       "<META HTTP-EQUIV='Content-Style-Type' "
	       "CONTENT='text/css'>\n"
	       "\n"
	       "<LINK REL='STYLESHEET' "
	       "HREF='%s/mediatex.css'>\n"
	       "\n"
	       "</HEAD>\n"
	       "\n"
	       "<BODY >\n"
	       "<TABLE CELLPADDING=3>\n",
	       title, title, url)) goto error;

  rc = TRUE;
 error:
  if(!rc) {
    logEmit(LOG_ERR, "%s", "htmlMainHead fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : htmlLeftPageTop
 * Description: 
 * Synopsis   : 
 * Input      : FILE* fd
 *              char* rep = relative path from HOME
 * Output     : TRUE on success
 =======================================================================*/
int
htmlLeftPageHeadBasic(FILE* fd, char* rep, char* url)
{
  int rc = FALSE;

  logEmit(LOG_DEBUG, "%s", "htmlLeftPageHead");
  
  if (!fprintf(fd,
	       "<TR><TD ALIGN='LEFT' VALIGN='TOP' WIDTH=192>"
	       "<TABLE  WIDTH='100%%'>\n"
	       "<TR><TD>\n"
	       "<A NAME='tex2html1'\n"
	       "HREF='%s/%s'><IMG\n"
	       "ALIGN='BOTTOM' BORDER='0' "
	       "SRC='%s/logo.png'\n"
	       "ALT='top'></A>\n"
	       "<P>\n",
	       url, rep, url)) goto error;

  rc = TRUE;
 error:
  if(!rc) {
    logEmit(LOG_ERR, "%s", "htmlLeftPageHead fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : htmlLeftPageTail
 * Description: 
 * Synopsis   : 
 * Input      : FILE* fd
 * Output     : TRUE on success
 =======================================================================*/
int
htmlLeftPageTail(FILE* fd)
{
  int rc = FALSE;

  logEmit(LOG_DEBUG, "%s", "htmlLeftPageTail");
  
  if (!fprintf(fd, "%s",
	       "\n"
	       "<P>\n"
	       "</TD></TR>\n"
	       "</TABLE></TD>\n"
	       )) goto error;

  rc = TRUE;
 error:
  if(!rc) {
    logEmit(LOG_ERR, "%s", "htmlLeftPageTail fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : htmlRightHeadBasic
 * Description: 
 * Synopsis   : 
 * Input      : FILE* fd
 *              char* masterUrl: viewvc only remains on master host
 * Output     : TRUE on success
 =======================================================================*/
int
htmlRightHeadBasic(FILE* fd, char* masterUrl, char* url)
{
  int rc = FALSE;

  logEmit(LOG_DEBUG, "%s", "htmlRightHead");
  
  //"<TD ALIGN='LEFT' VALIGN='TOP' WIDTH=504><TABLE  WIDTH='100%%'>\n"

  if (!fprintf(fd,
	       "<TD ALIGN='LEFT' VALIGN='TOP'><TABLE>\n"
	       "<TR><TD>\n"
	       "<TABLE CELLPADDING=3>\n"
	       "<TR><TD ALIGN='CENTER'><A NAME='tex2html7'\n"
	       "HREF='%s/index'>Index</A></TD>\n"
	       "<TD ALIGN='CENTER'><A NAME='tex2html8'\n"
	       "HREF='%s/cache'>Cache</A></TD>\n"
	       "<TD ALIGN='CENTER'><A NAME='tex2html9'\n"
	       "HREF='%s/score'>Score</A></TD>\n"
	       "<TD ALIGN='CENTER'><A NAME='tex2html10'\n"
	       "HREF='%s/cgi/viewvc.cgi'>Version</A></TD>\n"
	       "</TR>\n"
	       "</TABLE>\n"
	       "<DIV ALIGN='CENTER'>\n"
	       "\n"
	       "</DIV>\n"
	       "\n",
	       url, url, url, masterUrl)) goto error;

  rc = TRUE;
 error:
  if(!rc) {
    logEmit(LOG_ERR, "%s", "htmlRightHeadBasic fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : htmlMainTail
 * Description: 
 * Synopsis   : 
 * Input      : FILE* fd
 * Output     : TRUE on success
 =======================================================================*/
int
htmlMainTail(FILE* fd, char* date)
{
  int rc = FALSE;

  logEmit(LOG_DEBUG, "%s", "htmlMainTail");
  
  if (!fprintf(fd,
	       "</TD></TR>\n"
	       "</TABLE></TD>\n"
	       "</TR>\n"
	       "</TABLE>\n"
	       "<BR><HR>\n"
	       "<ADDRESS>%s</ADDRESS>\n"
	       "</BODY>\n"
	       "</HTML>\n\n",
	       date)) goto error;

  rc = TRUE;
 error:
  if(!rc) {
    logEmit(LOG_ERR, "%s", "htmlMainTail fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : 
 * Description: 
 * Synopsis   : 
 * Input      : FILE* fd
 * Output     : TRUE on success
 =======================================================================*/
int
truc(FILE* fd)
{
  int rc = FALSE;

  logEmit(LOG_DEBUG, "%s", "truc");
  
  goto error;

  rc = TRUE;
 error:
  if(!rc) {
    logEmit(LOG_ERR, "%s", "truc fails");
  }
  return rc;
}


/************************************************************************/

#ifdef utMAIN
#include "command.h"
GLOBAL_STRUCT_DEF;

/*=======================================================================
 * Function   : htmlLeftPage
 * Description: 
 * Synopsis   : 
 * Input      : FILE* fd
 * Output     : TRUE on success
 =======================================================================*/
int
htmlLeftPageMiddle(FILE* fd)
{
  int rc = FALSE;

  logEmit(LOG_DEBUG, "%s", "htmlLeftPageMiddle");
  
  htmlLink(fd, "tex2html2", "titles.html", "tous les documents");
  if (!fprintf(fd, "%s","\n")) goto error;
  htmlBr(fd);
  htmlBr(fd);
  htmlLink(fd, "tex2html3", "category_002.html", "média");
  if (!fprintf(fd, "%s","\n")) goto error;
  htmlBr(fd);
  if (!fprintf(fd, "%s","\n")) goto error;
  htmlUlOpen(fd);
  htmlLiOpen(fd);
  htmlLink(fd, "tex2html4", "category_001.html", "images");
  if (!fprintf(fd, "%s","\n")) goto error;
  htmlBr(fd);
  if (!fprintf(fd, "%s","\n")) goto error;
  htmlUlOpen(fd);
  htmlLiOpen(fd);
  htmlLink(fd, "tex2html5", "category_000.html", "animaux");
  if (!fprintf(fd, "%s", "\n")) goto error;
  htmlBr(fd);
  htmlLiClose(fd);
  htmlUlClose(fd);
  htmlLiClose(fd);
  htmlUlClose(fd);
  htmlBr(fd);
  if (!fprintf(fd, "%s", "\nCollaborateurs: ")) goto error;
  htmlBr(fd);
  if (!fprintf(fd, "%s", "\n")) goto error;
  htmlUlOpen(fd);
  htmlLiOpen(fd);
  htmlLink(fd, "tex2html6", "role_000.html", "dessinateur");
  if (!fprintf(fd, "%s", "\n")) goto error;
  htmlBr(fd);
  htmlLiClose(fd);
  htmlUlClose(fd);

  rc = TRUE;
 error:
  if(!rc) {
    logEmit(LOG_ERR, "%s", "htmlLeftPage fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : htmlRightTail
 * Description: 
 * Synopsis   : 
 * Input      : FILE* fd
 * Output     : TRUE on success
 =======================================================================*/
int
htmlRightTail(FILE* fd)
{
  int rc = FALSE;

  logEmit(LOG_DEBUG, "%s", "htmlRightTail");
  
  if (!fprintf(fd, "%s", "<P><P>")) goto error;
  htmlBr(fd);
  
  htmlBold(fd, "Document");
  if (!fprintf(fd, "%s", " : ")) goto error;
  if (!htmlCaps(fd, "coccinelle")) goto error;
  
  if (!fprintf(fd, "%s", "\n\n<P>\n")) goto error;
  if (!fprintf(fd, "%s", "(")) goto error;
  htmlLink(fd, "tex2html11", "category_000.html", "animaux");
  if (!fprintf(fd, "%s", ", ")) goto error;
  htmlLink(fd, "tex2html12", "category_001.html", "images");
  if (!fprintf(fd, "%s", 
	       ")\n"
	       "\n")) goto error;
  htmlUlOpen(fd);
  htmlLiOpen(fd);
  if (!fprintf(fd, "%s", 
	       "licence = pompé sur le net (non-free)\n")) goto error;
  htmlLiClose(fd);
  htmlUlClose(fd);
  if (!fprintf(fd, "%s",
	       "\n"
	       "<P>\n"
	       "Collaborateurs :\n"
	       "\n")) goto error;
  htmlUlOpen(fd);
  htmlLiOpen(fd);
  htmlLink(fd, "tex2html13", "role_000.html", "dessinateur");
  if (!fprintf(fd, "%s", "\n: ")) goto error;
  htmlLink(fd, "tex2html14", "human_000.html", "Gotlib");
  if (!fprintf(fd, "%s", "\n")) goto error;
  htmlLiClose(fd);
  htmlUlClose(fd);
  if (!fprintf(fd, "%s",
	       "\n"
	       "<P>\n"
	       "Enregistrements :\n"
	       "\n")) goto error;
  htmlUlOpen(fd);
  htmlLiOpen(fd);
  htmlLink(fd, "tex2html15", 
	   "https://localhost/~mdtx-hello/cgi/get.cgi?hash=365acad0657b0113990fe669972a65de&amp;size=15106", "url");

  if (!fprintf(fd, "%s", "\n")) goto error;
  htmlLink(fd, "tex2html16", 
	   "../extract/365acad0657b0113990fe669972a65de_15106.html", 
	   "(0.00)");

  if (!fprintf(fd, "%s", "\n")) goto error;
  htmlUlOpen(fd);
  htmlLiOpen(fd);
  if (!fprintf(fd, "%s","description = surf\n")) goto error;
  htmlLiClose(fd);
  htmlLiOpen(fd);
  if (!fprintf(fd, "%s", 
	       "format = PNG\n"
	       "\n")) goto error;
  htmlLiClose(fd);
  htmlUlClose(fd);
  htmlLiClose(fd);
  htmlUlClose(fd);

  rc = TRUE;
 error:
  if(!rc) {
    logEmit(LOG_ERR, "%s", "htmlRightTail fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : usage
 * Description: Print the usage.
 * Synopsis   : static void usage(char* programName)
 * Input      : programName = the name of the program; usually
 *                                  argv[0].
 * Output     : N/A
 =======================================================================*/
static void usage(char* programName)
{
  miscUsage(programName);
  miscOptions();
  return;
}


/*=======================================================================
 * Function   : main 
 * Author     : Nicolas ROCHE
 * modif      : 
 * Description: Unit test for md5sum module
 * Synopsis   : ./utcommand -i scriptPath
 * Input      : N/A
 * Output     : N/A
 =======================================================================*/
int 
main(int argc, char** argv)
{
  FILE* fd = stdout;
  char* htmlPath = "ut.html";
  //char* cssPath = "ut.css";
  // ---
  int rc = 0;
  int cOption = EOF;
  char* programName = *argv;
  char* options = MISC_SHORT_OPTIONS"";
  struct option longOptions[] = {
    MISC_LONG_OPTIONS,
    {0, 0, 0, 0}
  };
  
  // import mdtx environment
  getEnv(&env);

  // parse the command line
  while((cOption = getopt_long(argc, argv, options, longOptions, NULL)) 
	!= EOF) {
    switch(cOption) {
      
      GET_MISC_OPTIONS; // generic options
    }
    if (rc) goto optError;
  }

  // export mdtx environment
  if (!setEnv(programName, &env)) goto optError;

  /************************************************************************/
  if (!(fd = fopen(htmlPath, "w"))) {
    logEmit(LOG_ERR, "fopen fails: %s", strerror(errno));
    goto error;
  }

  if (!htmlMainHead(fd, "Main")) goto error;
  if (!htmlLeftPageHead(fd, "index")) 
    goto error;
  if (!htmlLeftPageMiddle(fd)) goto error;
  if (!htmlLeftPageTail(fd)) goto error;
  if (!htmlRightHead(fd, "https://localhost/~mdtx-hello/")) 
    goto error;
  if (!htmlRightTail(fd)) goto error;
  if (!htmlMainTail(fd, "2013-12-17")) goto error;

  if (fclose(fd)) {
    logEmit(LOG_ERR, "fclose fails: %s", strerror(errno));
    goto error;
  }
  /************************************************************************/

  rc = TRUE;
 error:
  ENDINGS;
  rc=!rc;
 optError:
  exit(rc);
}

#endif // utMAIN

/* Local Variables: */
/* mode: c */
/* mode: font-lock */
/* mode: auto-fill */
/* End: */


