/************************************************************************\
 * Magic Square solves magic squares.                                   *
 * Copyright (C) 2019  Asher Gordon <AsDaGo@posteo.net>                 *
 *                                                                      *
 * This file is part of Magic Square.                                   *
 *                                                                      *
 * Magic Square is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * Magic Square is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with Magic Square.  If not, see                                *
 * <https://www.gnu.org/licenses/>.                                     *
\************************************************************************/

/* write.c -- functions for outputting magic squares */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <endian.h>

#include "write.h"
#include "parse.h" /* For HLINE, VLINE, and JOINT */
#include "square.h"

/* The width to wrap at if the square is smaller than this */
#define MIN_WRAP_WIDTH 70

/* Static helper functions */
static int write_machine_cellval(cellval_t, FILE *);
static int write_machine_cell(cell_t, FILE *);

static size_t write_human_separator(size_t, size_t, FILE *);
static size_t write_human_other_nums(const cellval_t *,
				     size_t, size_t, FILE *);

/* Write a magic square in a machine-readable format */
size_t write_machine(const square_t *square, FILE *file) {
  size_t description_size,
    description_size_be, square_size_be, nums_size_be;
  const uint32_t magic_number = htobe32(MAGIC_NUMBER);

  /* Get the description size */
  description_size = square->description ? strlen(square->description) : 0;

  /* Convert endianness */
  description_size_be = htobe64(description_size);
  square_size_be = htobe64(square->size);
  nums_size_be = htobe64(square->nums_size);

  /* Write all the sizes and stuff */
  if (!(/* Write the magic number */
	fwrite(&magic_number, sizeof(magic_number), 1, file) &&

	/* Write the file version number */
	putc(FILE_VERSION, file) != EOF &&

	/* Write the square size */
	fwrite(&square_size_be, sizeof(square_size_be), 1, file) &&

	/* And the other numbers size */
	fwrite(&nums_size_be, sizeof(nums_size_be), 1, file) &&

	/* Write the description size */
	fwrite(&description_size_be, sizeof(description_size_be), 1, file))) {
    return 0; /* Zero to indicate an error even though we might have
		 written some stuff */
  }

  /* Now write the cells */
  for (size_t x = 0; x < square->size; x++) {
    for (size_t y = 0; y < square->size; y++) {
      if (write_machine_cell(square->cells[x][y], file))
	return 0;
    }
  }

  /* Write the other valid numbers */
  for (size_t i = 0; i < square->nums_size; i++) {
    if (write_machine_cellval(square->nums[i], file))
      return 0;
  }

  /* And the description */
  if (fwrite(square->description, sizeof(*(square->description)),
	     description_size, file) != description_size) {
    return 0;
  }

  /* Return the bytes written */
  return
    sizeof(magic_number) + sizeof(square->size) + sizeof(description_size) +
    (sizeof(**(square->cells)) * square->size * square->size) +
    (sizeof(*(square->nums)) * square->nums_size) +
    (sizeof(*(square->description)) * description_size);
}

/* Write a magic square in a human-readable format, returning 0 on
   error, number of bytes written on success. */
size_t write_human(const square_t *square, FILE *file) {
  struct {
    char *str;
    int length, index;
  } **outstrings; /* String representations of the numbers */
  size_t max_length = 0; /* The longest string in `outstrings' */
  size_t cell_width, cell_height; /* The output cell size */
  size_t bytes_written = 0; /* How many bytes we have written */
  size_t ret;

  /* First print the description if there is one */
  if (square->description)
    bytes_written += fprintf(file, "%s\n\n", square->description);

  outstrings = malloc(sizeof(*outstrings) * square->size);

  for (size_t x = 0; x < square->size; x++) {
    outstrings[x] = malloc(sizeof(*(outstrings[x])) * square->size);

    for (size_t y = 0; y < square->size; y++) {
      if (square->cells[x][y].mutable) {
	/* Don't output anything for this one */
	outstrings[x][y].str = NULL;
	outstrings[x][y].length = 0;
      }
      else {
	switch (square->cells[x][y].val.type) {
	case INT:
	  outstrings[x][y].length = asprintf(&(outstrings[x][y].str), "%lld",
					     square->cells[x][y].val.i);
	  break;
	case FLOAT:
	  outstrings[x][y].length = asprintf(&(outstrings[x][y].str), "%Lg",
					     square->cells[x][y].val.f);
	  break;
	default:
	  /* Invalid type! */
	  for (size_t i = 0; i <= x; i++) {
	    for (size_t j = 0; j < y; j++) {
	      if (outstrings[i][j].str)
		free(outstrings[i][j].str);
	    }

	    free(outstrings[i]);
	  }

	  free(outstrings);

	  errno = EINVAL;
	  return 0;
	}

	if (outstrings[x][y].length > max_length)
	  max_length = outstrings[x][y].length;
      }

      outstrings[x][y].index = 0;
    }//							     +----+
  }  //						    +--+     |	  |
     //						    |  |     |	  |
  /* Use a cell ratio of 2:1 since that looks best (+--+ and +----+
     for example). */
  for (cell_height = 1;
       cell_height * (cell_width = cell_height * 2) < max_length;
       cell_height++) {
    /* Check for overflow */
    if (cell_height == SIZE_MAX / 2) {
      errno = EOVERFLOW;
      goto error;
    }
  }

  /* Write the first separator */
  if (!(ret = write_human_separator(cell_width, square->size, file)))
    goto error;

  bytes_written += ret;

  /* Write the rest */
  for (size_t square_y = 0; square_y < square->size; square_y++) {
    for (size_t cell_y = 0; cell_y < cell_height; cell_y++) {
      if (putc(VLINE, file) == EOF)
	goto error;

      bytes_written++;

      for (size_t square_x = 0; square_x < square->size; square_x++) {
	for (size_t cell_x = 0; cell_x < cell_width; cell_x++) {
	  if (outstrings[square_x][square_y].index <
	      outstrings[square_x][square_y].length) {
	    /* There's still stuff left to print in this string */
	    if (putc(outstrings[square_x][square_y].str
		     [outstrings[square_x][square_y].index++],
		     file) == EOF) {
	      goto error;
	    }

	    bytes_written++;
	  }
	  else {
	    /* We're done printing this string, just fill the rest
	       with spaces */
	    if (putc(' ', file) == EOF)
	      goto error;

	    bytes_written++;
	  }
	}

	/* Add the seperator */
	if (putc(VLINE, file) == EOF)
	  goto error;

	bytes_written++;
      }

      /* Start a new line */
      if (putc('\n', file) == EOF)
	goto error;

      bytes_written++;
    }

    /* Print the separator */
    if (!(ret = write_human_separator(cell_width, square->size, file)))
      goto error;

    bytes_written += ret;
  }

  /* Write the other numbers if there are any */
  if (square->nums_size) {
    size_t wrap; /* Where to wrap the line */

    if (putc('\n', file) == EOF)
      goto error;

    bytes_written++;

    /* Wrap at the width of the square or at MIN_WRAP_WIDTH if the
       square is smaller than that */
    if ((wrap = square->size * (cell_width + 1) + 1) < MIN_WRAP_WIDTH)
      wrap = MIN_WRAP_WIDTH;

    ret = write_human_other_nums(square->nums, square->nums_size, wrap, file);

    if (!ret)
      goto error;

    bytes_written += ret;
  }

  goto success;

 error:
  bytes_written = 0;

 success:
  for (size_t x = 0; x < square->size; x++) {
    for (size_t y = 0; y < square->size; y++) {
      if (outstrings[x][y].str)
	free(outstrings[x][y].str);
    }

    free(outstrings[x]);
  }

  free(outstrings);

  return bytes_written;
}

/***************************\
|* Static helper functions *|
\***************************/

/* Write a single `cellval_t'; returns 0 on success, nonzero on
   error. */
static int write_machine_cellval(cellval_t cellval, FILE *file) {
  if (putc(cellval.type, file) == EOF)
    return 1;

  switch (cellval.type) {
    cellvali_t cellvali;

  case INT:
    cellvali = htobe64(cellval.i);

    if (!fwrite(&(cellvali), sizeof(cellvali), 1, file))
      return 1;

    break;
  case FLOAT:
    /* TODO: Make this portable. I could not figure out how to
       portably write long doubles (or any floating point numbers,
       for that matter) to a binary file. */

    if (!fwrite(&(cellval.f),
		sizeof(cellval.f), 1, file))
      return 1;

    break;
  default:
    return 1;
  }

  return 0;
}

/* Write a cell returning 0 on success and nonzero on error */
static int write_machine_cell(const cell_t cell, FILE *file) {
  int ret;

  /* Write the value */
  if ((ret = write_machine_cellval(cell.val, file)))
    return ret;

  /* And write the mutability */
  if (putc(cell.mutable, file) == EOF)
    return 1;

  return 0;
}

/* Write the separator (i.e. +--+--+, etc.), returning number of bytes
   written. */
static size_t write_human_separator(size_t cell_width, size_t square_width,
				    FILE *file) {
  size_t bytes_written = 0; /* How many bytes we have written */

  if (putc(JOINT, file) == EOF)
    return 0;

  bytes_written++;

  for (size_t i = 0; i < square_width; i++) {
    for (size_t j = 0; j < cell_width; j++) {
      if (putc(HLINE, file) == EOF)
	return 0;

      bytes_written++;
    }

    if (putc(JOINT, file) == EOF)
      return 0;

    bytes_written++;
  }

  if (putc('\n', file) == EOF)
    return 0;

  bytes_written++;

  return bytes_written;
}

/* Write `nums' in the format "(1, 2, 3)" attempting to wrap at `wrap'
   or don't wrap at all if `wrap' is 0. Returns number of bytes
   written. */
static size_t write_human_other_nums(const cellval_t *nums, size_t nums_size,
				     size_t wrap, FILE *file) {
  size_t bytes_written = 0;
  size_t line_length = 0;

  if (putc('(', file) == EOF)
    return 0;

  bytes_written++;
  line_length++;

  for (size_t i = 0; i < nums_size; i++) {
    char *str;
    int length;

    /* Get the string to write */
    switch (nums[i].type) {
    case INT:
      length = asprintf(&str, "%lld", nums[i].i);
      break;
    case FLOAT:
      length = asprintf(&str, "%Lg", nums[i].f);
      break;
    default:
      errno = EINVAL;
      return 0;
    }

    if (i) {
      if (putc(',', file) == EOF) {
	free(str);
	return 0;
      }

      bytes_written++;
      line_length++;

      if (wrap && line_length + length + 1 > wrap) {
	/* Wrap the line */
	if (fputs("\n ", file) == EOF) {
	  free(str);
	  return 0;
	}

	bytes_written += 2;
	line_length = 1;
      }
      else {
	if (putc(' ', file) == EOF) {
	  free(str);
	  return 0;
	}

	bytes_written++;
	line_length++;
      }
    }

    if (fputs(str, file) == EOF) {
      free(str);
      return 0;
    }

    bytes_written += length;
    line_length += length;

    free(str);
  }

  /* Print the closing parenthesis */
  if (fputs(")\n", file) == EOF)
    return 0;

  bytes_written += 2;

  return bytes_written;
}
