;;;; tree.lisp --- FFI definitions for libxml-clisp

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-06-14
;;; 
;;; $Hg: tree.lisp,v ae6366215b15 2009-09-03T20:01:35+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP")



;;;; A Namespace.

(defun make-namespace (&optional address)
  "Return a Namespace with address ADDRESS."
  (make-instance 'namespace :address address))

(defun make-namespace-list (&optional address)
  "Return a Namespace-list with address ADDRESS."
  (make-instance 'namespace-list :address address))

;;; Properties of a Namespace.

(defun namespace-next (namespace)
  (make-namespace (funcall-if-true #'$lxcl-ns-next
                                   (item-address namespace))))

(defun namespace-type (namespace)
  (funcall-if-true #'(lambda (address)
                       (enum-from-value '$xml-ns-type
                                        ($lxcl-ns-type address)))
                   (item-address namespace)))

(defun namespace-href (namespace)
  (make-xml-string (funcall-if-true #'$lxcl-ns-href
                                    (item-address namespace))))

(setf (symbol-function 'namespace-name) #'namespace-href)

(defun namespace-prefix (namespace)
  (make-xml-string (funcall-if-true #'$lxcl-ns-prefix
                                    (item-address namespace))))

(defun namespace-private (namespace)
  (funcall-if-true #'$lxcl-ns-private (item-address namespace)))

(defun namespace-context (namespace)
  (make-document (funcall-if-true #'$lxcl-ns-context
                                  (item-address namespace))))

(setf (symbol-function 'namespace-document) #'namespace-context)



;;;; A Dtd.

(defun make-dtd (&optional address)
  "Return a Dtd with address ADDRESS."
  (make-instance 'dtd :address address))

;;; Properties of a Dtd.

(defun dtd-private (dtd)
  (funcall-if-true #'$lxcl-dtd-private (item-address dtd)))

(defun dtd-type (dtd)
  (funcall-if-true #'(lambda (address)
                       (enum-from-value '$xml-element-type
                                        ($lxcl-dtd-type address)))
                   (item-address dtd)))

(defun dtd-name (dtd)
  (make-xml-string (funcall-if-true #'$lxcl-dtd-name
                                    (item-address dtd))))

(defun dtd-children (dtd)
  (make-node-list (funcall-if-true #'$lxcl-dtd-children
                                   (item-address dtd))))

(defun dtd-last (dtd)
  (make-node (funcall-if-true #'$lxcl-dtd-last
                              (item-address dtd))))

(defun dtd-parent (dtd)
  (make-document (funcall-if-true #'$lxcl-dtd-parent
                                  (item-address dtd))))

(defun dtd-next (dtd)
  (make-node (funcall-if-true #'$lxcl-dtd-next
                              (item-address dtd))))

(defun dtd-prev (dtd)
  (make-node (funcall-if-true #'$lxcl-dtd-prev
                              (item-address dtd))))

(setf (symbol-function 'dtd-previous) #'dtd-prev)

(defun dtd-doc (dtd)
  (make-document (funcall-if-true #'$lxcl-dtd-doc
                                  (item-address dtd))))

(setf (symbol-function 'dtd-document) #'dtd-doc)

(defun dtd-notations (dtd)
  (funcall-if-true #'$lxcl-dtd-notations (item-address dtd)))

(defun dtd-elements (dtd)
  (funcall-if-true #'$lxcl-dtd-elements (item-address dtd)))

(defun dtd-attributes (dtd)
  (funcall-if-true #'$lxcl-dtd-attributes (item-address dtd)))

(defun dtd-entities (dtd)
  (funcall-if-true #'$lxcl-dtd-entities (item-address dtd)))

(defun dtd-external-id (dtd)
  (make-xml-string (funcall-if-true #'$lxcl-dtd-external-id
                                    (item-address dtd))))

(defun dtd-system-id (dtd)
  (make-xml-string (funcall-if-true #'$lxcl-dtd-system-id
                                    (item-address dtd))))

(defun dtd-pentities (dtd)
  (funcall-if-true #'$lxcl-dtd-pentities (item-address dtd)))



;;;; An Attribute.

(defun make-attribute (&optional address)
  "Return an Attribute with address ADDRESS."
  (make-instance 'attribute :address address))

(defun make-attribute-list (&optional address)
  "Return an Attribute-list with address ADDRESS."
  (make-instance 'attribute-list :address address))

;;; Properties of an Attribute.

(defun attribute-private (attribute)
  (funcall-if-true #'$lxcl-attr-private (item-address attribute)))

(defun attribute-type (attribute)
  (funcall-if-true #'(lambda (address)
                       (enum-from-value '$xml-element-type
                                        ($lxcl-attr-type address)))
                   (item-address attribute)))

(defun attribute-name (attribute)
  (make-xml-string (funcall-if-true #'$lxcl-attr-name
                                    (item-address attribute))))

(defun attribute-children (attribute)
  (make-node-list (funcall-if-true #'$lxcl-attr-children
                                   (item-address attribute))))

(defun attribute-last (attribute)
  (make-node (funcall-if-true #'$lxcl-attr-last
                              (item-address attribute))))

(defun attribute-parent (attribute)
  (make-node (funcall-if-true #'$lxcl-attr-parent
                              (item-address attribute))))

(defun attribute-next (attribute)
  (make-attribute (funcall-if-true #'$lxcl-attr-next
                                   (item-address attribute))))

(defun attribute-prev (attribute)
  (make-attribute (funcall-if-true #'$lxcl-attr-prev
                                   (item-address attribute))))

(setf (symbol-function 'attribute-previous) #'attribute-prev)

(defun attribute-doc (attribute)
  (make-document (funcall-if-true #'$lxcl-attr-doc
                                  (item-address attribute))))

(setf (symbol-function 'attribute-document) #'attribute-doc)

(defun attribute-ns (attribute)
  (make-namespace (funcall-if-true #'$lxcl-attr-ns
                                   (item-address attribute))))

(setf (symbol-function 'attribute-namespace) #'attribute-ns)

(defun attribute-atype (attribute)
  (funcall-if-true #'(lambda (address)
                       (enum-from-value '$xml-attribute-type
                                        ($lxcl-attr-atype address)))
                   (item-address attribute)))

(defun attribute-psvi (attribute)
  (funcall-if-true #'$lxcl-attr-psvi (item-address attribute)))



;;; A Node.

(defun make-node (&optional address)
  "Return a Node with address ADDRESS."
  (make-instance 'node :address address))

(defun make-node-list (&optional address)
  "Return a Node-list with address ADDRESS."
  (make-instance 'node-list :address address))

;;; Properties of a Node.

(defun node-private (node)
  (funcall-if-true #'$lxcl-node-private (item-address node)))

(defun node-type (node)
  (funcall-if-true #'(lambda (address)
                       (enum-from-value '$xml-element-type
                                        ($lxcl-node-type address)))
                   (item-address node)))

(defun node-name (node)
  (make-xml-string (funcall-if-true #'$lxcl-node-name
                                    (item-address node))))

(defun node-children (node)
  (make-node-list (funcall-if-true #'$lxcl-node-children
                                   (item-address node))))

(defun node-last (node)
  (make-node (funcall-if-true #'$lxcl-node-last
                              (item-address node))))

(defun node-parent (node)
  (make-node (funcall-if-true #'$lxcl-node-parent
                              (item-address node))))

(defun node-next (node)
  (make-node (funcall-if-true #'$lxcl-node-next
                              (item-address node))))

(defun node-prev (node)
  (make-node (funcall-if-true #'$lxcl-node-prev
                              (item-address node))))

(setf (symbol-function 'node-previous) #'node-prev)

(defun node-doc (node)
  (make-document (funcall-if-true #'$lxcl-node-doc
                                  (item-address node))))

(setf (symbol-function 'node-document) #'node-doc)

(defun node-ns (node)
  (make-namespace (funcall-if-true #'$lxcl-node-ns
                                   (item-address node))))

(setf (symbol-function 'node-namespace) #'node-ns)

(defun node-content (node)
  (make-xml-string (funcall-if-true #'$lxcl-node-content
                                    (item-address node))))

(defun node-properties (node)
  (make-attribute-list (funcall-if-true #'$lxcl-node-properties
                                        (item-address node))))

(setf (symbol-function 'node-attributes) #'node-properties)

(defun node-ns-def (node)
  (make-namespace-list (funcall-if-true #'$lxcl-node-ns-def
                                        (item-address node))))

(setf (symbol-function 'node-namespace-definitions) #'node-ns-def)

(defun node-psvi (node)
  (funcall-if-true #'$lxcl-node-psvi (item-address node)))

(defun node-line (node)
  (funcall-if-true #'$lxcl-node-line (item-address node)))

(defun node-extra (node)
  (funcall-if-true #'$lxcl-node-extra (item-address node)))



;;; A Document.

(defun make-document (&optional address)
  "Return a Document with address ADDRESS."
  (make-instance 'document :address address))

;;; Properties of a Document.

(defun document-private (document)
  (funcall-if-true #'$lxcl-doc-private (item-address document)))

(defun document-type (document)
  (funcall-if-true #'(lambda (address)
                       (enum-from-value '$xml-element-type
                                        ($lxcl-doc-type address)))
                   (item-address document)))

(defun document-name (document)
  (funcall-if-true #'$lxcl-doc-name (item-address document)))

(defun document-children (document)
  (make-node-list (funcall-if-true #'$lxcl-doc-children
                                   (item-address document))))

(defun document-last (document)
  (make-node (funcall-if-true #'$lxcl-doc-next
                              (item-address document))))

(defun document-parent (document)
  (make-node (funcall-if-true #'$lxcl-doc-parent
                              (item-address document))))

(defun document-next (document)
  (make-node (funcall-if-true #'$lxcl-doc-next
                              (item-address document))))

(defun document-prev (document)
  (make-node (funcall-if-true #'$lxcl-doc-prev
                              (item-address document))))

(setf (symbol-function 'document-previous) #'document-prev)

(defun document-doc (document)
  (make-document (funcall-if-true #'$lxcl-doc-doc
                                  (item-address document))))

(setf (symbol-function 'document-self) #'document-doc)

(defun document-compression (document)
  (funcall-if-true #'$lxcl-doc-compression (item-address document)))

(defun document-standalone (document)
  (case (funcall-if-true #'$lxcl-doc-standalone (item-address document))
    (0 nil)
    (1 :standalone)
    (-1 :no-xml-declaration)
    (-2 :xml-declaration-with-unspecified-standalone)
    (t :null-document)))

(defun document-int-subset (document)
  (make-dtd (funcall-if-true #'$lxcl-doc-int-subset
                             (item-address document))))

(setf (symbol-function 'document-internal-subset) #'document-int-subset)

(defun document-ext-subset (document)
  (make-dtd (funcall-if-true #'$lxcl-doc-ext-subset
                             (item-address document))))

(setf (symbol-function 'document-external-subset) #'document-ext-subset)

(defun document-old-ns (document)
  (make-namespace (funcall-if-true #'$lxcl-doc-old-ns
                                   (item-address document))))

(setf (symbol-function 'document-old-namespace) #'document-old-ns)

(defun document-version (document)
  (make-xml-string (funcall-if-true #'$lxcl-doc-version
                                    (item-address document))))

(defun document-encoding (document)
  (make-xml-string (funcall-if-true #'$lxcl-doc-encoding
                                    (item-address document))))

(defun document-ids (document)
  (funcall-if-true #'$lxcl-doc-ids (item-address document)))

(defun document-refs (document)
  (funcall-if-true #'$lxcl-doc-refs (item-address document)))

(defun document-url (document)
  (make-xml-string (funcall-if-true #'$lxcl-doc-url
                                    (item-address document))))

(setf (symbol-function 'document-uri) #'document-url)

(defun document-charset (document)
  (funcall-if-true #'$lxcl-doc-charset (item-address document)))

(defun document-dict (document)
  (funcall-if-true #'$lxcl-doc-dict (item-address document)))

(defun document-psvi (document)
  (funcall-if-true #'$lxcl-doc-psvi (item-address document)))



;;;; Conditions.

(define-condition document-error (error)
  ((document :initarg :document :reader error-document)))

(define-condition empty-document-error (document-error)
  ()
  (:report (lambda (condition stream)
             (format stream "The document ~A is empty."
                     (error-document condition)))))



;;;; Operations.

(defparameter *document* nil
  "The current document.")

(defmethod free-item ((document document))
  ($xml-free-doc (item-address document)))

(defun document-element (document)
  "Return the Node representing the document element of DOCUMENT."
  (let ((address ($xml-doc-get-root-element (item-address document))))
    (if address
      (make-node address)
      (error 'empty-document-error :document document))))

(defun (setf document-element) (node document)
  "Set the document element of the document DOCUMENT to NODE.
Returns the document element of DOCUMENT if DOCUMENT already had a
document element, otherwise nil.  No action is taken if ROOT is nil."
  (if (null-item-p document)
    (error 'empty-document-error :document document)
    (let ((old-address ($xml-doc-set-root-element (item-address document)
                                                  (item-address node))))
      (and old-address (make-node old-address)))))

(defun node-namestring (node)
  "Return the value of the NAME property of NODE as a Unicode string."
  (xml-string-to-string (node-name node)))

(defun node-list-text (document node-list
                       &optional (replace-entities t))
  "Return the content of the text children of elements of NODE-LIST.
The return value is a string, which is the concatenation of the
contents of the text children of elements of NODE-LIST.  The context
for the operation is provided by the document DOCUMENT.  If
REPLACE-ENTITIES is true, then entity references will be replaced."
  (let* ((document-address (item-address document))
         (nl-children (node-children node-list))
         (nl-children-address (item-address nl-children))
         (text-address ($xml-node-list-get-string document-address
                                                  nl-children-address
                                                  (if replace-entities 1 0))))
    (if (null-address-p text-address)
      ""
      (let ((xml-string (make-xml-string text-address)))
        (prog1
          (xml-string-to-string xml-string)
          (free-item xml-string))))))

(defun node-list-map (function node-list &optional predicate)
  "Apply FUNCTION to elements of NODE-LIST that satisfy PREDICATE.
FUNCTION must be a function which takes a Node as argument.  The
return value is a list containing the results of applying FUNCTION to
the successive elements of NODE-LIST that satisfy PREDICATE.  If
PREDICATE is not supplied, then FUNCTION is applied to all the
successive elements of NODE-LIST."
  (loop for node = node-list then (node-next node)
     until (null-item-p node)
     when (or (null predicate) (funcall predicate node))
     collect (funcall function node)))

(defun append-child-with-text (parent name &key content namespace)
  "Create a fresh Node named NAME as a child of PARENT.
The freshly created Node is an element, and is appended to the list of
children of PARENT.  If CONTENT is supplied, a child text node of the
new element will also be created containing the string CONTENT.
Reserved XML characters, such as the ampersand, greater-than or
less-than signs, that appear in CONTENT, are automatically replaced by
their XML escaped entity representations.  If NAMESPACE is not
supplied or a null Namespace, then the freshly created element
inherits the namespace of PARENT.  NAME and CONTENT must be XML String
designators.  If CONTENT is not supplied or is a null XML String, then
the freshly created element is an empty element.  The return-value is
the freshly created element."
  (if (null-item-p parent)
    (error 'null-item-error :item parent)
    (with-xml-string (xname name)
      (if (null-item-p xname)
        (error 'null-item-error :item xname)
        (with-xml-string (xcontent content)
          (make-node ($xml-new-text-child (item-address parent)
                                          (and namespace
                                               (item-address namespace))
                                          (item-address xname)
                                          (item-address xcontent))))))))

(defun document-to-file (document filename &key encoding indent)
  "Write DOCUMENT to the file FILENAME using the encoding ENCODING.
FILENAME must be a pathname designator.  If ENCODING is not supplied,
then DOCUMENT will be written using UTF-8.  If INDENT is true, then
the output is indented.  Returns the number of octets written."
  (if (null-item-p document)
    (error 'null-item-error :item document)
    (let ((address (item-address document))
          (namestring (namestring filename))
          (encoding-name (encoding-name encoding))
          (format (if indent 1 0))
          (old-indent $xml-indent-tree-output))
      (flet ((do-write ()
               (let ((result ($xml-save-format-file-enc namestring address
                                                        encoding-name format)))
                 (if (plusp result) result (error 'document-writing-error
                                                  :document document)))))
        (unwind-protect
          (progn
            (setf $xml-indent-tree-output format)
            (if (probe-file filename)
              (restart-case (error 'file-exists-error
                                   :pathname (pathname filename))
                (write-new-file (new-filename)
                  :report "Write to a new file."
                  :interactive read-new-value
                  (document-to-file document new-filename
                                    :encoding encoding :indent indent))
                (overwrite ()
                  :report "Overwrite the file."
                  (do-write)))
              (do-write)))
          (setf $xml-indent-tree-output old-indent))))))

(defun add-attribute (node name &optional value)
  "Add a fresh Attribute to NODE with name NAME and value VALUE.
Returns the freshly created Attribute.  NAME and VALUE must be XML
String designators.  If VALUE is not supplied or is a null XML String,
then the freshly created Attribute has value the empty string."
  (if (null-item-p node)
    (error 'null-item-error :item node)
    (with-xml-string (xname name)
      (if (null-item-p xname)
        (error 'null-item-error :item xname)
        (with-xml-string (xvalue value)
          (make-attribute
           ($xml-new-prop (item-address node) (item-address xname)
                          (item-address xvalue))))))))

(defun attribute-value (node name)
  "Return the value of the attribute of NODE with name NAME.
NAME must be an XML String designator.  Entities in the value are
replaced.  Ignores namespaces associated to the attribute."
  (if (null-item-p node)
    (error 'null-item-error :item node)
    (with-xml-string (xname name)
      (if (null-item-p xname)
        (error 'null-item-error :item xname)
        (let ((value (make-xml-string ($xml-get-prop (item-address node)
                                                     (item-address xname)))))
          (prog1
            (xml-string-to-string value)
            (free-item value)))))))

(defun new-document (version)
  "Create a new Document with XML version VERSION.
VERSION must be an XML String desginator representing a string of the
form \"1.0\".  Returns the newly created Document."
  (with-xml-string (xversion version)
    (if (null-item-p xversion)
      (error 'null-item-error :item xversion)
      (make-document ($xml-new-doc (item-address xversion))))))

(defmacro with-new-document ((document version) &body body)
  "Evaluate BODY using a new Document with XML version VERSION.
VERSION must be an XML String designator representing a string like
\"1.0\".  During the evaluation, the variable DOCUMENT and the special
variable *DOCUMENT* are both bound to the new Document that has been
created.  That Document has dynamic extent, which ends when the form
is exited."
  (let ((created (gensym)))
    `(let* ((,created nil)
            (,document (new-document ,version))
            (*document* ,document))
       (unwind-protect (progn (setf ,created t)
                              (init-library)
                              ,@body)
         (when ,created
           (free-item ,document))))))

(defun new-node-in-document (document name &key namespace content replace-ents)
  "Create a new node with name NAME in the document DOCUMENT.
If REPLACE-ENTS is true, then entities are replaced in CONTENT.  NAME
and CONTENT must be XML String desginators.  XML special characters in
CONTENT need to be escaped first using `replace-entities'.  Returns
the newly created Node."
  (if (null-item-p document)
    (error 'empty-document-error :document document)
    (with-xml-string (xname name)
      (if (null-item-p xname)
        (error 'null-item-error :item xname)
        (with-xml-string (xcontent content)
          (make-node (funcall (if replace-ents
                                #'$xml-new-doc-node
                                #'$xml-new-doc-raw-node)
                              (item-address document)
                              (funcall-if-true #'item-address namespace)
                              (item-address xname)
                              (item-address xcontent))))))))



;;;; Writing to streams.

(defvar *xml-encoding* charset:utf-8
  "The default encoding for XML characters.")

(defvar *$xml-output-write-callback*
  (with-foreign-object (foreign-variable '$xml-output-write-callback-function)
    (setf (foreign-value foreign-variable)
            #'(lambda (context buffer len)
                (declare (ignore context))
                (write-string-from-address buffer *xml-encoding* :end len)
                len))
    ;; The foreign value of FOREIGN-VARIABLE is a foreign function.
    ;; Return the foreign address of that foreign function.
    (foreign-address (foreign-value foreign-variable)))
  "The foreign address of an XML Output Write Callback Function.
Such functions are foreign functions used in libxml to write to output
buffers.")

(define-condition document-writing-error (document-error)
  ()
  (:report (lambda (condition stream)
             (format stream "Unable to write the document ~A."
                     (error-document condition)))))

(defun write-document (document &key encoding indent)
  "Write DOCUMENT to the default XML output stream using ENCODING.
If INDENT is true, then the output is indented.  Returns the number of
octets written."
  (let ((*xml-encoding* (or encoding *xml-encoding*))
        ;; Encoding names are case-insensitive [XML spec, 4.3.3].
        (encoding-name (encoding-name encoding))
        (format (if indent 1 0))
        (old-indent $xml-indent-tree-output))
    (unwind-protect
      (progn
        (setf $xml-indent-tree-output format)
        (let ((result ($xml-save-format-file-to
                       ($xml-output-buffer-create-io
                        *$xml-output-write-callback* nil nil
                        ($xml-find-char-encoding-handler encoding-name))
                       (item-address document) encoding-name format)))
          (if (plusp result) result (error 'document-writing-error
                                           :document document))))
      (setf $xml-indent-tree-output old-indent))))



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; tree.lisp ends here