;;;; libxml-clisp.asd --- system definition for libxml-clisp

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-06-14
;;; 
;;; $Hg: libxml-clisp.asd,v 238ceb51574b 2009-09-13T19:14:52+05:30 raghu $

(defpackage "NET.RETROTEXTS.LIBXML-CLISP.SYSTEM"
  (:use "COMMON-LISP" "ASDF")
  (:export "*LIBXML-CLISP.SO*"))

(in-package "NET.RETROTEXTS.LIBXML-CLISP.SYSTEM")



;;;; System definition.

(defsystem "libxml-clisp"
  :description "CLISP FFI to libxml2."
  :version "0.1"
  :serial t
  :components ((:file "packages")
               (:module "foreign"
                        :components ((:c-source-file "libxml-clisp")
                                     (:file "types")
                                     (:file "foreign")
                                     (:file "common")
                                     (:file "string")
                                     (:file "tree")
                                     (:file "parser")
                                     (:file "xpath")
                                     (:file "reader")))
               (:module "examples"
                        :components ((:file "common")
                                     (:file "keyword")
                                     (:file "keyword-xpath")
                                     (:file "keyword-add")
                                     (:file "attribute-add")
                                     (:file "attribute-get")
                                     (:file "encoding")
                                     (:file "names")
                                     (:file "names-reader")
                                     (:file "xpath")))))



;;;; Operations on C source files.

(defmethod output-files ((operation compile-op) (source-file c-source-file))
  (declare (ignore operation))
  (list (make-pathname :type "so"
                       :defaults (component-pathname source-file))))

(defvar *cflags* "-fpic -DPIC -shared `xml2-config --cflags`"
  "Compiler flags for creating a shared C library using libxml2.")

(defvar *ldflags* "`xml2-config --libs`"
  "Linker flags for linking a shared object with libxml2.")

(defun c-compile-command (source-file output-file)
  "Return a shell command to create a shared C library using libxml2.
SOURCE-FILE is the filename of a C file that uses libxml2.
OUTPUT-FILE is the filename for the shared object file produced from
SOURCE-FILE."
  (format nil "cc ~A -o ~A ~A ~A"
          *cflags* output-file source-file *ldflags*))

(defmethod perform ((operation compile-op) (source-file c-source-file))
  (let ((source (namestring (component-pathname source-file)))
        (output (namestring (car (output-files operation source-file)))))
    (format t "~
;; Compiling file ~A ...~%"
            source)
    (if (zerop (ext:run-shell-command (c-compile-command source output)))
      (format t "~
;; Wrote file ~A~%"
              output)
      (error 'compile-error :operation operation :component source-file))))

;;; It is assumed that there is only one C source file in the current
;;; system, with ASDF component name "libxml-clisp".  Its output file
;;; is the shared object file created above.  "Loading" that output
;;; file means just setting the special variable *LIBXML-CLISP.SO* to
;;; its filename.  All other C source files will be compiled as above,
;;; but their output files will be ignored by the loading operation.

(defparameter *libxml-clisp.so* nil
  "The filename of the C wrapper library for libxml-clisp.")

(defmethod perform ((operation load-op) (source-file c-source-file))
  (declare (ignore operation))
  (let ((output (namestring (car (output-files (make-instance 'compile-op)
                                               source-file)))))
    (cond ((string= (component-name source-file) "libxml-clisp")
            (format t "~
;; Loading file ~A ...
;; Setting NET.RETROTEXTS.LIBXML-CLISP.SYSTEM:*LIBXML-CLISP.SO* ...~%"
                    output)
            (setf *libxml-clisp.so* output)
            (format t "~
;; NET.RETROTEXTS.LIBXML-CLISP.SYSTEM:*LIBXML-CLISP.SO* = \"~A\"
;; Loaded file ~A~%"
                    *libxml-clisp.so* output))
          (t (format t "~
;; Ignoring file ~A~%"
                     output)))))



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; libxml-clisp.asd ends here