;;;; reader.lisp --- FFI definitions for libxml-clisp

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-09-12
;;; 
;;; $Hg: reader.lisp,v 238ceb51574b 2009-09-13T19:14:52+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP")



(defparameter *text-reader* nil
  "The current Text Reader.")

(defmethod free-item ((text-reader text-reader))
  ($xml-free-text-reader (item-address text-reader)))

(defun make-text-reader (&optional address)
  "Return a Text Reader with address ADDRESS."
  (make-instance 'text-reader :address address))

(defun file-reader (filename &key encoding options)
  "Return a reader for processing  FILENAME using ENCODING.
FILENAME must be a pathname designator for an XML file.  If ENCODING
is non-nil, it is assumed to be the encoding of FILENAME.  If ENCODING
is nil, then the encoding of FILENAME is taken from the XML
declaration, if any, in FILENAME.  If the encoding of FILENAME cannot
be obtained from its XML declaration, then the encoding of FILENAME is
assumed to be UTF-8.  OPTIONS must be symbol or a list of symbols from
`*parser-options*'."
  (let* ((encoding-name (encoding-name encoding))
         (reader-address
           ($xml-reader-for-file (namestring filename) encoding-name
                                 (combine-options options))))
    (if reader-address
      (make-text-reader reader-address)
      (restart-case (error 'xml-file-error
                           :pathname (pathname filename))
        (process-new-file (new-filename)
          :report "Process another file."
          :interactive read-new-value
          (file-reader new-filename :encoding encoding :options options))))))

(define-condition text-reader-error (error)
  ((text-reader :initarg :text-reader :reader error-text-reader))
  (:report (lambda (condition stream)
             (format stream "Reader ~A cannot process data."
                     (error-text-reader condition)))))

(defun text-reader-read (text-reader)
  "Move the position of TEXT-READER to the next node in the input.
Returns true if the node was read successfully, and returns nil if
there are no more nodes to read."
  (if (null-item-p text-reader)
    (error 'null-item-error :item text-reader)
    (let ((reader-status ($xml-text-reader-read (item-address text-reader))))
      (if (minusp reader-status)
        (error 'text-reader-error :text-reader text-reader)
        (plusp reader-status)))))

(defun text-reader-node-type (text-reader)
  "Return the type of the node at the current position of TEXT-READER.
The node type is as in `$xml-reader-types'."
  (enum-from-value '$xml-reader-types
                   ($xml-text-reader-node-type (item-address text-reader))))

(defun text-reader-depth (text-reader)
  "Return the tree-depth of the node at the current position of TEXT-READER."
  (let ((depth ($xml-text-reader-depth (item-address text-reader))))
    (if (minusp depth)
      (error 'text-reader-error :text-reader text-reader)
      depth)))

(defun text-reader-name (text-reader)
  "Return the qualified name of the current node of TEXT-READER.
The qualified name is of the form `prefix:local-name'.  The return
value is an XML String representing the qualified name."
  (make-xml-string ($xml-text-reader-const-name (item-address text-reader))))

(defmacro with-text-reader ((reader filename &key encoding options) &body body)
  "Evaluate BODY using a Text Reader for processing FILENAME.
FILENAME must be a pathname designator for an XML file.  During the
evaluation, the variable READER and the special variable *TEXT-READER*
are both bound to a Text Reader for processing FILENAME.  That Text
Reader has dynamic extent, which ends when the form is exited.

If ENCODING is non-nil, it is assumed to be the encoding of FILENAME.
If ENCODING is nil, then the encoding of FILENAME is taken from the
XML declaration, if any, in FILENAME.  If the encoding of FILENAME
cannot be obtained from its XML declaration, then the encoding of
FILENAME is assumed to be UTF-8.  OPTIONS must be symbol or a list of
symbols from `*parser-options*'."
  (let ((created (gensym)))
    `(let* ((,created nil)
            (,reader (file-reader ,filename :encoding ,encoding
                                  :options ,options))
            (*text-reader* ,reader))
       (unwind-protect (progn (setf ,created t)
                              (init-library)
                              ,@body)
         (when ,created
           (free-item ,reader)
           (cleanup-parser))))))



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; reader.lisp ends here