;;;; keyword.lisp --- example for libxml-clisp tutorial

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-06-14
;;; 
;;; $Hg: keyword.lisp,v ae6366215b15 2009-09-03T20:01:35+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP.EXAMPLES")



(define-condition story-error (document-error)
  ()
  (:report (lambda (condition stream)
             (format stream
                     "The document element of ~A does not have name \"story\"."
                     (error-document condition)))))

(defun story-node-p (node)
  "Return true if NODE is a `story' Node."
  (string= (node-namestring node) "story"))

(defun flatten (list-of-lists)
  "Append all the elements of LIST-OF-LISTS."
  (mapcan #'copy-list list-of-lists))

(defun storyinfo-keywords (filename)
  "Return the content of all `storyinfo/keyword' Nodes in FILENAME.
FILENAME is a pathname designator for an XML file.  The return value
is a list of strings, each of which is the content of a `keyword'
Node."
  (with-xml-file (story filename)
    (let ((document-element (document-element story)))
      (if (story-node-p document-element)
        (flatten (get-storyinfo-keywords story document-element))
        (restart-case (error 'story-error :document story)
          (parse-new-file (new-filename)
            :report "Parse another file."
            :interactive read-new-value
            (storyinfo-keywords new-filename)))))))

(defun get-storyinfo-keywords (story node)
  (node-list-map #'(lambda (nd)
                     (keyword-children story nd))
                 (node-children node)
                 #'(lambda (nd)
                     (string= (node-namestring nd) "storyinfo"))))

(defun keyword-children (story node)
  "Return the content of all the `keyword' children of NODE.
The context is the document STORY."
  (node-list-map #'(lambda (nd)
                     (node-list-text story nd))
                 (node-children node)
                 #'(lambda (nd)
                     (string= (node-namestring nd) "keyword"))))

(defun test-storyinfo-keywords (string)
  (with-temp-file (test-file)
    (with-open-file (out test-file :direction :output)
      (write-string string out))
    (storyinfo-keywords test-file)))

(defvar *storyinfo-keywords-test-data*
  "<?xml version=\"1.0\"?>
<story>
  <storyinfo>
    <author>Foo B. Quux</author>
    <datewritten>2009-05-17</datewritten>
    <keyword>storyinfo-1/keyword-1</keyword>
    <keyword>storyinfo-1/keyword-2</keyword>
  </storyinfo>
  <storyinfo>
    <keyword>storyinfo-2/keyword-1</keyword>
    <keyword>storyinfo-2/keyword-2</keyword>
  </storyinfo>
  <body>
    <headline>The Headline</headline>
    <keyword>body/keyword</keyword>
    <para>This is the body text.</para>
  </body>
</story>")

(defvar *storyinfo-keywords-test-value*
        '("storyinfo-1/keyword-1" "storyinfo-1/keyword-2"
          "storyinfo-2/keyword-1" "storyinfo-2/keyword-2"))

(defun test-keyword ()
  (test-libxml-clisp #'test-storyinfo-keywords
                     *storyinfo-keywords-test-data*
                     *storyinfo-keywords-test-value*))

(provide-example 'keyword)



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; keyword.lisp ends here