/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

#ifndef TERM_PV_H_
#define TERM_PV_H_

#include "../config.h"

#include <unistd.h>
#include <string.h>
#include <stdarg.h>

#ifdef HAVE_TERMIOS_H
# include <termios.h>
#endif

#include <termui/term.h>

/************************************************************************/
/* terminal methods */

#define	TERM_FCN_RESET(f)	int f(struct term_s *tm)
#define TERM_FCN_GETSIZE(f)	int f(struct term_s *tm, unsigned int *x, unsigned int *y)
#define	TERM_FCN_MOVE(f)	int f(struct term_s *tm, enum term_direction_e dir, unsigned int n)
#define	TERM_FCN_SETPOS(f)	int f(struct term_s *tm, unsigned int x, unsigned int y)
#define	TERM_FCN_GETPOS(f)	int f(struct term_s *tm, unsigned int *x, unsigned int *y)
#define TERM_FCN_ATTRIB(f)	int f(struct term_s *tm, enum term_attrib_e attr)
#define TERM_FCN_ERASE(f)	int f(struct term_s *tm, enum term_direction_e dir)
#define TERM_FCN_BEEP(f)	int f(struct term_s *tm)
#define TERM_FCN_ERASELINE(f)	int f(struct term_s *tm, enum term_direction_e dir)
#define TERM_FCN_DELCHAR(f)	int f(struct term_s *tm, unsigned int n)
#define TERM_FCN_DELLINE(f)	int f(struct term_s *tm, enum term_direction_e dir, unsigned int n)
#define TERM_FCN_INSSTR(f)	int f(struct term_s *tm, const char * str, unsigned int n)
#define TERM_FCN_WRITESTR(f)	int f(struct term_s *tm, const char * str, unsigned int n)
#define TERM_FCN_WRITECHAR(f)	int f(struct term_s *tm, const char c, unsigned int n)
#define TERM_FCN_NEWLINE(f)	int f(struct term_s *tm)
#define TERM_FCN_READKEY(f)	int f(struct term_s *tm)

typedef	TERM_FCN_RESET(term_reset_t);
typedef TERM_FCN_GETSIZE(term_getsize_t);
typedef	TERM_FCN_MOVE(term_move_t);
typedef	TERM_FCN_SETPOS(term_setpos_t);
typedef	TERM_FCN_GETPOS(term_getpos_t);
typedef TERM_FCN_ATTRIB(term_attrib_t);
typedef TERM_FCN_ERASE(term_erase_t);
typedef TERM_FCN_BEEP(term_beep_t);
typedef TERM_FCN_ERASELINE(term_eraseline_t);
typedef TERM_FCN_DELCHAR(term_delchar_t);
typedef TERM_FCN_DELLINE(term_delline_t);
typedef TERM_FCN_INSSTR(term_insstr_t);
typedef TERM_FCN_WRITESTR(term_writestr_t);
typedef TERM_FCN_WRITECHAR(term_writechar_t);
typedef TERM_FCN_NEWLINE(term_newline_t);
typedef TERM_FCN_READKEY(term_readkey_t);

struct term_methods_s
{
  term_reset_t		*reset; /* mandatory */
  term_getsize_t	*getsize; /* mandatory */
  term_move_t		*move;
  term_setpos_t		*setpos;
  term_getpos_t		*getpos;
  term_attrib_t		*attrib;
  term_erase_t		*erase;
  term_beep_t		*beep;  
  term_eraseline_t	*eraseline;
  term_delchar_t	*delchar;
  term_delline_t	*delline;
  term_insstr_t		*insstr;
  term_writestr_t	*writestr; /* mandatory */
  term_writechar_t	*writechar; /* mandatory */
  term_newline_t	*newline; /* mandatory */
  term_readkey_t	*readkey; /* mandatory */
};

struct term_s
{
  int in;
  int out;

  struct term_methods_s	mt;

#ifdef HAVE_TERMIOS_H
  struct termios old;
#endif

  void *private;
};

#define TERM_TRY(fcn, ...)	(!(fcn) || fcn(__VA_ARGS__))

/************************************************************************/

/* terminal types settings */

int term_init(struct term_s *tm, int in, int out, void *private);
void term_cleanup(struct term_s *tm);

int term_printf_va(struct term_s *tm, const char *fmt, va_list list);

int term_set_none(struct term_s *tm);
int term_set_vt100(struct term_s *tm);
int term_set_vt102(struct term_s *tm);
int term_set_xterm(struct term_s *tm);

static inline int
fdnprintf(int fd, size_t n, const char *fmt, ...)
{
  char buf[n];
  int res;
  va_list list;

  va_start(list, fmt);
  res = vsnprintf(buf, n, fmt, list);
  !write(fd, buf, res > n ? n : res);
  va_end(list);

  return res;
}

static inline int
fdputs(const char *str, int fd)
{
  size_t len = strlen(str);
  return write(fd, str, len);
}

static inline int
fdputc(char c, int fd)
{
  return write(fd, &c, 1);
}

static inline int
fdgetc(int fd)
{
  unsigned char c;
  int res = read(fd, &c, 1);
  return res == 1 ? c : -1;
}

static inline int
fdgeti(int fd, int *result)
{
  int c, res;

  for (res = 0; (c = fdgetc(fd)) >= '0' && c <= '9'; 
       res = res * 10 + c - '0')
    ;

  *result = res;
  return c;
}

#endif

