/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include "term_pv.h"
#include <termui/bhv.h>

int term_reset(struct term_s *tm)
{
  return tm->mt.reset(tm);
}

int term_getsize(struct term_s *tm, unsigned int *x, unsigned int *y)
{
  return tm->mt.getsize(tm, x, y);
}

int term_move(struct term_s *tm, enum term_direction_e dir, unsigned int n)
{
  return tm->mt.move ? tm->mt.move(tm, dir, n) : TERM_RET_INVALID;
}

int term_setpos(struct term_s *tm, unsigned int x, unsigned int y)
{
  return tm->mt.setpos ? tm->mt.setpos(tm, x, y) : TERM_RET_INVALID;
}

int term_getpos(struct term_s *tm, unsigned int *x, unsigned int *y)
{
  return tm->mt.getpos ? tm->mt.getpos(tm, x, y) : TERM_RET_INVALID;
}

int term_attrib(struct term_s *tm, enum term_attrib_e attr)
{
  return tm->mt.attrib ? tm->mt.attrib(tm, attr) : TERM_RET_INVALID;
}

int term_erase(struct term_s *tm, enum term_direction_e dir)
{
  return tm->mt.erase ? tm->mt.erase(tm, dir) : TERM_RET_INVALID;
}

int term_beep(struct term_s *tm)
{
  return tm->mt.beep ? tm->mt.beep(tm) : TERM_RET_INVALID;
}

int term_eraseline(struct term_s *tm, enum term_direction_e dir)
{
  return tm->mt.eraseline ? tm->mt.eraseline(tm, dir) : TERM_RET_INVALID;
}

int term_delchar(struct term_s *tm, unsigned int n)
{
  return tm->mt.delchar ? tm->mt.delchar(tm, n) : TERM_RET_INVALID;
}

int term_delline(struct term_s *tm, enum term_direction_e dir, unsigned int n)
{
  return tm->mt.delline ? tm->mt.delline(tm, dir, n) : TERM_RET_INVALID;
}

int term_insstr(struct term_s *tm, const char * str, unsigned int n)
{
  return tm->mt.insstr ? tm->mt.insstr(tm, str, n) : TERM_RET_INVALID;
}

int term_writestr(struct term_s *tm, const char * str, unsigned int n)
{
  return tm->mt.writestr(tm, str, n);
}

int term_writechar(struct term_s *tm, const char c, unsigned int n)
{
  return tm->mt.writechar(tm, c, n);
}

int term_newline(struct term_s *tm)
{
  return tm->mt.newline(tm);
}

int term_readkey(struct term_s *tm)
{
  return tm->mt.readkey(tm);
}

/* terminal context init */

#ifdef HAVE_TERMIOS_H

static int
term_set_raw(struct term_s *tm)
{
  struct termios	tio;
  int			fd = tm->in;

  if (!isatty(fd))
    return TERM_RET_INVALID;

  if (!tcgetattr(fd, &tio))
    {
      tm->old = tio;

#if 1
      tio.c_iflag &= ~(IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL | IXON);
      tio.c_oflag &= ~OPOST;
      tio.c_lflag &= ~(ECHO | ECHONL | ICANON | ISIG | IEXTEN);
      tio.c_cflag &= ~(CSIZE | PARENB);
      tio.c_cflag |= CS8;
#else
      cfmakeraw(&tio);
#endif

      if (!tcsetattr(fd, 0, &tio))
	return TERM_RET_OK;
    }

  return TERM_RET_IOERROR;
}

void
term_cleanup(struct term_s *tm)
{
  if (isatty(tm->in))
    tcsetattr(tm->in, 0, &tm->old);
}

#endif

int
term_init(struct term_s *tm, 
	  int in, int out,
	  void *private)
{
  tm->in = in;
  tm->out = out;
  tm->private = private;

  term_set_none(tm);
#ifdef HAVE_TERMIOS_H
  term_set_raw(tm);
#endif

  return TERM_RET_OK;
}

struct term_s *
term_alloc(int in, int out, void *private)
{
  struct term_s *tm;

  if ((tm = malloc(sizeof (struct term_s))))
    term_init(tm, in, out, private);

  return tm;
}

int
term_set(struct term_s *tm,
	 const char *type)
{
  term_set_none(tm);

  if (!type)
    return TERM_RET_INVALID;

  if (!strcmp(type, "xterm") || !strcmp(type, "rxvt"))
    return term_set_xterm(tm);

  if (!strcmp(type, "vt100"))
    return term_set_vt100(tm);

  if (!strcmp(type, "vt102"))
    return term_set_vt102(tm);

  return TERM_RET_INVALID;
}

void
term_free(struct term_s *tm)
{
#ifdef HAVE_TERMIOS_H
  term_cleanup(tm);
#endif
  free(tm);
}

void *term_private(struct term_s *tm)
{
  return tm->private;
}

/* behavior process */

int
term_behave(struct term_behavior_s *bhv)
{
  struct term_s		*tm = bhv->tm;
  int			k;

  /* pre-initialise context */
  if (bhv->bhvstart)
    {
      int	res = bhv->bhvstart(bhv);

      if (res < 0)
	return res;
    }

  while (1)
    {
      switch (k = term_readkey(tm))
	{
	default: {
	  term_keyevent_t	*ev = bhv->keyevent[k];

	  if (ev)
	    {
	      int	res = ev(k, bhv);

	      if (res <= 0)
		return res;

	      bhv->lastkey = k;
	    }

	} break;

	case (TERM_RET_IOERROR):
	  return TERM_RET_IOERROR;

	case (TERM_RET_INVALID):
	  break;
	}
    }
}

