/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

#include <stdio.h>
#include <errno.h>

#include "console_pv.h"
#include "command_pv.h"

static void
console_args_usage(struct console_ctx_s *con,
		   const struct command_entry_s *e)
{
  struct command_args_s	*desc = e->args_desc;
  unsigned int		i;

  console_printf(con,
		 "\n"
		 "%1A%s%A - %s\n",
		 e->cmd, e->desc);

  if (e->longdesc)
    {
      console_printf(con,
		     "\n"
		     "%1ADescription%A: \n");
      console_printf(con, e->longdesc);
      console_printf(con, "\n");
    }

  if (desc)
    {
      int	done = 0;

      for (i = 0; desc[i].str_short; i++)
	{
	  if (((1 << i) & e->args_disabled))
	    continue;		/* skip entry disabled options */

	  if (!(desc[i].acl & con->acl))
	    continue;

	  if (!done++)
	    console_printf(con,
			   "\n"
			   "%1AOptions%A:\n");

	  console_printf(con, "    %-8s%-16s%s\n",
			 desc[i].str_short, desc[i].str_long,
			 desc[i].desc);
	}
    }
}

/************************************************************************/

COMMAND_CMD_FUNC(console_builtin_help)
{
  int		done = 0;

  if (argc)
    {
      /* show detailed help about command(s) */
      struct command_entry_s	*e;

      if ((e = console_find_command(con, argv[0])))
	{
	  done++;
	  console_args_usage(con, e);
	}
    }

  if (!done)
    {
      /* show general help message */
      console_printf(con,
		     /*********************************** 79 cols ***********************************/
		     "\n"
		     "Type `%1Alist%A' to show available commands list.\n"
		     "Type `%1Ahelp%A [command ...]' to display detailed command help.\n"
		     "\n"
		     "This console user interface is driven by libtermui (C) 2005 A.Becoulet\n"
	      );
    }

  return 0;
}

/************************************************************************/

/* 
 *  Write available commands table to console
*/

static unsigned int
command_cmd_table_r(struct console_ctx_s *con,
		    struct command_entry_s *root,
		    int depth, const char *prefix)
{
  char		buf[256];
  unsigned int	n = 0;

  while (root)
    {
      if (root->flag & COMMAND_FLAG_HIDDEN)
	continue;

      snprintf(buf, 256, "%s%s%s",
	       prefix, depth ? "." : "", root->cmd);

      if (root->flag & COMMAND_FLAG_ISDIR)
	{
	  if (root->acl & con->acl)
	    {
	      if (n)
		console_printf(con, "\n");
	      command_cmd_table_r(con, root->subdir, depth + 1, buf);
	      n = 0;
	    }
	  else
	    console_printf(con, "\n  %31A%-32s%A %36A[access denied]%A\n\n", buf);
	}
      else
	{
	  console_printf(con, root->acl & con->acl
			 ? "  %-32s %36A%-32s%A\n"
			 : "  %31A%-32s%A %36A[access denied]%A\n",
			 buf, root->desc);
	  n++;
	}

      root = root->next;
    }

  if (n)
    console_printf(con, "\n");

  return n;
}

static void
command_cmd_table(struct console_ctx_s *con)
{
  console_printf(con,
		 "\n"
		 "  %1A%-32s %-35s%A\n",
		 "Command path", "Description");

  console_printf(con, "--------------------------------------------------------------------------------\n");

  command_cmd_table_r(con, *con->root, 0, "");
}

COMMAND_CMD_FUNC(console_builtin_list)
{
  command_cmd_table(con);

  return 0;
}

/************************************************************************/

COMMAND_CMD_FUNC(console_builtin_batch)
{
  unsigned int	i;
  int		res = 0;

  for (i = 0; i < argc; i++)
    if (command_execute_file(con->acl, *con->root, argv[i], con, 0, 1) == -ENOENT)
      res = -ENOENT;

  return res;
}

/************************************************************************/

COMMAND_CMD_FUNC(console_builtin_quit)
{
  return -ECANCELED;
}

