/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <alloca.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>

#include "console_pv.h"
#include "command_pv.h"

static GETLINE_FCN_PROMPT(console_prompt)
{
  struct console_ctx_s *con = private;

  return term_printf(con->tm, con->prompt);
}

struct console_ctx_s *
console_alloc(struct command_entry_s **root,
	    int in, int out, const char *type)
{
  struct console_ctx_s *con;

  if (!(con = malloc(sizeof (struct console_ctx_s))))
    goto err;

  if (!(con->tm = term_alloc(in, out, con)))
    goto err_free;

  term_set(con->tm, type);

  if (!(con->bhv = getline_alloc(con->tm, CONSOLE_LINE_MAXLEN)))
    goto err_term;

  getline_complete_init(con->bhv, console_complete);

  getline_history_init(con->bhv, 64);

  con->acl = COMMAND_ACL_ALL;
  con->root = root;
  con->prompt = "$ ";

  getline_setprompt(con->bhv, console_prompt);

  return con;

#if 0
 err_bhv:
  getline_free(con->bhv);
#endif
 err_term:
  term_free(con->tm);
 err_free:
  free(con);
 err:
  return NULL;
}

/* 
 *  Free console ressources
 */

void
console_free(struct console_ctx_s *con)
{
  getline_free(con->bhv);
  term_free(con->tm);
  free(con);
}

/* Set console prompt string */

void console_set_prompt(struct console_ctx_s *con,
			const char *prompt)
{
  con->prompt = prompt;
}

/* do basic telnet protocol handshaking */

void console_telnet_setup(struct console_ctx_s *con)
{
  term_telnet_bhv_init(con->bhv);
  term_telnet_send_setup(con->tm);
}

void
console_register(struct console_ctx_s *con,
		struct command_entry_s *list)
{
  command_register_root(con->root, list);
}

struct command_entry_s *
console_find_command(struct console_ctx_s *con,
		    char *path)
{
  struct command_entry_s	*e;

  if ((e = command_find_entry(con->acl, *con->root, &path))
      && (e->flag & COMMAND_FLAG_ISCMD))
    return e;

  return NULL;
}

int
console_process(struct console_ctx_s *con)
{
  const char		*rline;
  char			line_[CONSOLE_LINE_MAXLEN], *line = line_;
  int			res;

  if (!(rline = getline_process(con->bhv)))
    return -EIO;

  /* skip blank line */
  if (!*(rline += strspn(rline, " \t\n")))
    return 0;

  /* skip comment */
  if (*rline == '#')
    return 0;

  getline_history_addlast(con->bhv);

  /* copy line to writable buffer for args spliting */
  strncpy(line, rline, CONSOLE_LINE_MAXLEN);

  res = command_execute(con->acl, *con->root, line, con);

  console_printf(con, "\n");

  return res;
}

/* 
 *  Wait for user entry and return line content
 */

const char *
console_input(struct console_ctx_s *con,
	      const char *prompt)
{
  const char	*old_prompt, *res;

  old_prompt = con->prompt;

  res = getline_process(con->bhv);

  con->prompt = old_prompt;

  return res;
}

int
console_printf(struct console_ctx_s *con,
	       const char *fmt, ...)
{
  int		res = 0;
  va_list	list;

  if (con)	/* nicely skip non existing console */
    {
      va_start(list, fmt);

      res = term_printf_va(con->tm, fmt, list);

      va_end(list);
    }

  return res;
}

void *
console_get_private(struct console_ctx_s *con)
{
  return con->pv;
}

void
console_set_private(struct console_ctx_s *con,
		   void *pv)
{
  con->pv = pv;
}

void
console_set_acl(struct console_ctx_s *con,
	       int acl_mask)
{ 
  con->acl = acl_mask;
}

int console_save_history(struct console_ctx_s *con, FILE *file)
{
  const char *str;
  unsigned int i = 0;

  while ((str = getline_history_get(con->bhv, i++)) != NULL)
    fprintf(file, "%s\n", str);

  return TERM_RET_OK;
}

int console_load_history(struct console_ctx_s *con, FILE *file)
{
  char buf[CONSOLE_LINE_MAXLEN];

  while (fgets(buf, CONSOLE_LINE_MAXLEN, file) != NULL)
    {
      buf[strlen(buf) - 1] = '\0';
      getline_history_add(con->bhv, buf);
    }

  return TERM_RET_OK;
}

